﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.DataTypes;

using EffectMaker.DataModelMaker.UIControls.Interfaces;

using EffectMaker.Foundation.Interfaces;

using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor
{
    /// <summary>
    /// User control class for editor data model property editing.
    /// </summary>
    public partial class RuntimeDataModelPropertyEditor : UIUserControl, IDataModelEditor
    {
        /// <summary>The suggestion strings for the property type text box.</summary>
        private AutoCompleteStringCollection typeSuggestions =
            new AutoCompleteStringCollection();

        /// <summary>
        /// Constructor.
        /// </summary>
        public RuntimeDataModelPropertyEditor()
        {
            this.InitializeComponent();

            this.typeSuggestions.AddRange(
                (from tp in TypeManager.RuntimeTypes select tp.FullName).ToArray());
            this.txtType.AutoCompleteCustomSource = this.typeSuggestions;

            this.lblTypeWarning.Visibility = Visibility.Collapsed;
            this.lblNameWarning.Visibility = Visibility.Collapsed;

            this.Bindings.Add(new Binder(this, "PropertyType", "PropertyType"));
            this.Bindings.Add(new Binder(this, "IsTypeWarningVisible", "IsPropertyTypeInvalid"));
            this.Bindings.Add(new Binder(this, "PropertyName", "Name"));
            this.Bindings.Add(new Binder(this, "ArraySize", "ArraySize"));
            this.Bindings.Add(new Binder(this, "PropertyDescription", "Description"));
            this.Bindings.Add(new Binder(this, "OnCommitEditingExecutable", "OnCommitEditingExecutable"));
            this.Bindings.Add(new Binder(this, "OnCancelEditingExecutable", "OnCancelEditingExecutable"));
        }

        /// <summary>
        /// Get or set the value type of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyType
        {
            get { return this.txtType.Text; }
            set { this.txtType.Text = value; }
        }

        /// <summary>
        /// Get or set the value type of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsTypeWarningVisible
        {
            get { return this.lblTypeWarning.Visible; }
            set { this.lblTypeWarning.Visible = value; }
        }

        /// <summary>
        /// Get or set the name of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyName
        {
            get { return this.txtName.Text; }
            set { this.txtName.Text = value; }
        }

        /// <summary>
        /// Get or set the default value of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string ArraySize
        {
            get { return this.txtArraySize.Text; }
            set { this.txtArraySize.Text = value; }
        }

        /// <summary>
        /// Get or set the description of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyDescription
        {
            get { return this.txtDescription.Text; }
            set { this.txtDescription.Text = value; }
        }

        /// <summary>
        /// Get or set the executable for committing edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Confirm the edited contents and commit the modification to the data source.
        /// </summary>
        /// <returns>
        /// False if there are problems in the modifications and should cancel the action.
        /// </returns>
        public bool CommitEditing()
        {
            if (this.OnCommitEditingExecutable != null)
            {
                this.OnCommitEditingExecutable.Execute(new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>("PropertyType", this.PropertyType),
                    new KeyValuePair<string, object>("Name", this.PropertyName),
                    new KeyValuePair<string, object>("ArraySize", this.ArraySize),
                    new KeyValuePair<string, object>("Description", this.PropertyDescription)
                });
            }

            return true;
        }

        /// <summary>
        /// Cancel the edited contents.
        /// </summary>
        /// <param name="deleteContents">True to delete the editing contents.</param>
        public void CancelEditing(bool deleteContents)
        {
            if (deleteContents == true &&
                this.OnCancelEditingExecutable != null)
            {
                this.OnCancelEditingExecutable.Execute(null);
            }
        }
    }
}
