﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// Class for source data model renderable container.
    /// </summary>
    internal class SourceDataModelContainerRenderable :
        RenderableBase,
        ILogicalTreeElement
    {
        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The renderable for rendering the source data model.</summary>
        private SourceDataModelRenderable dataModelRenderable = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public SourceDataModelContainerRenderable(RenderableBase owner) :
            base()
        {
            this.BorderThickness = 0.0f;
            this.BorderColor = Color.Transparent;
            this.FillColor = Color.Transparent;

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.Owner = owner;

            this.dataModelRenderable = new SourceDataModelRenderable(this)
            {
                Visible = false,
                Bounds = new RectangleF(0.0f, 0.0f, this.Width, 0.0f)
            };

            this.Children.Add(this.dataModelRenderable);
            this.AddRenderable(this.dataModelRenderable);

            this.Bindings.Add(new Binder(this, "SourceDataModelContext", "SourceDataModelInstanceViewModel"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data info viewport.
        /// </summary>
        public RenderableBase Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner as ILogicalTreeElement; }
            protected set { this.Owner = value as RenderableBase; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the data context for the source data model renderable.
        /// </summary>
        public object SourceDataModelContext
        {
            get { return this.dataModelRenderable.DataContext; }
            set { this.dataModelRenderable.DataContext = value; }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            this.dataModelRenderable.Bounds = new RectangleF(
                0.0f,
                0.0f,
                this.Width,
                this.dataModelRenderable.Height);

            base.UpdateChildren(context);

            this.Size = new SizeF(this.Width, this.dataModelRenderable.Height);
        }
    }
}
