﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.UIControls.BinaryDataView;
using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// The viewport class for rendering the data model list.
    /// </summary>
    internal class ConversionViewViewport : Viewport, ILogicalTreeElement
    {
        /// <summary>Constant margin of the child renderable.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The renderable for rendering the selected binary data.</summary>
        private BinaryDataInfoRenderable binaryDataRenderable = null;

        /// <summary>The renderable for source data model renderable container.</summary>
        private SourceDataModelContainerRenderable dataModelContainerRenderable = null;

        /// <summary>A line segment renderable as the visual helper for connection.</summary>
        private LineSegment connectionHelperLine;

        /// <summary>The text block that shows the connector information.</summary>
        private TextBlock connectorInfo;

        /// <summary>The target connector to show the information.</summary>
        private ConnectorRenderable connectorInfoTarget = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="renderingControl">The control to be rendered to.</param>
        public ConversionViewViewport(Control renderingControl) :
            base(renderingControl)
        {
            this.Bindings = new BindingContainer(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.RenderContext = new ConnectorRenderContext(this)
            {
                ShowConnector = true
            };

            this.connectionHelperLine = new LineSegment()
            {
                BorderColor = Color.Red,
                BorderThickness = 2.0f,
                Visible = false,
                CanPick = false
            };

            this.AddRenderable(this.connectionHelperLine);

            this.binaryDataRenderable = new BinaryDataInfoRenderable(this)
            {
                Bounds = new RectangleF(
                    ChildMargin,
                    ChildMargin,
                    this.Width - (ChildMargin * 2.0f),
                    0.0f)
            };

            this.dataModelContainerRenderable = new SourceDataModelContainerRenderable(this)
            {
                Bounds = new RectangleF(
                    ChildMargin,
                    ChildMargin,
                    this.Width - (ChildMargin * 2.0f),
                    0.0f)
            };

            this.AddRenderable(this.binaryDataRenderable);
            this.AddRenderable(this.dataModelContainerRenderable);

            this.connectorInfo = new TextBlock()
            {
                TextColor = Color.White,
                FillColor = Color.FromArgb(128, Color.Black),
            };

            this.Bindings.Add(new Binder(this, "BinaryDataDataContext", "DisplayedBinaryDataViewModel"));
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        /// <param name="isDisposing">True if the renderable is being disposed.</param>
        protected override void Dispose(bool isDisposing)
        {
            // バインディングを破棄
            foreach (var binder in this.Bindings) {
                binder.Dispose();
            }

            this.Bindings.Clear();

            base.Dispose(isDisposing);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return null; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Get or set the data context for the binary data info renderable.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object BinaryDataDataContext
        {
            get
            {
                return this.binaryDataRenderable.DataContext;
            }

            set
            {
                this.dataModelContainerRenderable.DataContext = value;
                this.binaryDataRenderable.DataContext = value;
            }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get the size of all the list items.
        /// </summary>
        public SizeF ContentSize
        {
            get
            {
                float maxHeight = Math.Max(
                    this.binaryDataRenderable.Height,
                    this.dataModelContainerRenderable.Height);

                SizeF size = new SizeF(this.Width, maxHeight);

                size.Height += ChildMargin * 2.0f;

                return size;
            }
        }

        /// <summary>
        /// Get the flag indicating whether is connecting or not.
        /// </summary>
        public bool IsConnecting
        {
            get { return this.ConnectingItem != null; }
        }

        /// <summary>
        /// The connectible item that is currently connecting.
        /// </summary>
        public IConnectible ConnectingItem { get; private set; }

        /// <summary>
        /// The connectible item that is currently being connected.
        /// </summary>
        public IConnectible ConnectedItem { get; private set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Show the connection information.
        /// Hide the information if the connector is null.
        /// </summary>
        /// <param name="connector">The connector.</param>
        public void ShowConnectorInfo(ConnectorRenderable connector)
        {
            if (connector != this.connectorInfoTarget)
            {
                this.connectorInfoTarget = connector;

                string text = string.Empty;
                if (connector != null &&
                    connector.Owner is IDestinationConnectible)
                {
                    var owner = (IDestinationConnectible)connector.Owner;

                    // Converter name.
                    text = owner.Description + "\n" + owner.Converter.Name;

                    // Loop through the connected sources.
                    foreach (Guid srcGuid in owner.ConnectedSources)
                    {
                        // Append the type and name of the connected source.
                        ISourceConnectible src =
                            ConnectionSourceManager.FindSourceConnectible(srcGuid);
                        if (src == null)
                        {
                            continue;
                        }

                        text += string.Format("\n- {0} {1}", src.ValueType, src.DisplayName);
                    }
                }
                else if (connector != null &&
                    connector.Owner is ISourceConnectible)
                {
                    text = connector.Owner.Description;

                    foreach (ConnectionRenderable cn in
                        ConnectionRenderable.FindConnections(connector))
                    {
                        if (cn.Destination == null)
                        {
                            continue;
                        }

                        text += string.Format(
                            "\n- {0} {1} ({2})",
                            cn.Destination.ValueType,
                            cn.Destination.DisplayName,
                            cn.Destination.Converter.Name);
                    }
                }

                this.connectorInfo.Text = text;

                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Begin connecting the given connectible item.
        /// </summary>
        /// <param name="connectingItem">The item that is connecting.</param>
        public void BeginConnect(IConnectible connectingItem)
        {
            this.ConnectingItem = connectingItem;
            this.ConnectedItem = null;
        }

        /// <summary>
        /// Perform the connect effect.
        /// </summary>
        /// <param name="cursorLocation">The cursor location.</param>
        /// <param name="connectedItem">The connected item.</param>
        public void PerformConnectEffect(PointF cursorLocation, IConnectible connectedItem)
        {
            if (this.ConnectingItem == null)
            {
                return;
            }

            this.ConnectedItem = connectedItem;

            this.connectionHelperLine.Vertex1 =
                this.ConnectingItem.TransformedConnectorLocation;

            if (this.ConnectedItem == null)
            {
                this.connectionHelperLine.Vertex2 = cursorLocation;
                this.connectionHelperLine.BorderColor = Color.LightCyan;
            }
            else
            {
                this.connectionHelperLine.Vertex2 =
                    this.ConnectedItem.TransformedConnectorLocation;
                if (this.ConnectedItem.CanConnect(this.ConnectingItem) == true)
                {
                    this.connectionHelperLine.BorderColor = Color.DodgerBlue;
                }
                else
                {
                    this.connectionHelperLine.BorderColor = Color.Pink;
                }
            }

            this.connectionHelperLine.Visible = true;

            this.RequestRedraw();
        }

        /// <summary>
        /// Finish the connecting process.
        /// </summary>
        public void EndConnect()
        {
            if (this.ConnectingItem != null &&
                this.ConnectingItem.CanConnect(this.ConnectedItem) == true)
            {
                ISourceConnectible source = null;
                IDestinationConnectible destination = null;
                if (this.ConnectingItem is IDestinationConnectible)
                {
                    source = (ISourceConnectible)this.ConnectedItem;
                    destination = (IDestinationConnectible)this.ConnectingItem;
                }
                else if (this.ConnectedItem is IDestinationConnectible)
                {
                    source = (ISourceConnectible)this.ConnectingItem;
                    destination = (IDestinationConnectible)this.ConnectedItem;
                }

                if (destination != null && source != null)
                {
                    destination.AddConnectionSource(source.SourceGuid);
                }
            }

            this.ConnectingItem = null;
            this.ConnectedItem = null;
            this.connectionHelperLine.Visible = false;
            this.RequestRedraw();
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float listWidth = (this.Width - (ChildMargin * 2.0f)) / 3.0f;

            this.dataModelContainerRenderable.Bounds = new RectangleF(
                ChildMargin,
                ChildMargin,
                listWidth,
                this.dataModelContainerRenderable.Height);

            this.binaryDataRenderable.Bounds = new RectangleF(
                ChildMargin + (listWidth * 2.0f),
                ChildMargin,
                listWidth,
                this.binaryDataRenderable.Height);

            var myContext = context as ConnectorRenderContext;
            if (myContext != null)
            {
                myContext.ConnectingItem = this.ConnectingItem;
                myContext.ConnectedItem = this.ConnectedItem;
            }

            if (this.ConnectingItem != null)
            {
                var invMatrix = this.TransformationMatrix.Clone() as Matrix;
                invMatrix.Invert();

                var pts = new PointF[]
                {
                    this.connectionHelperLine.Vertex1,
                    this.connectionHelperLine.Vertex2
                };

                invMatrix.TransformPoints(pts);

                this.connectionHelperLine.Vertex1 = pts[0];
                this.connectionHelperLine.Vertex2 = pts[1];
            }

            base.UpdateChildren(context);

            this.connectorInfo.Visible =
                (this.connectorInfoTarget != null) &&
                (this.connectorInfoTarget.Visible == true) &&
                (string.IsNullOrEmpty(this.connectorInfo.Text) == false);

            if (this.connectorInfo.Visible == true)
            {
                float posY = (this.connectorInfoTarget.TransformedBounds.Top + this.connectorInfoTarget.TransformedBounds.Bottom) * 0.5f;
                if (this.connectorInfoTarget.ConnectorDirection == ConnectorRenderable.ConnectorDirections.Right)
                {
                    this.connectorInfo.HAlignment = HAlignment.Left;
                    this.connectorInfo.VAlignment = VAlignment.Center;
                    this.connectorInfo.Location = new PointF(
                        this.connectorInfoTarget.TransformedBounds.Right + 20.0f,
                        posY - context.Translation.Y);
                }
                else
                {
                    this.connectorInfo.HAlignment = HAlignment.Right;
                    this.connectorInfo.VAlignment = VAlignment.Center;
                    this.connectorInfo.Location = new PointF(
                        this.connectorInfoTarget.TransformedBounds.Left - 20.0f,
                        posY - context.Translation.Y);
                }
            }

            this.connectorInfo.Update(context);
        }

        /// <summary>
        /// Draw children items.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>True if any children is rendered.</returns>
        protected override bool DrawChildren(Graphics g)
        {
            bool result = base.DrawChildren(g);
            return result | this.connectorInfo.Draw(g);
        }
    }
}
