﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.UIControls.Selection;

using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// The renderable class for rendering the conversion connectors.
    /// </summary>
    internal class ConnectorRenderable : RenderableBase
    {
        /// <summary>The constant for the radius of the connector circle.</summary>
        private const float ConnectorRadius = 5.0f;

        /// <summary>The connector line renderable.</summary>
        private LineSegment connectorLine = new LineSegment();

        /// <summary>The connector circle renderable.</summary>
        private CircleShape connectorCircle = new CircleShape();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the connector.</param>
        public ConnectorRenderable(IConnectible owner) :
            base()
        {
            this.ConnectorID = Guid.NewGuid();

            this.Owner = owner;
            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;

            this.ConnectorLength = 5.0f;
            this.ConnectorDirection = ConnectorDirections.Left;
            this.Offset = SizeF.Empty;

            this.connectorCircle.Radius = new SizeF(ConnectorRadius, ConnectorRadius);
            this.connectorCircle.Center = new PointF(0.0f, 0.0f);
            this.connectorCircle.BorderThickness = 2.0f;
            this.connectorCircle.BorderColor = Color.Gray;
            this.connectorCircle.FillColor = Color.Beige;

            this.connectorLine.BorderThickness = 2.0f;
            this.connectorLine.BorderColor = Color.Gray;

            this.AddRenderable(this.connectorLine);
            this.AddRenderable(this.connectorCircle);
        }

        /// <summary>
        /// Enum for the connector direction.
        /// </summary>
        public enum ConnectorDirections
        {
            /// <summary>The connector is on the left side of the owner.</summary>
            Left,

            /// <summary>The connector is on the right side of the owner.</summary>
            Right,
        }

        /// <summary>
        /// Enum for the relationships between items.
        /// </summary>
        private enum RelationTypes
        {
            /// <summary>The item is this item itself.</summary>
            Self,

            /// <summary>The item is related to this item.</summary>
            Related,

            /// <summary>The item has no relationship with this item.</summary>
            None
        }

        /// <summary>
        /// Get the connector's Guid.
        /// </summary>
        public Guid ConnectorID { get; private set; }

        /// <summary>
        /// Get or set the owner renderable.
        /// </summary>
        public IConnectible Owner { get; private set; }

        /// <summary>
        /// Get or set the connector direction.
        /// </summary>
        public ConnectorDirections ConnectorDirection { get; set; }

        /// <summary>
        /// Get or set the length of the connector.
        /// </summary>
        public float ConnectorLength { get; set; }

        /// <summary>
        /// Get or set the extra offset distance.
        /// </summary>
        public SizeF Offset { get; set; }

        /// <summary>
        /// Get the center location of the connector. (the center of the circle)
        /// </summary>
        public PointF ConnectorLocation { get; private set; }

        /// <summary>
        /// Get the connector location after transformation. (the center of the circle)
        /// </summary>
        public PointF TransformedConnectorLocation { get; private set; }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        public override void Update(RenderContext context)
        {
            if (this.Owner != null)
            {
                base.Update(context);
            }
        }

        /// <summary>
        /// Render the item.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>False when the item is clipped and need not to be rendered.</returns>
        public override bool Draw(Graphics g)
        {
            if (this.Owner == null)
            {
                return false;
            }

            Color lineColor = this.connectorLine.BorderColor;
            Color circleBorderColor = this.connectorCircle.BorderColor;
            Color circleFillColor = this.connectorCircle.FillColor;

            var context = this.SavedRenderContext as ConnectorRenderContext;
            if (context != null)
            {
                if (context.ConnectingItem != null &&
                    this.Owner.CanConnect(context.ConnectingItem) == false)
                {
                    this.connectorLine.BorderColor = Color.Silver;
                    this.connectorCircle.BorderColor = Color.Silver;
                    this.connectorCircle.FillColor = Color.Silver;
                }
            }

            bool result = base.Draw(g);

            this.connectorLine.BorderColor = lineColor;
            this.connectorCircle.BorderColor = circleBorderColor;
            this.connectorCircle.FillColor = circleFillColor;

            return result;
        }

        /// <summary>
        /// Test if the object is picked with the conditions given by the pick context.
        /// </summary>
        /// <param name="context">The pick context.</param>
        public override void Pick(ObjectPickContext context)
        {
            if (this.Visible == false)
            {
                return;
            }

            if (this.Pick(context.PickPoint) == true)
            {
                context.AddPickedObject(this);
            }
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            switch (this.ConnectorDirection)
            {
                case ConnectorDirections.Left:
                    this.Size = new SizeF(
                        ConnectorRadius + this.ConnectorLength,
                        ConnectorRadius * 2.0f);

                    this.Location = new PointF(
                        -this.Width + this.Offset.Width,
                        -ConnectorRadius + this.Offset.Height);

                    this.Translation = new PointF(ConnectorRadius, ConnectorRadius);
                    this.ConnectorLocation = this.Translation;
                    break;

                case ConnectorDirections.Right:
                    this.Size = new SizeF(
                        ConnectorRadius + this.ConnectorLength,
                        ConnectorRadius * 2.0f);

                    this.Location = new PointF(
                        this.Owner.Width - this.Offset.Width,
                        -ConnectorRadius + this.Offset.Height);

                    this.Translation = new PointF(this.ConnectorLength, ConnectorRadius);
                    this.ConnectorLocation = this.Translation;
                    break;
            }

            this.UpdateAppearance();

            base.UpdateSelf(context);

            var p = new PointF(this.X + this.ConnectorLocation.X, this.Y + this.ConnectorLocation.Y);
            var pts = new PointF[] { p };
            context.TransformationMatrix.TransformPoints(pts);
            this.TransformedConnectorLocation = pts[0];
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            switch (this.ConnectorDirection)
            {
                case ConnectorDirections.Left:
                    this.connectorLine.Vertex1 = new PointF(0.0f, 0.0f);
                    this.connectorLine.Vertex2 = new PointF(this.ConnectorLength, 0.0f);
                    break;

                case ConnectorDirections.Right:
                    this.connectorLine.Vertex1 = new PointF(0.0f, 0.0f);
                    this.connectorLine.Vertex2 = new PointF(-this.ConnectorLength, 0.0f);
                    break;
            }

            base.UpdateChildren(context);
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        private void UpdateAppearance()
        {
            object selectedObj =
                SelectionManager.GetSelectedObject(SelectionGroups.BinaryConvertConnection);

            object mouseOverObj =
                SelectionManager.GetMouseOverObject(SelectionGroups.BinaryConvertConnection);

            RelationTypes selectedObjRelation = this.DetermineRelationship(selectedObj);
            RelationTypes mouseOverObjRelation = this.DetermineRelationship(mouseOverObj);

            Color strongColor = Color.Khaki;
            Color lightColor = Color.Beige;
            if (this.Owner.IsConnectionSet == false)
            {
                strongColor = Color.Crimson;
                lightColor = Color.Pink;
            }

            if (selectedObjRelation == RelationTypes.Self)
            {
                this.connectorLine.BorderColor = Color.Black;
                this.connectorCircle.BorderColor = Color.Black;
                this.connectorCircle.FillColor = strongColor;
            }
            else if (mouseOverObjRelation == RelationTypes.Self)
            {
                this.connectorLine.BorderColor = Color.Black;
                this.connectorCircle.BorderColor = Color.Black;
                this.connectorCircle.FillColor = strongColor;
            }
            else if (selectedObjRelation == RelationTypes.Related ||
                     mouseOverObjRelation == RelationTypes.Related)
            {
                this.connectorLine.BorderColor = Color.Black;
                this.connectorCircle.BorderColor = Color.Black;
                this.connectorCircle.FillColor = strongColor;
            }
            else
            {
                this.connectorLine.BorderColor = Color.Gray;
                this.connectorCircle.BorderColor = Color.Gray;
                this.connectorCircle.FillColor = lightColor;
            }
        }

        /// <summary>
        /// Determine the relationship between this item and the specified item.
        /// </summary>
        /// <param name="item">The item to determine relationship with.</param>
        /// <returns>The relationship.</returns>
        private RelationTypes DetermineRelationship(object item)
        {
            if (object.ReferenceEquals(item, this) == true)
            {
                return RelationTypes.Self;
            }
            else
            {
                // The connection is either selected nor mouse over.
                // Check if the selected object is related to this connection.
                if (item is ConnectionRenderable)
                {
                    var connection = (ConnectionRenderable)item;
                    if (connection.Source != null &&
                        connection.Destination != null)
                    {
                        if (connection.Source == this.Owner)
                        {
                            // The source connector of the connection is this connector.
                            return RelationTypes.Self;
                        }
                        else if (connection.Destination == this.Owner)
                        {
                            // The destination connector of the connection is this connector.
                            return RelationTypes.Self;
                        }
                        else
                        {
                            // Check if the connection has the same destination as
                            // any of the connections that connect to this connector.
                            foreach (var rd in connection.Destination.Connections)
                            {
                                if (rd.Source == this.Owner)
                                {
                                    return RelationTypes.Related;
                                }
                            }
                        }
                    }
                }
                else if (item is ConnectorRenderable)
                {
                    var connector = (ConnectorRenderable)item;
                    foreach (var rd in ConnectionRenderable.FindConnections(connector))
                    {
                        // Is the connector connects to this connector?
                        if (rd.Source == this.Owner ||
                            rd.Destination == this.Owner)
                        {
                            return RelationTypes.Related;
                        }
                    }
                }
            }

            return RelationTypes.None;
        }
    }
}
