﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelLogic;
using EffectMaker.DataModelLogic.BinaryConverters;

using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.DataTypes;

using EffectMaker.DataModelMaker.UIControls.Interfaces;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.DataModelMaker.UIControls.BinaryFieldEditor
{
    /// <summary>
    /// User control class for editor data model property editing.
    /// </summary>
    public partial class BinaryFieldEditor : UIUserControl
    {
        /// <summary>The converter parameter map.</summary>
        private Dictionary<string, string> converterParameters = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryFieldEditor()
        {
            this.InitializeComponent();

            var sendModificationTypes = new KeyValuePair<string, SendModificationTypes>[]
            {
                new KeyValuePair<string, SendModificationTypes>(Properties.Resources.SendModificationTypes_ModifiedDataOnly, SendModificationTypes.ModifiedDataOnly),
                new KeyValuePair<string, SendModificationTypes>(Properties.Resources.SendModificationTypes_ModifiedWithReset, SendModificationTypes.ModifiedWithReset),
                new KeyValuePair<string, SendModificationTypes>(Properties.Resources.SendModificationTypes_FullBinary, SendModificationTypes.FullBinary),
            };

            this.cbSendModificationType.DisplayMember = "Key";
            this.cbSendModificationType.ValueMember = "Value";
            this.cbSendModificationType.DataSource = sendModificationTypes;

            var fieldTypes = new KeyValuePair<string, BinaryFieldTypes>[]
            {
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesNormal, BinaryFieldTypes.Normal),
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesFixedSize, BinaryFieldTypes.FixedSize),
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesByteAlignment, BinaryFieldTypes.ByteAlignment),
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesDataOffset, BinaryFieldTypes.DataOffset),
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesDataSize, BinaryFieldTypes.DataSize),
                new KeyValuePair<string, BinaryFieldTypes>(Properties.Resources.BinaryFieldTypesSelectBinaryData, BinaryFieldTypes.SelectBinaryData),
            };

            this.cbFieldType.DisplayMember = "Key";
            this.cbFieldType.ValueMember = "Value";
            this.cbFieldType.DataSource = fieldTypes;

            this.cbConverter.DisplayMember = "Name";
            foreach (ConverterInfo info in ConverterManager.Converters)
            {
                this.cbConverter.Items.Add(info);
            }

            this.Bindings.Add(new Binder(this, "ValueType", "TargetType"));
            this.Bindings.Add(new Binder(this, "FieldName", "TargetName"));
            this.Bindings.Add(new Binder(this, "FieldTag", "Tag"));
            this.Bindings.Add(new Binder(this, "SendModificationType", "SendModificationType"));
            this.Bindings.Add(new Binder(this, "FieldType", "FieldType"));
            this.Bindings.Add(new Binder(this, "FieldSize", "FieldSize"));
            this.Bindings.Add(new Binder(this, "Converter", "ConverterInfo"));
            this.Bindings.Add(new Binder(this, "ConverterParameters", "ConverterParamKeyValuePairs"));
            this.Bindings.Add(new Binder(this, "OnCommitEditingExecutable", "OnCommitEditingExecutable"));
        }

        /// <summary>
        /// Get or set the value type of the field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string ValueType
        {
            get { return this.txtType.Text; }
            set { this.txtType.Text = value; }
        }

        /// <summary>
        /// Get or set the name of the field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string FieldName
        {
            get { return this.txtName.Text; }
            set { this.txtName.Text = value; }
        }

        /// <summary>
        /// Get or set the value type of the field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string FieldTag
        {
            get { return this.txtTag.Text; }
            set { this.txtTag.Text = value; }
        }

        /// <summary>
        /// Get or set the type of message to send when the data is modified.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public SendModificationTypes SendModificationType
        {
            get
            {
                if (this.cbSendModificationType.SelectedValue is SendModificationTypes)
                {
                    return (SendModificationTypes)this.cbSendModificationType.SelectedValue;
                }
                else
                {
                    return SendModificationTypes.ModifiedDataOnly;
                }
            }

            set
            {
                this.cbSendModificationType.SelectedValue = value;
            }
        }

        /// <summary>
        /// Get or set the type of the field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public BinaryFieldTypes FieldType
        {
            get
            {
                if (this.cbFieldType.SelectedValue is BinaryFieldTypes)
                {
                    return (BinaryFieldTypes)this.cbFieldType.SelectedValue;
                }
                else
                {
                    return BinaryFieldTypes.Normal;
                }
            }

            set
            {
                this.cbFieldType.SelectedValue = value;
            }
        }

        /// <summary>
        /// Get or set the size of the field in bytes.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int FieldSize
        {
            get
            {
                if (this.FieldType == BinaryFieldTypes.Normal)
                {
                    return 0;
                }
                else
                {
                    return (int)this.inpSize.Value;
                }
            }

            set
            {
                this.inpSize.Value = value;
            }
        }

        /// <summary>
        /// Get or set the binary converter to use for this field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ConverterInfo Converter
        {
            get
            {
                if (this.cbConverter.SelectedItem is ConverterInfo)
                {
                    return (ConverterInfo)this.cbConverter.SelectedItem;
                }
                else
                {
                    return ConverterManager.DefaultConverter;
                }
            }

            set
            {
                this.cbConverter.SelectedItem = value;
                this.UpdateConverterParameterListView();
            }
        }

        /// <summary>
        /// Get or set the binary converter to use for this field.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Dictionary<string, string> ConverterParameters
        {
            get
            {
                return this.converterParameters;
            }

            set
            {
                this.converterParameters = value;
                this.UpdateConverterParameterListView();
            }
        }

        /// <summary>
        /// Get or set the executable for committing edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Confirm the edited contents and commit the modification to the data source.
        /// </summary>
        /// <returns>
        /// False if there are problems in the modifications and should cancel the action.
        /// </returns>
        public bool CommitEditing()
        {
            if (this.OnCommitEditingExecutable != null)
            {
                var converterParams = this.MergeConverterParameters();

                this.OnCommitEditingExecutable.Execute(new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>("Tag", this.FieldTag),
                    new KeyValuePair<string, object>("SendModificationType", this.SendModificationType),
                    new KeyValuePair<string, object>("FieldType", this.FieldType),
                    new KeyValuePair<string, object>("FieldSize", this.FieldSize),
                    new KeyValuePair<string, object>("ConverterInfo", this.Converter),
                    new KeyValuePair<string, object>("ConverterParamKeyValuePairs", converterParams)
                });
            }

            return true;
        }

        /// <summary>
        /// Handle field type SelectedIndexChanged event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnFieldTypeChanged(object sender, EventArgs e)
        {
            BinaryFieldTypes fieldType = BinaryFieldTypes.Normal;
            if (this.cbFieldType.SelectedValue is BinaryFieldTypes)
            {
                fieldType = (BinaryFieldTypes)this.cbFieldType.SelectedValue;
            }

            switch (fieldType)
            {
                case BinaryFieldTypes.FixedSize:
                    this.lblSize.Text = Properties.Resources.SizeLabelText;
                    this.lblSize.Enabled = true;
                    this.inpSize.Enabled = true;
                    this.inpSize.Maximum = 65535;
                    this.inpSize.Minimum = 0;
                    break;

                case BinaryFieldTypes.ByteAlignment:
                    this.lblSize.Text = Properties.Resources.AlignmentLabelText;
                    this.lblSize.Enabled = true;
                    this.inpSize.Enabled = true;
                    this.inpSize.Maximum = 65535;
                    this.inpSize.Minimum = 0;
                    break;

                case BinaryFieldTypes.DataOffset:
                    this.lblSize.Text = Properties.Resources.AdjustValueLabelText;
                    this.lblSize.Enabled = true;
                    this.inpSize.Enabled = true;
                    this.inpSize.Maximum = 65535;
                    this.inpSize.Minimum = -65535;
                    break;

                case BinaryFieldTypes.DataSize:
                    this.lblSize.Text = Properties.Resources.AdjustValueLabelText;
                    this.lblSize.Enabled = true;
                    this.inpSize.Enabled = true;
                    this.inpSize.Maximum = 65535;
                    this.inpSize.Minimum = -65535;
                    break;

                case BinaryFieldTypes.SelectBinaryData:
                    this.lblSize.Text = Properties.Resources.SizeLabelText;
                    this.lblSize.Enabled = false;
                    this.inpSize.Enabled = false;
                    break;

                default:
                    this.lblSize.Text = Properties.Resources.SizeLabelText;
                    this.lblSize.Enabled = false;
                    this.inpSize.Enabled = false;
                    break;
            }
        }

        /// <summary>
        /// Handle converter type SelectedIndexChanged event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnConverterTypeChanged(object sender, EventArgs e)
        {
            this.UpdateConverterParameterListView();
        }

        /// <summary>
        /// Merge the converter parameter map and the parameters modified on the UI.
        /// </summary>
        /// <returns>The merged converter parameter map.</returns>
        private Dictionary<string, string> MergeConverterParameters()
        {
            Dictionary<string, string> map = this.ConverterParameters;
            if (map == null)
            {
                map = new Dictionary<string, string>();
            }

            foreach (DataGridViewRow row in this.converterParamGridView.Rows)
            {
                string name = row.Cells[0].Value as string;
                string value = row.Cells[1].Value as string;

                name = name.Trim();
                value = value.Trim();
                if (string.IsNullOrEmpty(value) == false)
                {
                    this.ConverterParameters[name] = value;
                }
            }

            return this.ConverterParameters;
        }

        /// <summary>
        /// Update the converter parameter list view.
        /// </summary>
        private void UpdateConverterParameterListView()
        {
            this.converterParamGridView.Rows.Clear();

            if (this.Converter == null)
            {
                return;
            }

            if (this.converterParamGridView == null ||
                this.converterParamGridView.ColumnCount <= 0)
            {
                return;
            }

            ConverterBase converter = this.Converter.Converter;
            foreach (string paramName in converter.EnumerateParameterNames())
            {
                string value;
                if (this.ConverterParameters != null &&
                    this.ConverterParameters.TryGetValue(paramName, out value) == true)
                {
                    this.converterParamGridView.Rows.Add(paramName, value);
                }
                else
                {
                    this.converterParamGridView.Rows.Add(paramName, string.Empty);
                }
            }
        }

        /// <summary>
        /// Handle CellValidataing event for the converter parameter data grid view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnValidatingConverterParameter(
            object sender,
            DataGridViewCellValidatingEventArgs e)
        {
            if (this.Converter == null)
            {
                return;
            }

            if (this.converterParamGridView == null)
            {
                return;
            }

            // Only validate the parameter values. (the values on column 1)
            if (e.ColumnIndex != 1)
            {
                return;
            }

            var row = this.converterParamGridView.Rows[e.RowIndex];

            string name = row.Cells[0].Value as string;
            string value = e.FormattedValue as string;

            name = name.Trim();
            value = value.Trim();

            if (string.IsNullOrEmpty(name) == false &&
                string.IsNullOrEmpty(value) == false &&
                this.Converter.Converter.ValidateParameter(name, value) == false)
            {
                row.ErrorText = Properties.Resources.ConverterParameterValueInvalid;
                e.Cancel = true;
            }
            else
            {
                row.ErrorText = string.Empty;
            }
        }
    }
}
