﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// The viewport class for rendering the data model list.
    /// </summary>
    internal class BinaryFieldGroupTargetListViewport : Viewport, ILogicalTreeElement
    {
        /// <summary>Constant list item spacing.</summary>
        private const float ListItemSpacing = 5.0f;

        /// <summary>Constant list item spacing.</summary>
        private const float ListItemMargin = 5.0f;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="renderingControl">The control to be rendered to.</param>
        public BinaryFieldGroupTargetListViewport(Control renderingControl) :
            base(renderingControl)
        {
            this.Bindings = new BindingContainer(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "TargetListSource", "BinaryFieldGroupTargetViewModels"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return null; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the binary field group target source.
        /// </summary>
        public IEnumerable TargetListSource
        {
            get
            {
                IEnumerable result =
                    from ch in this.Children
                    where ch is BinaryFieldGroupTargetListItemRenderable
                    select ch.DataContext;

                return result;
            }

            set
            {
                this.UpdateItemsFromDataSource(value);
            }
        }

        /// <summary>
        /// Get the size of all the list items.
        /// </summary>
        public SizeF ContentSize { get; private set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ListItemMargin * 2.0f);

            // Loop through list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
            }

            base.UpdateChildren(context);

            float height = ListItemMargin;

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                if ((child is BinaryFieldGroupTargetListItemRenderable) == false)
                {
                    continue;
                }

                // Setup location and size.
                child.Location = new PointF(ListItemMargin, height);

                height += child.Height + ListItemSpacing;
            }

            this.ContentSize = new SizeF(childWidth, height);
        }

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            // Clear all the data model renderers first.
            var targetListItemRenderableList =
                this.Renderables.Where(itm => itm is BinaryFieldGroupTargetListItemRenderable).AsEnumerable();

            this.RemoveRenderableRange(targetListItemRenderableList);

            float height = ListItemMargin;
            float maxWidth = 0.0f;

            // Loop through data model source items.
            foreach (object item in items)
            {
                // Create a data model view for the data model source.
                var renderable = new BinaryFieldGroupTargetListItemRenderable(this);

                // Set up the data context.
                renderable.DataContext = item;

                // Setup location and size.
                renderable.Location = new PointF(ListItemMargin, height);

                height += renderable.Height + ListItemSpacing;
                maxWidth = Math.Max(maxWidth, renderable.Width + (ListItemMargin * 2.0f));

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            this.ContentSize = new SizeF(maxWidth, height);
        }
    }
}
