﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Converters;

using EffectMaker.DataModelMaker.UIControls.ConversionView;
using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Class for rendering a binary field group.
    /// </summary>
    internal class BinaryDataInfoFieldGroupRenderable :
        RoundedRectangleShape,
        ILogicalTreeElement,
        IDestinationConnectible
    {
        /// <summary>Constant for the margin of the property renderables.</summary>
        public const float ChildMargin = 5.0f;

        /// <summary>Constant for the height of the binary field group information area.</summary>
        private const float FieldGroupInfoHeight = 58.0f;

        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The flag indicating whether should show the connector or not.</summary>
        private bool shouldShowConnector = true;

        /// <summary>The child renderables for rendering the binary fields.</summary>
        private List<BinaryDataInfoFieldRenderable> binaryFieldRenderables =
            new List<BinaryDataInfoFieldRenderable>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryDataInfoFieldGroupRenderable()
        {
            // Setup the fonts for rendering.
            smallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            largeFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public BinaryDataInfoFieldGroupRenderable(BinaryDataInfoFieldRenderable owner) :
            base()
        {
            this.Size = new SizeF(0.0f, FieldGroupInfoHeight);

            this.Translation = new PointF(ChildMargin, FieldGroupInfoHeight);
            this.BorderColor = Color.CadetBlue;
            this.BorderThickness = 2.0f;
            this.FillColor = Color.Beige;
            this.SetCornerRadius(5.0f);

            this.SeparatorLineColor = Color.Silver;

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);

            this.Owner = owner;

            this.Connector = new ConnectorRenderable(this)
            {
                ConnectorLength = 20.0f,
                ConnectorDirection = ConnectorRenderable.ConnectorDirections.Left,
                Offset = new SizeF(0.0f, -15.0f)
            };

            this.AddRenderable(this.Connector);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "IsTargetBinaryData", "IsTargetBinaryData"));
            this.Bindings.Add(new Binder(this, "FieldGroupGuid", "Guid"));
            this.Bindings.Add(new Binder(this, "TargetName", "TargetName"));
            this.Bindings.Add(new Binder(this, "TargetGuid", "TargetGuid"));
            this.Bindings.Add(new Binder(this, "BinaryFieldDataSources", "BinaryFieldViewModels"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data info renderable.
        /// </summary>
        public BinaryDataInfoFieldRenderable Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as BinaryDataInfoFieldRenderable; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get the connector.
        /// </summary>
        public ConnectorRenderable Connector { get; private set; }

        /// <summary>
        /// Get the location of the connector.
        /// </summary>
        public PointF ConnectorLocation
        {
            get { return this.Connector.ConnectorLocation; }
        }

        /// <summary>
        /// Get the transformed location of the connector.
        /// </summary>
        public PointF TransformedConnectorLocation
        {
            get { return this.Connector.TransformedConnectorLocation; }
        }

        /// <summary>
        /// Enumerate the Guid of the connected sources.
        /// </summary>
        public IEnumerable<Guid> ConnectedSources
        {
            get
            {
                if (this.Owner == null)
                {
                    return null;
                }

                return this.Owner.ConnectedSources;
            }
        }

        /// <summary>
        /// Enumerate connections that connect to the connectible.
        /// </summary>
        public IEnumerable<ConnectionRenderable> Connections
        {
            get
            {
                if (this.Owner == null)
                {
                    return null;
                }

                return this.Owner.Connections;
            }
        }

        /// <summary>
        /// Get the flag indicating whether the connection of the connectible is set.
        /// </summary>
        public bool IsConnectionSet
        {
            get
            {
                if (this.Owner == null)
                {
                    return false;
                }

                return this.Owner.IsConnectionSet;
            }
        }

        /// <summary>
        /// Get or set the converter for converting the data from the source connectible.
        /// </summary>
        public ConverterInfo Converter
        {
            get
            {
                if (this.Owner == null)
                {
                    return null;
                }

                return this.Owner.Converter;
            }

            set
            {
                if (this.Owner != null)
                {
                    this.Owner.Converter = value;
                }
            }
        }

        /// <summary>
        /// Get the description of the connectible.
        /// </summary>
        public string Description
        {
            get { return this.Owner.Description; }
        }

        /// <summary>
        /// Get the string as the type of the connectible value.
        /// </summary>
        public string ValueType
        {
            get { return this.TargetName; }
        }

        /// <summary>
        /// Get the name of the connectible.
        /// </summary>
        public string DisplayName
        {
            get { return this.Owner.TargetName; }
        }

        /// <summary>
        /// Get or set the color of separator lines.
        /// </summary>
        public Color SeparatorLineColor { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the referencing target is binary data or not.
        /// (Not if the target is a runtime data model.)
        /// </summary>
        public bool IsTargetBinaryData { get; set; }

        /// <summary>
        /// Get or set the target name.
        /// (the name of the referencing runtime data model or binary data definition.)
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// Get or set the target Guid.
        /// (the Guid of the referencing runtime data model or binary data definition.)
        /// </summary>
        public Guid TargetGuid { get; set; }

        /// <summary>
        /// Get or set Guid of the binary field group.
        /// </summary>
        public string FieldGroupGuid { get; set; }

        /// <summary>
        /// Get or set the binary field data sources.
        /// </summary>
        public IEnumerable BinaryFieldDataSources
        {
            get { return this.binaryFieldRenderables.Select(pr => pr.DataContext).AsEnumerable(); }
            set { this.UpdateItemsFromDataSource(value); }
        }

        /// <summary>
        /// Enumerates all the child binary field renderables.
        /// </summary>
        public IEnumerable<BinaryDataInfoFieldRenderable> BinaryFieldRenderables
        {
            get { return this.binaryFieldRenderables; }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Determine if the specified target can be connected to this connectible item.
        /// </summary>
        /// <param name="target">The connection target.</param>
        /// <returns>True if can connect.</returns>
        public bool CanConnect(IConnectible target)
        {
            if (target == null ||
                target is IDestinationConnectible)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Update all the connections.
        /// </summary>
        public void UpdateConnections()
        {
            foreach (BinaryDataInfoFieldRenderable rd in this.binaryFieldRenderables)
            {
                rd.UpdateConnections();
            }
        }

        /// <summary>
        /// Add a new connection source.
        /// </summary>
        /// <param name="sourceGuid">The Guid of the connection source.</param>
        public void AddConnectionSource(Guid sourceGuid)
        {
            if (this.Owner != null)
            {
                this.Owner.AddConnectionSource(sourceGuid);
            }
        }

        /// <summary>
        /// Remove existing connection sources.
        /// </summary>
        /// <param name="sourceGuidList">The list of Guid of the connection sources.</param>
        public void RemoveConnectionSources(IEnumerable<Guid> sourceGuidList)
        {
            if (this.Owner != null)
            {
                this.Owner.RemoveConnectionSources(sourceGuidList);
            }
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);

            var childRenderable = child as BinaryDataInfoFieldRenderable;
            if (childRenderable != null)
            {
                this.binaryFieldRenderables.Add(childRenderable);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);

            foreach (RenderableBase child in children)
            {
                var childRenderable = child as BinaryDataInfoFieldRenderable;
                if (childRenderable != null)
                {
                    this.binaryFieldRenderables.Add(childRenderable);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);

            var childRenderable = child as BinaryDataInfoFieldRenderable;
            if (childRenderable != null)
            {
                this.binaryFieldRenderables.Remove(childRenderable);
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);

            // To prevent modifying the property renderable list while looping it,
            // copy the children to another array first.
            var tmpChildren = children;
            if (tmpChildren == this.binaryFieldRenderables)
            {
                tmpChildren = children.ToArray();
            }

            foreach (RenderableBase child in tmpChildren)
            {
                var childRenderable = child as BinaryDataInfoFieldRenderable;
                if (childRenderable != null)
                {
                    this.binaryFieldRenderables.Remove(childRenderable);
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();

            this.binaryFieldRenderables.Clear();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ChildMargin * 2.0f);

            // Loop through binary field renderables to update the content size.
            float location = 0.0f;
            foreach (RenderableBase child in this.binaryFieldRenderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
                child.Location = new PointF(child.X, location);

                location += child.Height;
            }

            var myRC = context as ConnectorRenderContext;
            if (myRC != null &&
                myRC.ShowConnector == true &&
                this.shouldShowConnector == true)
            {
                // Update connector length
                float length = 15.0f;
                ILogicalTreeElement element = this.Parent;
                while (element != null)
                {
                    if (element is BinaryDataInfoFieldRenderable)
                    {
                        length += BinaryDataInfoFieldRenderable.ChildMargin;
                    }

                    element = element.Parent;
                }

                this.Connector.Visible = true;
                this.Connector.ConnectorLength = length;
            }
            else
            {
                this.Connector.Visible = false;
            }

            base.UpdateChildren(context);

            // Loop through binary field renderables to find out the height of the children.
            float height = FieldGroupInfoHeight;
            foreach (RenderableBase child in this.binaryFieldRenderables)
            {
                height += child.Height;
            }

            this.Size = new SizeF(this.Width, height);
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            base.DrawBackground(g);

            // Draw the separation line on the top edge of the field renderables.
            using (var pen = new Pen(this.SeparatorLineColor, 1.0f))
            {
                foreach (var fieldRenderable in this.binaryFieldRenderables)
                {
                    var rect = fieldRenderable.TransformedBounds;
                    g.DrawLine(pen, rect.Location, new PointF(rect.Right, rect.Top));
                }
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            string strTargetType;
            if (this.IsTargetBinaryData == true)
            {
                strTargetType = Properties.Resources.BinaryFieldGroupTargetTypeBinaryData;
            }
            else
            {
                strTargetType = Properties.Resources.BinaryFieldGroupTargetTypeDataModel;
            }

            // Compute the text size of the target type (binary data or data model).
            SizeF targetTypeTextSize = g.MeasureString(
                strTargetType,
                BinaryDataInfoFieldGroupRenderable.smallFont);

            // Compute the text size of the target type name (name of the binary data or data model).
            SizeF targetTypeNameTextSize = g.MeasureString(
                this.TargetName,
                BinaryDataInfoFieldGroupRenderable.smallFont);

            // Compute the text size of the target name (the property name).
            SizeF targetNameTextSize = g.MeasureString(
                this.Owner.TargetName,
                BinaryDataInfoFieldGroupRenderable.largeFont);

            // The rectangle for the target type.
            RectangleF targetTypeTextRect = rect;
            targetTypeTextRect.X += 3.0f;
            targetTypeTextRect.Y += 6.0f;
            targetTypeTextRect.Width = rect.Width - 3.0f;
            targetTypeTextRect.Height = targetTypeTextSize.Height;

            // The rectangle for the target type name.
            RectangleF targetTypeNameTextRect = rect;
            targetTypeNameTextRect.X += 3.0f;
            targetTypeNameTextRect.Y = targetTypeTextRect.Bottom;
            targetTypeNameTextRect.Width = rect.Width - 3.0f;
            targetTypeNameTextRect.Height = targetTypeTextSize.Height;

            // The rectangle for the target name.
            RectangleF targetNameTextRect = rect;
            targetNameTextRect.X += 2.0f;
            targetNameTextRect.Y = rect.Top + FieldGroupInfoHeight - targetNameTextSize.Height - 5.0f;
            targetNameTextRect.Width = rect.Width - 2.0f;
            targetNameTextRect.Height = targetNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    strTargetType,
                    BinaryDataInfoFieldGroupRenderable.smallFont,
                    Brushes.DarkKhaki,
                    targetTypeTextRect,
                    strFormat);

                g.DrawString(
                    this.TargetName,
                    BinaryDataInfoFieldGroupRenderable.smallFont,
                    Brushes.DarkCyan,
                    targetTypeNameTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;

                g.DrawString(
                    this.Owner.TargetName,
                    BinaryDataInfoFieldGroupRenderable.largeFont,
                    Brushes.DarkBlue,
                    targetNameTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            // Clear all the binary field renderers first.
            this.RemoveRenderableRange(this.binaryFieldRenderables);

            float height = 0.0f;

            // Loop through binary field data source items.
            foreach (object item in items)
            {
                // Create a property renderable for the item source.
                var renderable = new BinaryDataInfoFieldRenderable(this);

                // Set up the data context.
                renderable.DataContext = item;

                // Setup location and size.
                renderable.Location = new PointF(0.0f, height);

                height += renderable.Height;

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            this.Size = new SizeF(this.Width, height + FieldGroupInfoHeight);
        }
    }
}
