﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.Interfaces;

using EffectMaker.Foundation.Interfaces;

using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataEditor
{
    /// <summary>
    /// User control class for editor data model editing.
    /// </summary>
    public partial class BinaryDataEditor : UIUserControl, IDataModelEditor
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryDataEditor()
        {
            this.InitializeComponent();

            this.AddBinding("BinaryDataFileName", "FileName");
            this.AddBinding("BinaryDataName", "Name");
            this.AddBinding("HasBinaryHeader", "HasBinaryHeader");
            this.AddBinding("IsUserData", "IsUserData");
            this.AddBinding("OnCommitEditingExecutable", "OnCommitEditingExecutable");
            this.AddBinding("OnCancelEditingExecutable", "OnCancelEditingExecutable");

            var binder1 = new Binder(
                this.cbSourceDataModel,
                "AvailableItems",
                "AvailableSourceDataModels");
            binder1.Mode = BindingMode.OneWay;

            this.Bindings.Add(binder1);

            var binder2 = new Binder(
                this.cbSourceDataModel,
                "SelectedItem",
                "SourceDataModelGuid");
            binder2.Mode = BindingMode.OneWay;

            this.Bindings.Add(binder2);

            this.cbSourceDataModel.DrawItem += this.OnDrawSourceDataModelComboBoxItem;
        }

        /// <summary>
        /// Get or set the source file name the binary data should be defined.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string BinaryDataFileName
        {
            get { return this.txtFileName.Text; }
            set { this.txtFileName.Text = value; }
        }

        /// <summary>
        /// Get or set the name of the binary data.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string BinaryDataName
        {
            get { return this.txtName.Text; }
            set { this.txtName.Text = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether the binary data has binary header.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool HasBinaryHeader
        {
            get { return this.chkBinaryHeader.Checked; }
            set { this.chkBinaryHeader.Checked = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether the binary data is user data.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsUserData
        {
            get { return this.chkUserData.Checked; }
            set { this.chkUserData.Checked = value; }
        }

        /// <summary>
        /// Get or set the executable for committing edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Confirm the edited contents and commit the modification to the data source.
        /// </summary>
        /// <returns>
        /// False if there are problems in the modifications and should cancel the action.
        /// </returns>
        public bool CommitEditing()
        {
            if ((this.cbSourceDataModel.SelectedItem is Guid) == false)
            {
                return false;
            }

            if (this.OnCommitEditingExecutable != null)
            {
                this.OnCommitEditingExecutable.Execute(new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>("FileName", this.BinaryDataFileName),
                    new KeyValuePair<string, object>("Name", this.BinaryDataName),
                    new KeyValuePair<string, object>("HasBinaryHeader", this.HasBinaryHeader),
                    new KeyValuePair<string, object>("IsUserData", this.IsUserData),
                    new KeyValuePair<string, object>("SourceDataModelGuid", (Guid)this.cbSourceDataModel.SelectedItem),
                });
            }

            return true;
        }

        /// <summary>
        /// Cancel the edited contents.
        /// </summary>
        /// <param name="deleteContents">True to delete the editing contents.</param>
        public void CancelEditing(bool deleteContents)
        {
            if (deleteContents == true &&
                this.OnCancelEditingExecutable != null)
            {
                this.OnCancelEditingExecutable.Execute(null);
            }
        }

        /// <summary>
        /// Handle DrawItem event for the source data model combo box item.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDrawSourceDataModelComboBoxItem(object sender, DrawItemEventArgs e)
        {
            UIComboBox combobox = sender as UIComboBox;
            if (sender != this.cbSourceDataModel)
            {
                return;
            }

            string text = string.Empty;
            if (e.Index < 0)
            {
                if (this.cbSourceDataModel.SelectedIndex >= 0)
                {
                    text = this.cbSourceDataModel.GetItemText(this.cbSourceDataModel.SelectedIndex);
                }
            }
            else
            {
                text = this.cbSourceDataModel.GetItemText(this.cbSourceDataModel.Items[e.Index]);
            }

            int lastPeriodIndex = text.LastIndexOf('.');
            string nameSpace = string.Empty;
            if (lastPeriodIndex > 0)
            {
                nameSpace = "(" + text.Substring(0, lastPeriodIndex) + ")";
            }

            string name = text.Substring(lastPeriodIndex + 1);

            var savedTextRenderingHint = e.Graphics.TextRenderingHint;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            e.DrawBackground();

            SizeF nameStrSize = e.Graphics.MeasureString(name, e.Font);

            using (var brush = new SolidBrush(e.ForeColor))
            {
                e.Graphics.DrawString(name, e.Font, brush, e.Bounds.Location);

                if (string.IsNullOrEmpty(nameSpace) == false)
                {
                    brush.Color = Color.Gray;
                    e.Graphics.DrawString(
                        nameSpace,
                        e.Font,
                        brush,
                        e.Bounds.Left + (int)nameStrSize.Width + 5,
                        e.Bounds.Top);
                }
            }

            e.DrawFocusRectangle();

            e.Graphics.TextRenderingHint = savedTextRenderingHint;
        }
    }
}
