﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.Layout;

namespace EffectMaker.DataModelMaker.UIControls.BasicControls
{
    /// <summary>
    /// Control class for image buttons.
    /// </summary>
    public class ImageButton : Panel
    {
        /// <summary>The button state.</summary>
        private ButtonStates buttonState = ButtonStates.Normal;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ImageButton()
        {
        }

        /// <summary>
        /// The enum to represent the button states.
        /// </summary>
        private enum ButtonStates
        {
            /// <summary>The normal button state.</summary>
            Normal = 0x00000000,

            /// <summary>The button state when mouse over.</summary>
            MouseOver = 0x00000001,

            /// <summary>The button state when mouse button down on the button.</summary>
            MouseDown = 0x00000002,
        }

        /// <summary>
        /// Get or set the image to show as the button.
        /// </summary>
        public Image ButtonImage { get; set; }

        /// <summary>
        /// Get or set the image to show when mouse over.
        /// </summary>
        public Image MouseOverImage { get; set; }

        /// <summary>
        /// Get or set the image to show when mouse down.
        /// </summary>
        public Image MouseDownImage { get; set; }

        /// <summary>
        /// Get or set the image to show when disabled.
        /// </summary>
        public Image DisabledImage { get; set; }

        /// <summary>
        /// Get or set the horizontal alignment for the image.
        /// </summary>
        public HAlignment HorizontalAlignment { get; set; }

        /// <summary>
        /// Get or set the vertical alignment for the image.
        /// </summary>
        public VAlignment VerticalAlignment { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether alpha blending is enabled.
        /// </summary>
        public bool IsAlphaBlendingEnabled { get; set; }

        /// <summary>
        /// Handle MouseEnter event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseEnter(EventArgs e)
        {
            this.buttonState |= ButtonStates.MouseOver;

            base.OnMouseEnter(e);
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            this.buttonState &= ~ButtonStates.MouseOver;

            base.OnMouseLeave(e);
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            this.buttonState |= ButtonStates.MouseDown;

            base.OnMouseDown(e);
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            this.buttonState &= ~ButtonStates.MouseDown;

            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            // Determine the image to render.
            Image image = this.ButtonImage;
            if (this.Enabled == false)
            {
                if (this.DisabledImage != null)
                {
                    image = this.DisabledImage;
                }
            }
            else if ((this.MouseDownImage != null) &&
                     (this.buttonState & ButtonStates.MouseDown) != 0)
            {
                image = this.MouseDownImage;
            }
            else if ((this.MouseOverImage != null) &&
                     (this.buttonState & ButtonStates.MouseOver) != 0)
            {
                image = this.MouseOverImage;
            }

            // Determine the size and location for rendering the image.
            var destRect = this.DisplayRectangle;
            if (this.HorizontalAlignment != HAlignment.Stretch)
            {
                destRect.Width = image.Width;

                // The horizontal location.
                if (this.HorizontalAlignment == HAlignment.Center)
                {
                    destRect.X += (this.DisplayRectangle.Width - image.Width) / 2;
                }
                else if (this.HorizontalAlignment == HAlignment.Right)
                {
                    destRect.X += this.DisplayRectangle.Width - image.Width;
                }
            }

            if (this.VerticalAlignment != VAlignment.Stretch)
            {
                destRect.Height = image.Height;

                // The vertical location.
                if (this.VerticalAlignment == VAlignment.Center)
                {
                    destRect.Y += (this.DisplayRectangle.Height - image.Height) / 2;
                }
                else if (this.VerticalAlignment == VAlignment.Bottom)
                {
                    destRect.Y += this.DisplayRectangle.Height - image.Height;
                }
            }

            // Render the image.
            e.Graphics.DrawImage(image, destRect);
        }
    }
}
