﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Properties;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.Generators
{
    //// ☆ パディングについて ☆
    ////
    //// -- struct終端のパディング
    //// struct s1 {
    ////     s32 a;  // pos: 0
    ////     u8  b;  // pos: 4
    //// }
    ////
    //// sizeof(s1) == 8
    ////
    //// -- property前のパディング
    //// struct s2 {
    ////     u8  a[6];  // pos: 0
    ////     s32 b;     // pos: 8
    //// }
    ////
    //// sizeof(s2) == 12
    ////
    //// -- アライメントサイズの伝染
    //// struct s3 {
    ////     u8 a[2];  // pos: 0
    ////     s2 b;     // pos: 4
    //// }
    ////
    //// sizeof(s3) == 16
    ////

    /// <summary>
    /// ランタイムデータモデルバリデータです.
    /// </summary>
    public class RuntimeDataModelValidator
    {
        /// <summary>
        /// 定義データです.
        /// </summary>
        private RuntimeDataModelRootDefinition root;

        /// <summary>
        /// パディングデータです.
        /// </summary>
        private Dictionary<string, PaddingInfo> paddingInfo;

        /// <summary>
        /// 定数値です.
        /// 配列の要素数を解決するため使います.
        /// </summary>
        private Dictionary<string, int> constants;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelValidator(RuntimeDataModelRootDefinition definition)
        {
            this.root = definition;
            this.paddingInfo = new Dictionary<string,PaddingInfo>();
            this.constants = new Dictionary<string, int>();

            // primitiveな型を登録
            // TODO: ランタイムデータ定義的な所でまとめて管理したい
            this.paddingInfo.Add("bool", new PaddingInfo(1, 1));
            this.paddingInfo.Add("char", new PaddingInfo(1, 1));
            this.paddingInfo.Add("u8", new PaddingInfo(1, 1));
            this.paddingInfo.Add("s16", new PaddingInfo(2, 2));
            this.paddingInfo.Add("u16", new PaddingInfo(2, 2));
            this.paddingInfo.Add("s32", new PaddingInfo(4, 4));
            this.paddingInfo.Add("u32", new PaddingInfo(4, 4));
            this.paddingInfo.Add("u64", new PaddingInfo(8, 8));
            this.paddingInfo.Add("float", new PaddingInfo(4, 4));
            this.paddingInfo.Add("f32", new PaddingInfo(4, 4));
            this.paddingInfo.Add("nw::math::VEC2", new PaddingInfo(4, 8));
            this.paddingInfo.Add("nw::math::VEC3", new PaddingInfo(4, 12));
            this.paddingInfo.Add("nw::math::VEC4", new PaddingInfo(4, 16));
            this.paddingInfo.Add("nw::eftcom::Guid", new PaddingInfo(1, 16));
            this.paddingInfo.Add("uint8_t", new PaddingInfo(1, 1));
            this.paddingInfo.Add("int16_t", new PaddingInfo(2, 2));
            this.paddingInfo.Add("uint16_t", new PaddingInfo(2, 2));
            this.paddingInfo.Add("int32_t", new PaddingInfo(4, 4));
            this.paddingInfo.Add("uint32_t", new PaddingInfo(4, 4));
            this.paddingInfo.Add("uint64_t", new PaddingInfo(8, 8));
            this.paddingInfo.Add("nn::util::Float2", new PaddingInfo(4, 8));
            this.paddingInfo.Add("nn::util::Float3", new PaddingInfo(4, 12));
            this.paddingInfo.Add("nn::util::Float4", new PaddingInfo(4, 16));
            this.paddingInfo.Add("nn::vfx::viewer::detail::Guid", new PaddingInfo(1, 16));
            this.paddingInfo.Add("*", new PaddingInfo(4, 4));  // ポインタ用

            // 定数値を登録
            // TODO: ランタイムデータ定義的な所でまとめて管理したい
            this.constants.Add("EFT_TEXTURE_PATTERN_NUM", 32);
            this.constants.Add("SystemParameters_MaxTexturePatternTableSize", 32);
        }

        /// <summary>
        /// バリデートを行います.
        /// </summary>
        public void Validate()
        {
            // それぞれのstructureについてバリデートを行う
            foreach (var structure in this.root.DataModels)
            {
                // structure名を取得
                string structureFullName;
                if (string.IsNullOrEmpty(structure.Namespace))
                {
                    structureFullName = structure.Name;
                }
                else
                {
                    structureFullName = string.Format("{0}::{1}", structure.Namespace, structure.Name);
                }

                if (paddingInfo.ContainsKey(structureFullName) == false)
                {
                    // structure名を指定してバリデート
                    Validate(structureFullName);
                }
            }
        }

        /// <summary>
        /// structureをバリデートします.
        /// </summary>
        /// <param name="fullName"></param>
        private void Validate(string fullName)
        {
            // fullNameで指定されたstructureをバリデートする
            foreach (var structure in this.root.DataModels)
            {
                // structure名を取得
                string structureFullName;
                if (string.IsNullOrEmpty(structure.Namespace))
                {
                    structureFullName = structure.Name;
                }
                else
                {
                    structureFullName = string.Format("{0}::{1}", structure.Namespace, structure.Name);
                }

                // あの…
                if (structureFullName != fullName)
                {
                    // すいません、人違いでした
                    continue;
                }

                // コメントの有無もついでにチェック
                if (string.IsNullOrEmpty(structure.Description))
                {
                    Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_WarningEmptyTypeDescription, structure.Name);
                }

                int maxAlignment = 0;  // 最大アライメント
                int pos = 0;           // メモリ位置

                // プロパティごとのメモリ配置をチェック
                foreach (var property in structure.Properties)
                {
                    // コメントの有無もついでにチェック
                    if (string.IsNullOrEmpty(property.Description))
                    {
                        Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_WarningEmptyMemberDescription, structure.Name, property.Name);
                    }

                    // パディングデータを取得
                    PaddingInfo info = GetPaddingInfo(property);

                    // パディングデータがなければ、structureをバリデートしてパディングデータを登録させる
                    if (info == null)
                    {
                        this.Validate(property.FullType);
                    }

                    // パディングデータを再取得
                    info = GetPaddingInfo(property);

                    // それでもなければ問題あり
                    if (info == null)
                    {
                        Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_InvalidTypeName, property.FullType);
                        continue;
                    }

                    // 要求するパディングサイズを計算
                    int padding = (info.Alignment - pos % info.Alignment) % info.Alignment;

                    // パディングが足りないときはログ出力
                    if (padding != 0)
                    {
                        Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_WarningPropertyPadding, structure.Name, property.Name, padding);
                    }

                    // 配列サイズを取得
                    int arraySize = GetArraySize(property.ArraySize);
                    if (arraySize == -1)
                    {
                        Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_InvalidArraySize, property.ArraySize);
                        arraySize = 1;
                    }

                    // このpropertyのメモリ情報を加える
                    maxAlignment = info.Alignment > maxAlignment ? info.Alignment : maxAlignment;
                    pos += info.Size * GetArraySize(property.ArraySize);
                }

                // structureのお尻に入れるパディングサイズを計算
                if (maxAlignment != 0)
                {
                    int padding = (maxAlignment - pos % maxAlignment) % maxAlignment;

                    // パディングが足りないときはメッセージを追加
                    if (padding != 0)
                    {
                        Logger.Log(LogLevels.Warning, Resources.RuntimeDataModelValidator_WarningStructurePadding, structure.Name, padding);
                    }
                }

                // パディングデータを登録
                paddingInfo[structureFullName] = new PaddingInfo(maxAlignment, pos);
                break;
            }
        }

        /// <summary>
        /// 配列サイズを取得します.
        /// </summary>
        /// <param name="arraySize">文字列で書かれた配列サイズ</param>
        /// <returns>配列サイズを返します.</returns>
        private int GetArraySize(string arraySize)
        {
            // 配列サイズ未登録のときは1
            if (string.IsNullOrEmpty(arraySize))
            {
                return 1;
            }

            int size;

            // 数字を解析
            if (int.TryParse(arraySize, out size))
            {
                return size;
            }

            // 定数値を取得
            if (constants.TryGetValue(arraySize, out size))
            {
                return size;
            }

            // エラー
            return -1;
        }

        /// <summary>
        /// パディングデータを取得します.
        /// </summary>
        /// <param name="property">プロパティデータ</param>
        /// <returns>パディングデータを返します.</returns>
        private PaddingInfo GetPaddingInfo(RuntimeDataModelPropertyDefinition property)
        {
            // ポインタ情報を返す
            if (property.FullType.EndsWith("*"))
            {
                return paddingInfo["*"];
            }

            // パディングデータを取得
            PaddingInfo info = null;
            paddingInfo.TryGetValue(property.FullType, out info);

            return info;
        }

        /// <summary>
        /// パディングデータです.
        /// </summary>
        private class PaddingInfo
        {
            /// <summary>
            /// コンストラクタです.
            /// </summary>
            public PaddingInfo()
            {
            }

            /// <summary>
            /// コンストラクタです.
            /// </summary>
            /// <param name="alignment">アライメント</param>
            /// <param name="size">サイズ</param>
            public PaddingInfo(int alignment, int size)
            {
                Alignment = alignment;
                Size = size;
            }

            /// <summary>
            /// アライメントを設定/取得します.
            /// </summary>
            public int Alignment { get; set; }

            /// <summary>
            /// サイズを設定/取得します.
            /// </summary>
            public int Size { get; set; }
        }
    }
}
