﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Properties;
using EffectMaker.DataModelMaker.Core.Writers;

namespace EffectMaker.DataModelMaker.Generators
{
    /// <summary>
    /// ランタイムデータモデルメーカーです.
    /// </summary>
    public class RuntimeDataModelGenerator : IGenerator
    {
        /// <summary>
        /// 定義データです.
        /// </summary>
        private RuntimeDataModelRootDefinition root;

        /// <summary>
        /// ファイル定義です.
        /// </summary>
        private List<RuntimeFileDefinition> fileDefinitions;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelGenerator()
        {
            this.root = new RuntimeDataModelRootDefinition();
            this.fileDefinitions = new List<RuntimeFileDefinition>();
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="definition">The definition.</param>
        public RuntimeDataModelGenerator(RuntimeDataModelRootDefinition definition)
        {
            this.root = definition;
            this.fileDefinitions = new List<RuntimeFileDefinition>();
        }

        /// <summary>
        /// Event triggered when the generation progress is advanced.
        /// </summary>
        public event EventHandler ProgressAdvanced;

        /// <summary>
        /// Evaluate total step count for Generate().
        /// </summary>
        /// <returns>The total stap count.</returns>
        public int EvaluateTotalProgressSteps()
        {
            return 1;
        }

        /// <summary>
        /// 定義ファイルをロードします.
        /// </summary>
        /// <param name="path">入力ファイルパス.</param>
        /// <returns>ロードに成功したらtrueを返却します.</returns>
        public bool LoadDefinitionFile(string path)
        {
            // ファイルパス/ファイルが存在することをチェック.
            if (Path.IsPathRooted(path) == false ||
                File.Exists(path) == false)
            {
                return false;
            }

            // 破棄処理.
            if (this.root != null)
            {
                this.root.Dispose();
            }

            // デシリアライズ.
            using (XmlReader reader = new XmlTextReader(new FileStream(path, FileMode.Open)))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(RuntimeDataModelRootDefinition));
                this.root = serializer.Deserialize(reader) as RuntimeDataModelRootDefinition;
                if (this.root == null)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 定義ファイルをセーブします.
        /// </summary>
        /// <param name="path">出力ファイルパス.</param>
        /// <returns>セーブに成功したらtrueを返却します.</returns>
        public bool SaveDefinitionFile(string path)
        {
            // シリアライズ.
            using (TextWriter writer = new StreamWriter(path))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(RuntimeDataModelRootDefinition));
                serializer.Serialize(writer, this.root);
            }

            return true;
        }

        /// <summary>
        /// C# ファイルを出力します.
        /// </summary>
        /// <param name="path">出力フォルダパス.</param>
        /// <returns>出力に成功したらtrueを返却します.</returns>
        public bool Generate(string path)
        {
            // 準備しようね.
            this.Setup();

            foreach (RuntimeFileDefinition fileDef in this.fileDefinitions)
            {
                // 小畑さん対策.
                Debug.Assert(string.IsNullOrEmpty(fileDef.FileName) == false, "ランタイム側の出力ファイル名設定しようね!!");

                // 構造体の依存関係を解決するためにソートする.
                this.SortStructures(fileDef);

                // 文字列生成.
                var code = (new RuntimeDataModelWriter()).Write(fileDef);

                // 出力ファイルパス.
                string outputPath = Path.Combine(path, fileDef.FileName);
                File.WriteAllText(outputPath, code.Replace("　", " "), Encoding.UTF8);
            }

            if (this.ProgressAdvanced != null)
            {
                this.ProgressAdvanced(this, EventArgs.Empty);
            }

            return true;
        }

        /// <summary>
        /// 定義データを設定します.
        /// </summary>
        /// <param name="definition">設定する定義データです.</param>
        /// <returns>定義データの設定に成功したらtrueを返却します.</returns>
        public bool SetDefinitionData(DefinitionBase definition)
        {
            // キャスト.
            var rootDef = definition as RuntimeDataModelRootDefinition;
            if (rootDef == null)
            {
                // 異常終了.
                return false;
            }

            // 破棄処理.
            if (this.root != null)
            {
                this.root.Dispose();
            }

            // 定義データを設定.
            this.root = rootDef;

            // 正常終了.
            return true;
        }

        /// <summary>
        /// 定義データを取得します.
        /// </summary>
        /// <returns>定義データを返却します.</returns>
        public DefinitionBase GetDefinitionData()
        {
            return this.root;
        }

        /// <summary>
        /// セットアップ.
        /// </summary>
        private void Setup()
        {
            //--------------------------------------------------------------------
            // 各ファイル定義を構築する.
            //--------------------------------------------------------------------
            foreach (RuntimeDataModelDefinition structure in this.root.DataModels)
            {
                bool isHit = false;
                for (int i = 0; i < this.fileDefinitions.Count; ++i)
                {
                    // ファイル名が一致するもの
                    if (this.fileDefinitions[i].FileName == structure.FileName)
                    {
                        // ファイル定義の構造体に追加.
                        this.fileDefinitions[i].Structures.Add(structure);
                        isHit = true;

                        // インクルードを調べる.
                        for (int j = 0; j < structure.Includes.Count; ++j)
                        {
                            bool isHitIncludes = false;

                            // インクルードファイル名が一致するものを見つける.
                            for (int k = 0; k < this.fileDefinitions[i].Includes.Count; ++k)
                            {
                                if (this.fileDefinitions[i].Includes[k].File == structure.Includes[j].File)
                                {
                                    isHitIncludes = true;
                                }
                            }

                            // インクルードの定義に追加する.
                            if (!isHitIncludes && !string.IsNullOrEmpty(structure.Includes[j].File))
                            {
                                this.fileDefinitions[i].Includes.Add(structure.Includes[j]);
                            }
                        }

                        break;
                    }
                }

                if (!isHit)
                {
                    // インクルードガードの文字列を構築.
                    string define = structure.FileName.ToUpper();
                    define = define.Replace('.', '_');
                    define = "__" + define + "__";

                    // ファイル定義を生成.
                    RuntimeFileDefinition fdef = new RuntimeFileDefinition();
                    fdef.FileName = structure.FileName;
                    fdef.Namespace = structure.Namespace;
                    fdef.Define = define;
                    fdef.Structures.Add(structure);
                    // 名前空間ごとに、RuntimeDataModelNamespaceDefinitionを生成する
                    Regex regex = new Regex("::");
                    string[] namespaceArray = regex.Split(fdef.Namespace);
                    foreach (var name in namespaceArray)
                    {
                        fdef.Namespaces.Add(new RuntimeDataModelNamespaceDefinition(){ Name = name });
                    }

                    // 各構造体のインクルード定義をループ.
                    foreach (var includeDef in structure.Includes)
                    {
                        // インクルードファイルの定義がある場合.
                        if (includeDef.File != null)
                        {
                            bool isHitFile = false;

                            // インクルードファイル名が一致するものを検索.
                            for (int k = 0; k < fdef.Includes.Count; ++k)
                            {
                                if (fdef.Includes[k].File == includeDef.File)
                                {
                                    isHitFile = true;
                                    break;
                                }
                            }

                            // 検索にヒットしなかった場合.
                            if (!isHitFile && !string.IsNullOrEmpty(includeDef.File))
                            {
                                // インクルード定義を追加.
                                fdef.Includes.Add(includeDef);
                            }
                        }
                    }

                    // ファイル定義を追加.
                    this.fileDefinitions.Add(fdef);
                }
            }

            //--------------------------------------------------------------------
            //  フォーマットのための処理.
            //--------------------------------------------------------------------
            int maxTypeStringCount   = 0;
            int maxMemberStringCount = 0;

            // データ型の最大文字数と変数名の最大文字数を算出.
            foreach (RuntimeDataModelDefinition structure in this.root.DataModels)
            {
                for (int i = 0; i < structure.Properties.Count; ++i)
                {
                    // タイプ名の最大文字数判定.
                    if (maxTypeStringCount < structure.Properties[i].FullType.Length)
                    {
                        maxTypeStringCount = structure.Properties[i].FullType.Length;
                    }

                    // メンバ名の文字数.
                    int memberCount = structure.Properties[i].Name.Length;
                    if (!string.IsNullOrEmpty(structure.Properties[i].ArraySize))
                    {
                        memberCount += 2;   // []で2個分.
                        memberCount += structure.Properties[i].ArraySize.ToString().Length; // 数字の文字数分.
                    }

                    // メンバ名の最大文字数判定.
                    if (maxMemberStringCount < memberCount)
                    {
                        maxMemberStringCount = memberCount;
                    }
                }
            }

            // 空白4文字分を加算.
            maxTypeStringCount   += 4;
            maxMemberStringCount += 4;

            // 綺麗に出力されるようにスペース数を調整する.
            foreach (RuntimeDataModelDefinition structure in this.root.DataModels)
            {
                for (int i = 0; i < structure.Properties.Count; ++i)
                {
                    // 最大文字数との差分を求める.
                    int spaceCount1 = maxTypeStringCount   - structure.Properties[i].FullType.Length;   // データ型のあとのスペース.
                    int spaceCount2 = maxMemberStringCount - structure.Properties[i].Name.Length;       // 変数名のあとのスペース.

                    if (!string.IsNullOrEmpty(structure.Properties[i].ArraySize))
                    {
                        spaceCount2 -= 2;   // []の2個分.
                        spaceCount2 -= structure.Properties[i].ArraySize.ToString().Length; // 配列数の文字数分.
                    }

                    // 既にスペース1を設定済みの場合.
                    if (!string.IsNullOrEmpty(structure.Properties[i].Space1))
                    {
                        // 設定しなおすために初期値に戻す.
                        structure.Properties[i].Space1 = string.Empty;
                    }

                    // 既にスペース2を設定済みの場合.
                    if (!string.IsNullOrEmpty(structure.Properties[i].Space2))
                    {
                        // 設定しなおすために初期値に戻す.
                        structure.Properties[i].Space2 = string.Empty;
                    }

                    // 算出した差分の数分だけスペースを入れる.
                    for (int j = 0; j < spaceCount1; ++j)
                    {
                        structure.Properties[i].Space1 += " ";
                    }

                    // 算出した差分の数分だけスペースを入れる.
                    for (int j = 0; j < spaceCount2; ++j)
                    {
                        structure.Properties[i].Space2 += " ";
                    }
                }
            }
        }

        /// <summary>
        /// 構造体の定義順をソートします.
        /// </summary>
        /// <param name="fileDef">設定する定義データです.</param>
        private void SortStructures(RuntimeFileDefinition fileDef)
        {
            var sortedList = new List<RuntimeDataModelDefinition>();
            foreach (var structure in fileDef.Structures)
            {
                // 最初の1個は無条件で追加
                if (sortedList.Count == 0)
                {
                    sortedList.Add(structure);
                    continue;
                }

                // ソート先リストを先頭からチェック
                var enumedTypes = new List<string>();
                bool inserted = false;
                foreach (var sortStruct in sortedList)
                {
                    // 利用している型名をリストアップ
                    foreach (var prop in sortStruct.Properties)
                    {
                        if (!enumedTypes.Contains(prop.TypeName))
                        {
                            enumedTypes.Add(prop.TypeName);
                        }
                    }

                    // 現在ソート済みのリストに今処理している構造体の名前があったら
                    if (enumedTypes.Contains(structure.Name))
                    {
                        // その構造体の前に挿入する
                        sortedList.Insert(sortedList.IndexOf(sortStruct), structure);
                        inserted = true;
                        break;
                    }
                }

                // 挿入されなかった場合は末尾に追加
                if (!inserted)
                {
                    sortedList.Add(structure);
                }
            }

            // 構造体リストを書き戻す
            fileDef.Structures = sortedList;
        }
    }
}
