﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModelLogic;
using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.DataModelMaker.Core.Writers;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.Generators
{
    /// <summary>
    /// バイナリ変換情報生成クラスです.
    /// このクラスでは，ソースコードを生成しやすいようにUI側が持つデータを整理します.
    /// </summary>
    public class BinaryConversionInfoGenerator : IGenerator
    {
        /// <summary>
        /// 定義データです.
        /// </summary>
        private RuntimeDataModelRootDefinition root;

        /// <summary>
        /// 生成済みオーナーのインスタンス名リスト.
        /// </summary>
        List<string> createOwnerNameList = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public BinaryConversionInfoGenerator()
        {
            this.root = null;
            this.createOwnerNameList = new List<string>();
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="definition">The definition.</param>
        public BinaryConversionInfoGenerator(RuntimeDataModelRootDefinition definition)
        {
            this.root = definition;
            this.createOwnerNameList = new List<string>();
        }

        /// <summary>
        /// Event triggered when the generation progress is advanced.
        /// </summary>
        public event EventHandler ProgressAdvanced;

        /// <summary>
        /// Evaluate total step count for Generate().
        /// </summary>
        /// <returns>The total stap count.</returns>
        public int EvaluateTotalProgressSteps()
        {
            return 1;
        }

        /// <summary>
        /// 定義ファイルをロードします.
        /// </summary>
        /// <param name="filename">ロードする定義ファイル名です.</param>
        /// <returns>ロードに成功したらtrueを返却します.</returns>
        public bool LoadDefinitionFile(string filename)
        {
            // The binary conversion info generator has no file to load.
            return false;
        }

        /// <summary>
        /// 定義ファイルをセーブします.
        /// </summary>
        /// <param name="filename">セーブする定義ファイル名です.</param>
        /// <returns>セーブに成功したらtrueを返却します.</returns>
        public bool SaveDefinitionFile(string filename)
        {
            // The binary conversion info generator has no file to save.
            return false;
        }

        /// <summary>
        /// コードを生成します.
        /// </summary>
        /// <param name="path">出力ファイルパス.</param>
        /// <returns>ソースコードの生成に成功したらtrueを返却します.</returns>
        public bool Generate(string path)
        {
            // 適切なデータ型にキャスト.
            if (this.root == null)
            {
                // 異常終了.
                return false;
            }

            // コンバート情報の定義を生成する.
            List<BinaryConversionInfoDefinition> conversionInfoList =
                this.CreateBinaryConversionInfoList(this.root.BinaryDatas);
            if (conversionInfoList == null)
            {
                // 異常終了.
                return false;
            }

            // ファイルの定義ごとにループ処理.
            foreach (BinaryConversionInfoDefinition converDef in conversionInfoList)
            {
                Debug.Assert(string.IsNullOrEmpty(converDef.FileName) == false, "バイナリコンバート情報の出力ファイル名設定しようね!");

                // ソースコードとなる文字列を生成.
                var code = (new BinaryConversionInfoWriter()).Write(converDef);

                // 出力ファイルパスを設定.
                string outputPath = Path.Combine(path, converDef.FileName + ".cs");

                // ファイルに書き出す.
                File.WriteAllText(outputPath, code, Encoding.UTF8);
            }

            if (this.ProgressAdvanced != null)
            {
                this.ProgressAdvanced(this, EventArgs.Empty);
            }

            // 正常終了.
            return true;
        }

        /// <summary>
        /// 定義データを取得します.
        /// </summary>
        /// <returns>定義データを返却します.</returns>
        public DefinitionBase GetDefinitionData()
        {
            return this.root;
        }

        /// <summary>
        /// コンバート情報を生成します.
        /// </summary>
        /// <param name="binaryDatas">コンバート定義データ.</param>
        /// <returns>コンバート情報を返却します. 生成に失敗した場合は null が返却されます.</returns>
        private List<BinaryConversionInfoDefinition> CreateBinaryConversionInfoList(IList<BinaryDataDefinition> binaryDatas)
        {
            // バイナリデータがない.
            if (binaryDatas == null)
            {
                return null;
            }

            // バイナリデータ数がゼロの場合.
            if (binaryDatas.Count == 0)
            {
                return null;
            }

            //----------------------------------------------
            // まず，同じファイル名のバイナリ定義をまとめる.
            //----------------------------------------------
            List<BinaryDataGroupDefinition> groups = new List<BinaryDataGroupDefinition>();
            foreach (BinaryDataDefinition binDef in binaryDatas)
            {
                // ファイル名が一致する定義を取得する.
                var def = (from g in groups
                           where (g.FileName == binDef.FileName) && (g.IsUserData == binDef.IsUserData)
                           select g).FirstOrDefault();

                // 見つかった場合.
                if (def != null)
                {
                    // グループに追加する.
                    def.BinaryDatas.Add(binDef);
                }
                else
                {
                    // 新しいグループを生成.
                    BinaryDataGroupDefinition newGroup = new BinaryDataGroupDefinition()
                    {
                        FileName = binDef.FileName,
                        IsUserData = binDef.IsUserData
                    };
                    newGroup.BinaryDatas.Add(binDef);

                    // グループが見つからないぼっちは，新しいグループとして追加.
                    groups.Add(newGroup);
                }
            }

            //-----------------------------------------------
            // 定義データ名を調べて，そのグループに属していないものがあれば追加する.
            //-----------------------------------------------
            foreach (BinaryDataGroupDefinition groupDef in groups)
            {
                for (int i = 0; i < groupDef.BinaryDatas.Count; ++i)
                {
                    for (int j = 0; j < groupDef.BinaryDatas[i].Fields.Count; ++j)
                    {
                        // 出力先のGUIDを取得.
                        Guid guid = groupDef.BinaryDatas[i].Fields[j].OutputFieldGuid;

                        // GUIDからバイナリデータ定義かどうかを判断.
                        BinaryDataDefinition binDef = this.GetBinaryDataDefinitionFromGuid(guid);
                        if (binDef == null)
                        {
                            // バイナリデータ定義でない場合.
                            continue;
                        }

                        // 名前が一致するものを取得する.
                        var binData = (from b in groupDef.BinaryDatas
                                       where b.Name == binDef.Name
                                       select b).FirstOrDefault();

                        // 見つからなかった場合.
                        if (binData == null)
                        {
                            // 名前が一致する定義データを取得する.
                            var definition = (from d in binaryDatas
                                              where d.Name == binDef.Name
                                              select d).FirstOrDefault();

                            // 定義がある場合.
                            if (definition != null)
                            {
                                // バイナリデータ定義を挿入する.
                                // ※ この挿入処理で，依存順番を解決している.
                                groupDef.BinaryDatas.Insert(i, definition);
                            }
                        }
                    }
                }
            }

            //-----------------------------------------------
            // ファイルコンバートしやすい形に整形する.
            //-----------------------------------------------
            List<BinaryConversionInfoDefinition> conversionInfos = new List<BinaryConversionInfoDefinition>();
            foreach (BinaryDataGroupDefinition groupDef in groups)
            {
                // リストを初期化.
                // ※ファイル毎(=グループごと)に初期化する必要あり.
                this.createOwnerNameList.Clear();

                string ownerName = "binary0";       // 適当な名前.
                string origChildName = "null";      // 子供は持ってない状態を初期状態とする.
                string childName = origChildName;

                // 親リストのメモリを確保.
                int[] parentList = new int[groupDef.BinaryDatas.Count];

                // 親リストを -1 で初期化.
                for (int i = 0; i < groupDef.BinaryDatas.Count; ++i)
                {
                    parentList[i] = -1;
                }

                // バイナリ変換情報の定義を生成.
                BinaryConversionInfoDefinition convDef = new BinaryConversionInfoDefinition()
                {
                    FileName   = groupDef.FileName,
                    Owner      = ownerName,
                    IsUserData = groupDef.IsUserData,
                };

                // 各バイナリデータごとにループ.
                for (int i = 0; i < groupDef.BinaryDatas.Count; ++i)
                {
                    BinaryDataDefinition binaryDataDef = groupDef.BinaryDatas[i];

                    // オーナー名.
                    ownerName = char.ToLowerInvariant(binaryDataDef.Name[0]) + binaryDataDef.Name.Substring(1);

                    // データモデルタイプ名を決める.
                    string dataModelTypeName = "null";
                    if (binaryDataDef.SourceDataModelInstance != null)
                    {
                        // 入力データモデルのGUIDを引っ張りだす.
                        var guid = binaryDataDef.SourceDataModelInstance.DataModelDefinitionGuid;

                        // ワークスペースマネジャーに定義を問い合わせる.
                        var dataModelDef =
                            WorkspaceManager.FindDefinition(guid) as EditorDataModelDefinition;
                        if (dataModelDef != null)
                        {
                            // 得られたクラス名を設定.
                            dataModelTypeName = "typeof(" + dataModelDef.Name + ")";
                        }
                    }

                    // インスタンス生成処理の文字列を設定.
                    string createOwner = string.Format(
                        "            var {0} = new BinaryStructDefinition({1}) {{ HasBinaryHeader = {2} }};\r\n\r\n",
                        ownerName,
                        dataModelTypeName,
                        binaryDataDef.HasBinaryHeader ? "true" : "false");

                    // 各フィールドが，親を持つかどうかを調べる.
                    for (int j = 0; j < binaryDataDef.Fields.Count; ++j)
                    {
                        BinaryFieldDefinition binaryFieldDef = binaryDataDef.Fields[j];

                        // チャイルド名を初期化.
                        origChildName = "null";
                        childName = origChildName;

                        // チャイルドのインスタン生成コードを初期化.
                        string childInstance = string.Empty;

                        // 各バイナリデータを調べる.
                        for (int k = 0; k < groupDef.BinaryDatas.Count; ++k)
                        {
                            // 名前が一致していたら，子供になっている.
                            string outputName = this.GetOutputDataNameFromGuid(binaryFieldDef.OutputFieldGuid);
                            if (outputName == groupDef.BinaryDatas[k].Name)
                            {
                                // 親のインデックス.
                                parentList[k] = i;

                                // チャイルド名.
                                origChildName = char.ToLowerInvariant(groupDef.BinaryDatas[k].Name[0]) + groupDef.BinaryDatas[k].Name.Substring(1);
                                childName = origChildName;

                                // データモデルタイプ.
                                if (binaryFieldDef.FieldType == BinaryFieldTypes.SelectBinaryData)
                                {
                                    // 入力データによってバイナリデータが変わるため，データ型は決められないのでnullにする.
                                    dataModelTypeName = "null";
                                }
                                else if (binaryFieldDef.FieldType == BinaryFieldTypes.Normal &&
                                         binaryFieldDef.InputPropertyGuidList.Count == 1)
                                {
                                    // 入力データが決まっているので，データ型を設定する.
                                    dataModelTypeName = "typeof(" + this.GetInputDataModelNameFromGuid(binaryFieldDef.InputPropertyGuidList[0]) + ")";
                                }

                                // データモデル名がnullの場合は判別できないので,
                                // 名前に数字を割り当てて別扱いできるようにする.
                                // ex. stripeUserData1 のようにする.
                                if (this.createOwnerNameList.Contains(childName) &&
                                    dataModelTypeName == "null")
                                {
                                    int count = 1;  // ここに入った時点で既に最初のやつはあるので1とする.

                                    // 想定最大数は255とする(足らなくなったら増やすこと).
                                    for (int n = 0; n < 255; ++n)
                                    {
                                        // 数字付きの名前にする.
                                        string cmpName = childName + count.ToString();

                                        // インスタンスリストに含まれているかチェック.
                                        if (!this.createOwnerNameList.Contains(cmpName))
                                        {
                                            // 含まれていなかった数字付きの名前をチャイルド名として設定する.
                                            childName = cmpName;

                                            // ループ脱出.
                                            break;
                                        }
                                    }
                                }

                                // インスタンス生成コードの文字列.
                                childInstance = string.Format(
                                     "            var {0} = new BinaryStructDefinition({1}) {{ HasBinaryHeader = {2} }};\r\n\r\n",
                                     childName,
                                     dataModelTypeName,
                                     groupDef.BinaryDatas[k].HasBinaryHeader ? "true" : "false");

                                // ループ脱出.
                                break;
                            }
                        }

                        // 変換情報定義を生成.
                        this.CreateConversionInfoDefinition(
                            ownerName,
                            origChildName,
                            childName,
                            createOwner,
                            childInstance,
                            convDef.FieldDefinitions,
                            groupDef.BinaryDatas[i].Fields[j]);
                    }
                }

                string parentName    = "binary0";   // 適当な名前(つかわれることはまずない, エラー判別用).
                string dataModelName = "object";    // 適当な名前(つかわれることはまずない, エラー判別用).

                // ルートとなるインデックスを取得.
                int parentIndex = this.GetRootIndex(parentList);
                if (parentIndex != -1)
                {
                    // ルートとなるバイナリの名前.
                    parentName = char.ToLowerInvariant(groupDef.BinaryDatas[parentIndex].Name[0]) + groupDef.BinaryDatas[parentIndex].Name.Substring(1);

                    // ルートとなる入力データモデルのGUIDを取得.
                    Guid dataModelGuid = groupDef.BinaryDatas[parentIndex].SourceDataModelInstance.DataModelDefinitionGuid;

                    // ワークスペースマネージャに問い合わせ.
                    DefinitionBase defBase = WorkspaceManager.FindDefinition(dataModelGuid);

                    // キャストして欲しいデータ型かチェックする.
                    EditorDataModelDefinition dataModel = defBase as EditorDataModelDefinition;
                    if (dataModel != null)
                    {
                        // ルートとなる入力データモデルの名前を設定.
                        dataModelName = dataModel.Name;
                    }
                }

                // ルートとなるバイナリ名とデータモデル名を設定.
                convDef.RootBinaryName    = parentName;
                convDef.RootDataModelName = dataModelName;

                // 定義を追加.
                conversionInfos.Add(convDef);
            }

            // 生成した情報を返す.
            return conversionInfos;
        }

        /// <summary>
        /// ルートとなるインデックスを取得します.
        /// </summary>
        /// <param name="parentList">親インデックリスト.</param>
        /// <returns>ルートとなるインデックスを返却します. ルートになるものが見つからない場合は -1 を返却します.</returns>
        private int GetRootIndex(int[] parentList)
        {
            // 親リストの中から，親がいないものを探し出す.
            for (int i = 0; i < parentList.Length; ++i)
            {
                if (parentList[i] == -1)
                {
                    // 発見したらインデックスを返却する.
                    return i;
                }
            }

            // 見つからなかった.
            return -1;
        }

        /// <summary>
        /// GUIDからバイナリデータ定義を取得します.
        /// </summary>
        /// <param name="outputPropertyGuid">出力先のGUID</param>
        /// <returns>検索結果を返却します.</returns>
        private BinaryDataDefinition GetBinaryDataDefinitionFromGuid(Guid outputPropertyGuid)
        {
            // ワークスペースマネージャに問い合わせ.
            DefinitionBase defBase = WorkspaceManager.FindDefinition(outputPropertyGuid);

            // キャストして返す.
            return defBase as BinaryDataDefinition;
        }

        /// <summary>
        /// 出力先となる変数名を取得します.
        /// </summary>
        /// <param name="outputPropertyGuid">出力先のプロパティGUID</param>
        /// <returns>出力先の変数名を取得します。取得に失敗した場合は string.Empty を返却します.</returns>
        private string GetOutputDataNameFromGuid(Guid outputPropertyGuid)
        {
            // ワークスペースマネージャに問い合わせ.
            DefinitionBase defBase = WorkspaceManager.FindDefinition(outputPropertyGuid);

            // バイナリ定義かチェック.
            {
                BinaryDataDefinition binDef = defBase as BinaryDataDefinition;
                if (binDef != null)
                {
                    return binDef.Name;
                }
            }

            // 構造体かチェック.
            {
                RuntimeDataModelDefinition structDef = defBase as RuntimeDataModelDefinition;
                if (structDef != null)
                {
                    return structDef.Name;
                }
            }

            // メンバー変数かチェック.
            {
                RuntimeDataModelPropertyDefinition memberDef = defBase as RuntimeDataModelPropertyDefinition;
                if (memberDef != null)
                {
                    return memberDef.Name;
                }
            }

            // 見つからなかった.
            return string.Empty;
        }

        /// <summary>
        /// 入力元になる変数名を取得します.
        /// </summary>
        /// <param name="inputPropertyGuid">入力元のプロパティGUID</param>
        /// <returns>入力元の変数名を取得します。 string.Empty を返却します.</returns>
        private string GetInputDataNameFromGuid(Guid inputPropertyGuid)
        {
            // ワークスペースマネージャに問い合わせ.
            DefinitionBase defBase = WorkspaceManager.FindDefinition(inputPropertyGuid);

            // クラスかどうかチェック.
            {
                EditorDataModelDefinition classDef = defBase as EditorDataModelDefinition;
                if (classDef != null)
                {
                    return classDef.Name;
                }
            }

            // プロパティかどうかチェック.
            {
                EditorDataModelPropertyDefinition propDef = defBase as EditorDataModelPropertyDefinition;
                if (propDef != null)
                {
                    return propDef.Name;
                }
            }

            // 見つからなかった.
            return string.Empty;
        }

        /// <summary>
        /// 入力元になるデータモデル名を取得します.
        /// </summary>
        /// <param name="inputPropertyGuid">入力元のプロパティGUID</param>
        /// <returns>入力元の変数名を取得します。 string.Empty を返却します.</returns>
        private string GetInputDataModelNameFromGuid(Guid inputPropertyGuid)
        {
            DefinitionBase defBase = WorkspaceManager.FindDefinition(inputPropertyGuid);

            // クラスかどうかチェック.
            {
                EditorDataModelDefinition classDef = defBase as EditorDataModelDefinition;
                if (classDef != null)
                {
                    return classDef.Name;
                }
            }

            // プロパティかどうかチェック.
            {
                EditorDataModelPropertyDefinition propDef = defBase as EditorDataModelPropertyDefinition;
                if (propDef != null)
                {
                    // ジェネリックの場合は<>つきで返す.
                    if (propDef.IsGeneric)
                    {
                        return propDef.Type + "<" + propDef.ElementType + ">";
                    }

                    return propDef.Type;
                }
            }

            // 再帰する.
            {
                SourceDataModelInstanceDefinition srcDMDef = defBase as SourceDataModelInstanceDefinition;
                if (srcDMDef != null)
                {
                    return this.GetInputDataModelNameFromGuid(srcDMDef.DataModelDefinitionGuid);
                }
            }

            // 再帰する.
            {
                SourcePropertyInstanceDefinition srcPropDef = defBase as SourcePropertyInstanceDefinition;
                if (srcPropDef != null)
                {
                    return this.GetInputDataModelNameFromGuid(srcPropDef.DataModelInstance.DataModelDefinitionGuid);
                }
            }

            // 見つからなかった.
            return string.Empty;
        }

        /// <summary>
        /// コンバート情報を生成します.
        /// </summary>
        /// <param name="ownerName">オーナー名.</param>
        /// <param name="origChildName">オリジナルのチャイルド名.</param>
        /// <param name="childName">チャイルド名.</param>
        /// <param name="createOwner">生成オーナー名.</param>
        /// <param name="childInstance">チャイルドのインスタンス生成コードとなる文字列です.</param>
        /// <param name="fieldInfoList">格納先のリスト.</param>
        /// <param name="fieldDef">入力となるバイナリフィールド定義データ.</param>
        private void CreateConversionInfoDefinition(
            string ownerName,
            string origChildName,
            string childName,
            string createOwner,
            string childInstance,
            List<BinaryConversionFieldInfoDefinition> fieldInfoList,
            BinaryFieldDefinition fieldDef)
        {
            // 実データが無い場合かどうか.
            bool isAllowEmptyInput =
                fieldDef.FieldType == BinaryFieldTypes.FixedSize ||
                fieldDef.FieldType == BinaryFieldTypes.ByteAlignment ||
                fieldDef.FieldType == BinaryFieldTypes.DataOffset ||
                fieldDef.FieldType == BinaryFieldTypes.DataSize;

            // 入力データがあること または，空入力OKの場合のみ処理.
            if (fieldDef.InputPropertyGuidList.Count > 0 || isAllowEmptyInput == true)
            {
                // インスタンス名を設定.
                string instanceName = createOwner;

                string converterName   = string.Empty;
                string converterParams = string.Empty;

                // コンバータを見つける.
                var converterInfo = ConverterManager.FindConverter(fieldDef.Converter);
                if (converterInfo != null)
                {
                    if (converterInfo.Namespace == "EffectMaker.DataModelLogic.BinaryConverters")
                    {
                        // The namespace is defined in the using list, no need to output the full name.
                        converterName = converterInfo.Name;
                    }
                    else
                    {
                        // The namespace is not defined in the list, output the full name.
                        converterName = converterInfo.FullName;
                    }

                    // Compose the converter parameters.
                    if (fieldDef.ConverterParameters != null &&
                        fieldDef.ConverterParameters.Count > 0)
                    {
                        var writer = new SourceCodeWriter<ConverterParameterDefinition>(
                            EffectMaker.DataModelMaker.Core.Properties.Resources.BCI_ConverterParam);

                        var builder = new StringBuilder();
                        foreach (var param in fieldDef.ConverterParameters)
                        {
                            if (converterInfo.Converter.ValidateParameter(param.Name, param.Value) == true)
                            {
                                // 文字列パラメータの値はダブルクォートで囲んで出力する
                                if (converterInfo.Converter.IsStringParameter(param.Name))
                                {
                                    if (!param.Value.StartsWith("\"") && !param.Value.EndsWith("\""))
                                    {
                                        param.Value = "\"" + param.Value + "\"";
                                    }
                                }

                                writer.EvaluateTemplateTokens(param);
                                writer.Write(builder);
                            }
                        }

                        converterParams = builder.ToString();
                    }
                }

                // 特殊なデータであるかどうかのフラグ.
                bool hasExtraData =
                    (fieldDef.FieldType != BinaryFieldTypes.Normal) &&
                    (fieldDef.FieldType != BinaryFieldTypes.SelectBinaryData);

                // タグ名を設定する.
                string binaryTag = string.Empty;
                if (string.IsNullOrEmpty(fieldDef.Tag))
                {
                    binaryTag = "null";
                }
                else
                {
                    binaryTag = "\"" + fieldDef.Tag + "\"";
                }

                // インスタンス名が登録されているかチェック.
                if (this.createOwnerNameList.Contains(ownerName))
                {
                    // 既に生成済みなので，生成させない.
                    instanceName = string.Empty;
                }

                // 構造体名を設定.
                string structName = origChildName;

                // チャイルド名がnullではなくて，数字付き名前も登録されていない場合.
                if (origChildName != "null" && !this.createOwnerNameList.Contains(childName))
                {
                    // 生成インスタンス名を差し替え.
                    instanceName += childInstance;

                    // 構造体名を数字付き名前に差し替え.
                    structName = childName;
                }

                // バイナリ変換のためのフィールド定義を生成.
                BinaryConversionFieldInfoDefinition infoDef = new BinaryConversionFieldInfoDefinition()
                {
                    Owner                = ownerName,
                    CreateOwner          = instanceName,
                    Struct               = structName,
                    Converter            = converterName,
                    ConverterParams      = converterParams,
                    HasExData            = hasExtraData,
                    ExSize               = fieldDef.FieldSize,
                    OutputName           = this.GetOutputDataNameFromGuid(fieldDef.OutputFieldGuid),
                    SendModificationType = "SendModificationTypes." + fieldDef.SendModificationType.ToString(),
                    Tag                  = binaryTag
                };

                // 引数となるGUIDの文字列.
                string guidList = string.Empty;

                // 入力プロパティ数を取得.
                int inputPropertyCount = fieldDef.InputPropertyGuidList.Count;
                if (inputPropertyCount <= 0)
                {
                    infoDef.InputName = "None";
                }
                else
                {
                    // 入力数だけループ.
                    for (int i = 0; i < inputPropertyCount; ++i)
                    {
                        // 入力データのGUIDから定義を引っ張り出す.
                        DefinitionBase definitionBase =
                            WorkspaceManager.FindDefinition(fieldDef.InputPropertyGuidList[i]);

                        // 入力プロパティであるかチェック.
                        SourcePropertyInstanceDefinition srcProp =
                            definitionBase as SourcePropertyInstanceDefinition;
                        if (srcProp == null)
                        {
                            continue;
                        }

                        string path = this.ComposeInputPropertyPath(srcProp);
                        if (string.IsNullOrEmpty(path) == true)
                        {
                            Logger.Log(LogLevels.Warning, "BinaryConversionInfoGenerator.CreateConversionInfoDefinition : Failed composing input property path.");
                            continue;
                        }

                        // 引数となるGUIDのリストを生成.
                        guidList += string.Format(
                            "new InputPropertyDefinition(\"{0}\", new Guid(\"{1}\"))",
                            path,
                            srcProp.PropertyDefinitionGuid);

                        // 入力プロパティ名を取得.
                        infoDef.InputName += this.GetInputDataNameFromGuid(srcProp.PropertyDefinitionGuid);

                        // 改行コードと，整形のためのスペースを入れる.
                        if (i < inputPropertyCount - 1)
                        {
                            guidList += ",\r\n                ";
                            infoDef.InputName += ", ";
                        }
                    }

                    // 特殊形式の場合.
                    if (infoDef.HasExData == true &&
                        string.IsNullOrEmpty(guidList) == false)
                    {
                        guidList = ",\r\n                " + guidList;
                    }
                }

                // GUIDの引数リストを設定.
                infoDef.GuidList = guidList;

                // メソッド名を変更.
                switch (fieldDef.FieldType)
                {
                    case BinaryFieldTypes.Normal:
                        break;

                    case BinaryFieldTypes.FixedSize:
                        infoDef.Method = "AddFixedSizeFieldDefinition";
                        break;

                    case BinaryFieldTypes.ByteAlignment:
                        infoDef.Method = "AddAlignmentFieldDefinition";
                        break;

                    case BinaryFieldTypes.DataOffset:
                        infoDef.Method = "AddDataOffsetFieldDefinition";
                        break;

                    case BinaryFieldTypes.DataSize:
                        infoDef.Method = "AddDataSizeFieldDefinition";
                        break;

                    case BinaryFieldTypes.SelectBinaryData:
                        infoDef.Method = "AddSelectBinaryDataFieldDefinition";
                        break;
                }

                // フィールド定義を追加.
                fieldInfoList.Add(infoDef);

                if (!this.createOwnerNameList.Contains(ownerName))
                {
                    // 生成済みリストに登録.
                    this.createOwnerNameList.Add(ownerName);
                }

                // チャイルド名がnullではなくて，登録もされていない場合.
                if (structName != "null" && !this.createOwnerNameList.Contains(structName))
                {
                    this.createOwnerNameList.Add(structName);
                }
            }
            else if (fieldDef.GroupDefinition != null)
            {
                for (int i = 0; i < fieldDef.GroupDefinition.Fields.Count; ++i)
                {
                    // 再帰呼び出し.
                    this.CreateConversionInfoDefinition(
                        ownerName,
                        origChildName,
                        childName,
                        createOwner,
                        childInstance,
                        fieldInfoList,
                        fieldDef.GroupDefinition.Fields[i]);
                }
            }
        }

        /// <summary>
        /// Compose input property path.
        /// </summary>
        /// <param name="srcPropertyDef">The source property definition.</param>
        /// <returns>The composed path.</returns>
        private string ComposeInputPropertyPath(SourcePropertyInstanceDefinition srcPropertyDef)
        {
            string path = string.Empty;

            DefinitionBase def = srcPropertyDef;
            while (def != null)
            {
                if (def is SourcePropertyInstanceDefinition)
                {
                    var currSrcPropertyDef = (SourcePropertyInstanceDefinition)def;

                    // Get the editor data model property definition.
                    var propertyDef =
                        WorkspaceManager.FindDefinition(currSrcPropertyDef.PropertyDefinitionGuid) as EditorDataModelPropertyDefinition;
                    if (propertyDef == null)
                    {
                        return null;
                    }

                    // Add the property name to the beginning of the path.
                    path =
                        propertyDef.Name +
                        (string.IsNullOrEmpty(path) ? string.Empty : ".") +
                        path;
                }

                def = def.Parent;
            }

            return path;
        }
    }
}
