﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.Collections;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.DataModelMaker.Core.Definitions
{
    /// <summary>
    /// データモデルのルートを定義するクラスです.
    /// </summary>
    [XmlType("DataModelDefinition")]
    public class EditorDataModelRootDefinition : DefinitionBase
    {
        /// <summary>The index to the editing version, -1 represents the latest version.</summary>
        private int editingVersionIndex = -1;

        /// <summary>The editor data model definitions.</summary>
        private DefinitionList<EditorDataModelDefinition> dataModelDefs = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public EditorDataModelRootDefinition()
        {
            this.dataModelDefs = new DefinitionList<EditorDataModelDefinition>(this);

            // Add the default version.
            this.Versions = new DefinitionList<EditorVersionDefinition>(this);
            this.Versions.Add(new EditorVersionDefinition());

            this.IsVersionEnabled = true;
        }

        /// <summary>
        /// Get or set the flag indicating whether versioning is enabled.
        /// When enabled, the data models will be exported to either
        /// "Versioned" or "NonVersioned" folders depending on the same
        /// flag in each data model definition.
        /// </summary>
        public bool IsVersionEnabled { get; set; }

        /// <summary>
        /// Get the editing version definition.
        /// </summary>
        [XmlIgnore]
        public int EditingVersionIndex
        {
            get
            {
                if (this.editingVersionIndex < 0)
                {
                    return this.Versions.Count - 1;
                }
                else
                {
                    return this.editingVersionIndex;
                }
            }

            set
            {
                if (value < 0 || value >= this.Versions.Count)
                {
                    this.editingVersionIndex = -1;
                }
                else
                {
                    this.editingVersionIndex = value;
                }
            }
        }

        /// <summary>
        /// Get the editing version definition.
        /// </summary>
        [XmlIgnore]
        public EditorVersionDefinition EditingVersion
        {
            get
            {
                if (this.editingVersionIndex < 0)
                {
                    return this.LatestVersion;
                }
                else
                {
                    return this.Versions[this.editingVersionIndex];
                }
            }

            set
            {
                int index = this.Versions.IndexOf(value);
                if (index < 0 || index >= this.Versions.Count)
                {
                    this.editingVersionIndex = -1;
                }
                else
                {
                    this.editingVersionIndex = index;
                }
            }
        }

        /// <summary>
        /// Get the previous version definition of the editing one.
        /// </summary>
        [XmlIgnore]
        public EditorVersionDefinition PreviousVersion
        {
            get
            {
                int index = this.editingVersionIndex;
                if (index < 0)
                {
                    index = this.LatestVersionIndex;
                }

                --index;
                if (index < 0 || index >= this.Versions.Count)
                {
                    return null;
                }
                else
                {
                    return this.Versions[index];
                }
            }
        }

        /// <summary>
        /// Get the next version definition of the editing one.
        /// </summary>
        [XmlIgnore]
        public EditorVersionDefinition NextVersion
        {
            get
            {
                int index = this.editingVersionIndex;
                if (index < 0 || index >= this.Versions.Count - 1)
                {
                    return null;
                }

                return this.Versions[index + 1];
            }
        }

        /// <summary>
        /// Get the latest version.
        /// </summary>
        [XmlIgnore]
        public EditorVersionDefinition LatestVersion
        {
            get { return this.Versions[this.Versions.Count - 1]; }
        }

        /// <summary>
        /// Get the index of the latest version.
        /// </summary>
        [XmlIgnore]
        public int LatestVersionIndex
        {
            get { return this.Versions.Count - 1; }
        }

        /// <summary>
        /// Get or set the editor versions.
        /// </summary>
        [XmlIgnore]
        public DefinitionList<EditorVersionDefinition> Versions { get; set; }

        /// <summary>
        /// Get or set the editor versions for serialization.
        /// </summary>
        [XmlArray("Versions")]
        [XmlArrayItem("Version")]
        public EditorVersionDefinition[] SerializationVersions
        {
            get
            {
                return this.Versions.ToArray();
            }

            set
            {
                this.DisposeChildren(this.Versions);
                this.Versions.Clear();
                if (value != null && value.Length > 0)
                {
                    this.Versions.AddRange(value);
                }
            }
        }

        /// <summary>
        /// データモデルの定義です.
        /// </summary>
        [XmlElement("DataModel")]
        public DefinitionList<EditorDataModelDefinition> DataModels
        {
            get { return this.dataModelDefs; }
            set { this.SetChildDefinitionList(ref this.dataModelDefs, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChildren(this.DataModels);
            this.DisposeChildren(this.Versions);
            base.Dispose();
        }

        /// <summary>
        /// 使用準備を行います.
        /// </summary>
        public void Setup()
        {
            this.SortNameSpace();
        }

        /// <summary>
        /// Get the version by its index.
        /// </summary>
        /// <param name="index">The index to the version.</param>
        /// <returns>The version.</returns>
        public VersionXml GetVersionByIndex(int index)
        {
            if (index < 0 || index >= this.Versions.Count)
            {
                return null;
            }

            return this.Versions[index].Version;
        }

        /// <summary>
        /// Find the index to the specified version.
        /// </summary>
        /// <param name="version">The version to find.</param>
        /// <returns>The version index.</returns>
        public int FindVersionIndex(Version version)
        {
            for (int i = 0; i < this.Versions.Count; ++i)
            {
                if (version == this.Versions[i].Version)
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// 短いタイプ名と名前空間名を設定します.
        /// </summary>
        private void SortNameSpace()
        {
            for (int i = 0; i < this.DataModels.Count; ++i)
            {
                this.DataModels[i].SortNameSpace();
            }
        }
    }

    /// <summary>
    /// Editor version definition.
    /// </summary>
    [Serializable]
    public class EditorVersionDefinition : DefinitionBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorVersionDefinition()
        {
            this.Version = new Version(0, 0, 0, 0);
        }

        /// <summary>
        /// Get or set the version.
        /// </summary>
        [XmlElement]
        public VersionXml Version { get; set; }
    }

    /// <summary>
    /// Class that holds data model definitions.
    /// </summary>
    [XmlType("DataModel")]
    public class EditorDataModelDefinition : DefinitionBase, INameDescriptionObject
    {
        /// <summary>String constant for formatting the name space version.</summary>
        private const string NamespaceVersionFormatString = ".Version_{0}_{1}_{2}_{3}";

        /// <summary>The editor data model property definitions.</summary>
        private DefinitionList<EditorDataModelPropertyDefinition> properties = null;

        /// <summary>The editor data model namespace definitions.</summary>
        private DefinitionList<EditorDataModelUsingNamespaceDefinition> namespaces = null;

        /// <summary>The name of the data model.</summary>
        private string name = string.Empty;

        /// <summary>The namespace of the data model.</summary>
        private string nameSpace = string.Empty;

        /// <summary>The export folder for the data model source file.</summary>
        private string exportFolder = string.Empty;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public EditorDataModelDefinition()
        {
            this.IsVersionEnabled = true;
            this.CreatedVersionIndex = -1;
            this.DeletedVersionIndex = -1;
            this.properties = new DefinitionList<EditorDataModelPropertyDefinition>(this);
            this.namespaces = new DefinitionList<EditorDataModelUsingNamespaceDefinition>(this);
            this.SuperClasses = new List<Guid>();
            this.Export = true;
        }

        /// <summary>
        /// Get the editing version index.
        /// </summary>
        [XmlIgnore]
        public int EditingVersionIndex
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                return root.EditingVersionIndex;
            }
        }

        /// <summary>
        /// Get the editing version.
        /// </summary>
        [XmlIgnore]
        public Version EditingVersion
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                return root.EditingVersion.Version;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether versioning is enabled for the data model.
        /// </summary>
        [XmlAttribute]
        public bool IsVersionEnabled { get; set; }

        /// <summary>
        /// Get or set the the created version index of the data model.
        /// The version index points to the versions in the data model root definition.
        /// </summary>
        [XmlAttribute]
        public int CreatedVersionIndex { get; set; }

        /// <summary>
        /// Get or set the deleted version index of the data model.
        /// If the data model is not deleted, this version remains -1.
        /// The version index points to the versions in the data model root definition.
        /// </summary>
        [XmlAttribute]
        public int DeletedVersionIndex { get; set; }

        /// <summary>
        /// Get the the created version of the data model.
        /// </summary>
        [XmlIgnore]
        public VersionXml CreatedVersion
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                if (root == null)
                {
                    return null;
                }

                return root.GetVersionByIndex(this.CreatedVersionIndex);
            }
        }

        /// <summary>
        /// Get the the deleted version of the data model.
        /// If the data model is not deleted, this version remains null.
        /// </summary>
        [XmlIgnore]
        public VersionXml DeletedVersion
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                if (root == null)
                {
                    return null;
                }

                return root.GetVersionByIndex(this.DeletedVersionIndex);
            }
        }

        /// <summary>
        /// 名前を設定/取得します.
        /// </summary>
        [XmlAttribute("Name")]
        public string Name
        {
            get
            {
                return this.name;
            }

            set
            {
                // First remove the registered type.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.RemoveEditorType(this.nameSpace, this.name);
                }

                this.name = value;

                // Now register the data model with the new name.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.AddEditorDataModel(this);
                }
            }
        }

        /// <summary>Get or set name spaces the data model class uses.</summary>
        [XmlElement("UsingNamespace")]
        public DefinitionList<EditorDataModelUsingNamespaceDefinition> UsingNamespaces
        {
            get { return this.namespaces; }
            set { this.SetChildDefinitionList(ref this.namespaces, value); }
        }

        /// <summary>Get or set the name space of the data model class.</summary>
        [XmlElement("Namespace")]
        public string Namespace
        {
            get
            {
                return this.nameSpace;
            }

            set
            {
                // First remove the registered type.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.RemoveEditorType(this.nameSpace, this.name);
                }

                this.nameSpace = value;

                // Now register the data model with the new name.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.AddEditorDataModel(this);
                }
            }
        }

        /// <summary>
        /// Get the custom attributes for the data model class.
        /// </summary>
        public string Attributes
        {
            get
            {
                return string.Empty;
            }
        }

        /// <summary>Get the name space alias definitions.</summary>
        [XmlIgnore]
        public string NamespaceAliases
        {
            get
            {
                string aliases = string.Empty;
                var root = this.FindParent<EditorDataModelRootDefinition>();

                bool isLastVersion =
                    root.EditingVersionIndex >= root.LatestVersionIndex ||
                    (this.DeletedVersionIndex >= 0 &&
                     this.DeletedVersionIndex <= root.EditingVersionIndex);

                // The previous version namespace alias.
                if (root != null &&
                    this.IsVersionEnabled == true &&
                    root.EditingVersionIndex > this.CreatedVersionIndex)
                {
                    string versionStr =
                        this.FormatNamespaceVersionString(root.PreviousVersion.Version);

                    aliases += string.Format(
                        EffectMaker.DataModelMaker.Core.Properties.Resources.EditorDataModelTemplate_PrevVersionNamespaceAlias,
                        this.nameSpace + versionStr);
                }

                // The current version namespace alias.
                if (root != null)
                {
                    string versionStr = string.Empty;
                    if (this.IsVersionEnabled == true && isLastVersion == false)
                    {
                        versionStr = this.FormatNamespaceVersionString(root.EditingVersion.Version);
                    }

                    aliases += string.Format(
                        EffectMaker.DataModelMaker.Core.Properties.Resources.EditorDataModelTemplate_CurrVersionNamespaceAlias,
                        this.nameSpace + versionStr);

                    aliases += string.Format(
                        EffectMaker.DataModelMaker.Core.Properties.Resources.EditorDataModelTemplate_CurrVersionSerializerNamespaceAlias,
                        versionStr);
                }

                return aliases;
            }
        }

        /// <summary>Get or set the namespace for the editing version.</summary>
        [XmlIgnore]
        public string CurrVersionNamespace
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                string versionStr =
                    this.FormatNamespaceVersionString(root.EditingVersion.Version);

                return this.nameSpace + versionStr;
            }
        }

        /// <summary>Get or set the description of the class.</summary>
        [XmlElement("Description")]
        public string Description { get; set; }

        /// <summary>
        /// Get or set all the properties of the data model, including deleted properties.
        /// </summary>
        [XmlElement("Property")]
        public DefinitionList<EditorDataModelPropertyDefinition> AllProperties
        {
            get { return this.properties; }
            set { this.SetChildDefinitionList(ref this.properties, value); }
        }

        /// <summary>
        /// Enumerate the properties that is available for the editing version.
        /// </summary>
        [XmlIgnore]
        public IEnumerable<EditorDataModelPropertyDefinition> Properties
        {
            get
            {
                // Find the properties that presents in the editing version.
                return from p in this.properties
                       where p.EditingVersionDefinition != null &&
                             p.EditingVersionDefinition.Action != VersionActions.Delete
                       select p;
            }
        }

        /// <summary>
        /// Get or set the super classes.
        /// </summary>
        [XmlElement("SuperClass")]
        public List<Guid> SuperClasses { get; set; }

        /// <summary>
        /// ソースコードを書き出すかどうかのフラグを設定/取得します.
        /// </summary>
        [XmlElement("Export")]
        public bool Export { get; set; }

        /// <summary>
        /// 出力先の相対ファイルパスを設定/取得します.
        /// </summary>
        [XmlElement("ExportFolder")]
        public string ExportFolder
        {
            get
            {
                return this.exportFolder;
            }

            set
            {
                this.exportFolder = value;
                PathManager.AddEditorDataModelExportFolder(this.exportFolder);
            }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            TypeManager.RemoveEditorType(this.nameSpace, this.name);

            this.DisposeChildren(this.UsingNamespaces);
            this.DisposeChildren(this.Properties);
            base.Dispose();
        }

        /// <summary>
        /// Create a new property definition for the editing version.
        /// </summary>
        /// <returns>The created property definition.</returns>
        public EditorDataModelPropertyDefinition CreatePropertyForEditingVersion()
        {
            // Find the root definition to get the editing version.
            var rootDef = this.FindParent<EditorDataModelRootDefinition>();

            // Create the property definition and add it to our property list.
            var property = new EditorDataModelPropertyDefinition(rootDef.EditingVersionIndex);
            this.properties.Add(property);

            return property;
        }

        /// <summary>
        /// Remove the given property definition from the editing version.
        /// </summary>
        /// <param name="property">The property definition to remove.</param>
        public void RemovePropertyFromEditingVersion(EditorDataModelPropertyDefinition property)
        {
            int index = this.properties.IndexOf(property);
            if (index < 0 || index >= this.properties.Count)
            {
                return;
            }

            // Find the root definition to get the editing version.
            var rootDef = this.FindParent<EditorDataModelRootDefinition>();

            // The version to delete is also the created version,
            // just remove the whole property and return.
            if (property.Versions.Count == 1 &&
                property.Versions[0].VersionIndex == rootDef.EditingVersionIndex)
            {
                this.properties.RemoveAt(index);
                return;
            }

            // First remove all the versions after the editing version.
            // (also remove the editing version, will be added back later)
            for (int i = property.Versions.Count - 1; i >= 0; --i)
            {
                if (property.Versions[i].VersionIndex >= rootDef.EditingVersionIndex)
                {
                    property.Versions.RemoveAt(i);
                }
            }

            // Add the editing version, and assign the action as "delete."
            property.Versions.Add(new EditorPropertyVersionDefinition()
                {
                    VersionIndex = rootDef.EditingVersionIndex,
                    Action = VersionActions.Delete,
                    PropertyDefinition = null,
                });
        }

        /// <summary>
        /// プロパティに定義されている名前空間と短いタイプ名を設定します.
        /// </summary>
        public void SortNameSpace()
        {
            List<EditorDataModelUsingNamespaceDefinition> namespaces = new List<EditorDataModelUsingNamespaceDefinition>();

            foreach (EditorDataModelUsingNamespaceDefinition namespaceDef in this.UsingNamespaces)
            {
                // 重複チェック.
                bool isFind = false;
                string usingName = namespaceDef.Name;
                for (int i = 0; i < namespaces.Count; ++i)
                {
                    // 名前が同じかどうか調べる.
                    if (namespaces[i].Name == usingName)
                    {
                        isFind = true;
                        break;
                    }
                }

                // 重複チェックに引っかからなかったら追加.
                if (!isFind && !string.IsNullOrEmpty(usingName))
                {
                    namespaceDef.Name = usingName;
                    namespaces.Add(namespaceDef);
                }
            }

            foreach (EditorDataModelPropertyDefinition propDef in this.Properties)
            {
                bool isFind = false;
                string usingName = propDef.Namespace;
                for (int i = 0; i < namespaces.Count; ++i)
                {
                    if (namespaces[i].Name == usingName)
                    {
                        isFind = true;
                        break;
                    }
                }

                // 重複チェックに引っかからなかったら追加.
                if (!isFind && !string.IsNullOrEmpty(usingName))
                {
                    EditorDataModelUsingNamespaceDefinition namespaceDef = new EditorDataModelUsingNamespaceDefinition();
                    namespaceDef.Name = usingName;
                    namespaces.Add(namespaceDef);
                }
            }

            // ソート.
            var sortList = (from ns in namespaces
                            orderby ns.Name
                            select ns).Distinct();

            this.UsingNamespaces.Clear();
            foreach (var ns in sortList)
            {
                this.UsingNamespaces.Add(ns);
            }
        }

        /// <summary>
        /// Format version string for the namespace.
        /// </summary>
        /// <param name="version">The version.</param>
        /// <returns>The formatted string.</returns>
        private string FormatNamespaceVersionString(Version version)
        {
            var root = this.FindParent<EditorDataModelRootDefinition>();
            if (root != null &&
                version < root.LatestVersion.Version)
            {
                return string.Format(
                    NamespaceVersionFormatString,
                    version.Major,
                    version.Minor,
                    version.Build,
                    version.Revision);
            }

            return string.Empty;
        }
    }

    /// <summary>
    /// Class that holds using statements definitions.
    /// </summary>
    [XmlType("UsingNamespace")]
    public class EditorDataModelUsingNamespaceDefinition : DefinitionBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorDataModelUsingNamespaceDefinition()
        {
            this.Name = string.Empty;
        }

        /// <summary>Get or set the name of the property.</summary>
        [XmlAttribute("Name")]
        public string Name { get; set; }
    }

    /// <summary>
    /// The definition data for editor data model properties.
    /// </summary>
    [XmlType("PropertyDef")]
    public class PropertyDefinitionData : ICloneable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public PropertyDefinitionData()
        {
            this.Name = string.Empty;
            this.Namespace = string.Empty;
            this.Type = "object";
            this.ElementNamespace = string.Empty;
            this.ElementType = string.Empty;
            this.Description = string.Empty;
            this.DefaultValue = string.Empty;
            this.XmlIgnore = false;
            this.CustomSetterEnabled = false;
            this.CustomSetter = null;
            this.CustomGetterEnabled = false;
            this.CustomGetter = null;
            this.EnableFieldDeclaration = false;
            this.EnableNullOriginal = false;
            this.VersionConverter = null;
            this.AlwaysApplyVersionConverter = false;
        }

        /// <summary>Get or set the name of the property.</summary>
        [XmlAttribute]
        public string Name { get; set; }

        /// <summary>Get or set the namespace of the type of the property.</summary>
        [XmlAttribute]
        [DefaultValue("")]
        public string Namespace { get; set; }

        /// <summary>Get or set the name of the type of the property.</summary>
        [XmlAttribute]
        public string Type { get; set; }

        /// <summary>Get or set the namespace of the element type of the property.</summary>
        [XmlAttribute]
        [DefaultValue("")]
        public string ElementNamespace { get; set; }

        /// <summary>Get or set the element type of the property.</summary>
        [XmlAttribute]
        [DefaultValue("")]
        public string ElementType { get; set; }

        /// <summary>Get or set the description of the property.</summary>
        [XmlElement]
        public string Description { get; set; }

        /// <summary>Get or set string representation of the default value of the property.</summary>
        [XmlElement]
        [DefaultValue("")]
        public string DefaultValue { get; set; }

        /// <summary>The flag indicating whether to ignore this property for XML serialization.</summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool XmlIgnore { get; set; }

        /// <summary>Get or set the flag indicating whether to enable custom setter or not.</summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool CustomSetterEnabled { get; set; }

        /// <summary>Get or set the user defined custom setter for the property.</summary>
        [XmlElement]
        [DefaultValue("")]
        public string CustomSetter { get; set; }

        /// <summary>Get or set the flag indicating whether to enable custom getter or not.</summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool CustomGetterEnabled { get; set; }

        /// <summary>Get or set the user defined custom getter for the property.</summary>
        [XmlElement]
        [DefaultValue("")]
        public string CustomGetter { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to define backing field
        /// for the property when it has custom getter or setter.
        /// </summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool EnableFieldDeclaration { get; set; }

        /// <summary>
        /// ReadXml時に規定値をnullとして扱うか否かを取得または設定します。
        /// ユーザーデータのインスタンスを直接保持しているプロパティ用の設定です。
        /// </summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool EnableNullOriginal { get; set; }

        /// <summary>Get or set the version converter for the property.</summary>
        [XmlElement]
        [DefaultValue("")]
        public string VersionConverter { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to always apply the version converter.
        /// The version converter only generated to the source code at this version if
        /// this flag is false, otherwise, the version converter will be applied to all
        /// the versions after this version.
        /// </summary>
        [XmlAttribute]
        [DefaultValue(false)]
        public bool AlwaysApplyVersionConverter { get; set; }

        /// <summary>
        /// Clone the instance.
        /// </summary>
        /// <returns>The new instance.</returns>
        public object Clone()
        {
            var newItem = new PropertyDefinitionData();

            newItem.Name = this.Name;
            newItem.Namespace = this.Namespace;
            newItem.Type = this.Type;
            newItem.ElementNamespace = this.ElementNamespace;
            newItem.ElementType = this.ElementType;
            newItem.Description = this.Description;
            newItem.DefaultValue = this.DefaultValue;
            newItem.XmlIgnore = this.XmlIgnore;
            newItem.CustomSetterEnabled = this.CustomSetterEnabled;
            newItem.CustomSetter = this.CustomSetter;
            newItem.CustomGetterEnabled = this.CustomGetterEnabled;
            newItem.CustomGetter = this.CustomGetter;
            newItem.EnableFieldDeclaration = this.EnableFieldDeclaration;
            newItem.EnableNullOriginal = this.EnableNullOriginal;
            newItem.VersionConverter = this.VersionConverter;
            newItem.AlwaysApplyVersionConverter = this.AlwaysApplyVersionConverter;

            return newItem;
        }
    }

    /// <summary>
    /// Definition of the differences of an editor data model property
    /// from an older version to the next.
    /// </summary>
    [XmlType("PropertyVersion")]
    public class EditorPropertyVersionDefinition : DefinitionBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorPropertyVersionDefinition()
        {
            this.VersionIndex = -1;
            this.Action = VersionActions.Create;
            this.PropertyDefinition = null;
        }

        /// <summary>
        /// Get or set the version.
        /// </summary>
        [XmlAttribute]
        public int VersionIndex { get; set; }

        /// <summary>
        /// Get or set the version.
        /// </summary>
        [XmlIgnore]
        public VersionXml Version
        {
            get
            {
                var root = this.FindParent<EditorDataModelRootDefinition>();
                if (root == null)
                {
                    return null;
                }

                return root.GetVersionByIndex(this.VersionIndex);
            }
        }

        /// <summary>
        /// Get or set the action that is applied to the property at the version.
        /// E.g. created, modified or deleted.
        /// </summary>
        [XmlAttribute]
        public VersionActions Action { get; set; }

        /// <summary>
        /// Get or set the property definition for the version.
        /// </summary>
        public PropertyDefinitionData PropertyDefinition { get; set; }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.PropertyDefinition = null;
            base.Dispose();
        }
    }

    /// <summary>
    /// Class that holds data model property definitions.
    /// </summary>
    [XmlType("Property")]
    public class EditorDataModelPropertyDefinition : DefinitionBase,
                                                     INameDescriptionObject,
                                                     IEditingVersionLockable
    {
        /// <summary>The editor property version definitions.</summary>
        private DefinitionList<EditorPropertyVersionDefinition> versionDefs = null;

        /// <summary>
        /// The latest edited version index.
        /// This version is set according to the editing version
        /// of the EditorDataModelRootDefinition, and could be
        /// different from the editing version of this property.
        /// E.g.
        /// The editing version is 1.5.0.0 in EditorDataModelRootDefinition,
        /// but the property is created at 1.3.0.0 and hasn't modified since.
        /// Hence, 1.5.0.0 does not exist in the property's version list.
        /// It will take the definition from 1.3.0.0, which is the latest
        /// version definition before the version set in EditorDataModelRootDefinition.
        /// </summary>
        private int latestEditedVersionIndex = -1;

        /// <summary>The editing property version definition.</summary>
        private EditorPropertyVersionDefinition editingVersionDef = null;

        /// <summary>The editing property definition data.</summary>
        private PropertyDefinitionData editingPropertyDef = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorDataModelPropertyDefinition()
        {
            this.versionDefs = new DefinitionList<EditorPropertyVersionDefinition>(this);

            // Create a default version.
            var defaultVersion = new EditorPropertyVersionDefinition();
            defaultVersion.PropertyDefinition = new PropertyDefinitionData();

            this.versionDefs.Add(defaultVersion);
        }

        /// <summary>
        /// Constructor with the created version of this property.
        /// </summary>
        /// <param name="createdVersionIndex">The created version index.</param>
        public EditorDataModelPropertyDefinition(int createdVersionIndex) :
            this()
        {
            this.versionDefs = new DefinitionList<EditorPropertyVersionDefinition>(this);

            this.versionDefs.Add(new EditorPropertyVersionDefinition()
            {
                VersionIndex = createdVersionIndex,
                Action = VersionActions.Create,
                PropertyDefinition = new PropertyDefinitionData(),
            });
        }

        /// <summary>
        /// Get the name of the owner data model.
        /// </summary>
        [XmlIgnore]
        public string DataModelName
        {
            get
            {
                var dataModelDef = this.FindParent<EditorDataModelDefinition>();
                if (dataModelDef == null)
                {
                    return string.Empty;
                }

                return dataModelDef.Name;
            }
        }

        /// <summary>
        /// Get or set the definition of versions of the property.
        /// </summary>
        [XmlIgnore]
        public EditingVersionLock EditingVersionLock { get; set; }

        /// <summary>
        /// Get or set the definition of versions of the property.
        /// </summary>
        [XmlIgnore]
        public DefinitionList<EditorPropertyVersionDefinition> Versions
        {
            get { return this.versionDefs; }
            set { this.SetChildDefinitionList(ref this.versionDefs, value); }
        }

        /// <summary>
        /// Get the created version definition of the property.
        /// </summary>
        public EditorPropertyVersionDefinition CreatedVersion
        {
            get
            {
                if (this.versionDefs != null && this.versionDefs.Count > 0)
                {
                    return this.versionDefs[0];
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// Get or set the array of version info for serialization.
        /// </summary>
        [XmlArray("Versions")]
        [XmlArrayItem("Version")]
        public EditorPropertyVersionDefinition[] SerializationVersions
        {
            get
            {
                return this.Versions.ToArray();
            }

            set
            {
                this.DisposeChildren(this.Versions);
                this.Versions.Clear();
                if (value != null && value.Length > 0)
                {
                    this.Versions.AddRange(value);
                }
            }
        }

        /// <summary>
        /// Get the flag indicating whether the property
        /// has history entry for the editing version.
        /// </summary>
        [XmlIgnore]
        public bool HasEditingVersion
        {
            get
            {
                var rootDef = this.FindParent<EditorDataModelRootDefinition>();
                return this.editingVersionDef.Version == rootDef.EditingVersion.Version;
            }
        }

        /// <summary>
        /// Get the editing property version.
        /// </summary>
        [XmlIgnore]
        public EditorPropertyVersionDefinition EditingVersionDefinition
        {
            get
            {
                this.UpdateEditingVersion();
                return this.editingVersionDef;
            }
        }

        /// <summary>
        /// Get the property definition of the editing version.
        /// </summary>
        [XmlIgnore]
        public PropertyDefinitionData EditingPropertyDefinition
        {
            get
            {
                this.UpdateEditingVersion();
                return this.editingPropertyDef;
            }
        }

        /// <summary>Get or set the name of the property.</summary>
        [XmlIgnore]
        public string Name
        {
            get { return this.EditingPropertyDefinition.Name; }
            set { this.EditingPropertyDefinition.Name = value; }
        }

        /// <summary>
        /// The full type name without namespace.
        /// </summary>
        [XmlIgnore]
        public string FullTypeWithoutNamespace
        {
            get
            {
                // Find the information of the property value type.
                EditorTypeInfo primaryInfo =
                    TypeManager.FindEditorType(this.Namespace, this.Type);
                if (primaryInfo == null)
                {
                    return this.Type;
                }

                // Compose the complete type string.
                string typeString = primaryInfo.TypeName;
                if (primaryInfo.IsGeneric == true)
                {
                    EditorTypeInfo elementInfo =
                       TypeManager.FindEditorType(this.ElementNamespace, this.ElementType);
                    if (elementInfo != null)
                    {
                        typeString += "<" + elementInfo.TypeName + ">";
                    }
                    else
                    {
                        typeString += "<" + this.ElementType + ">";
                    }
                }

                return typeString;
            }
        }

        /// <summary>
        /// Get or set the namespace of the type of the property.
        /// </summary>
        [XmlIgnore]
        public string Namespace
        {
            get { return this.EditingPropertyDefinition.Namespace; }
            set { this.EditingPropertyDefinition.Namespace = value; }
        }

        /// <summary>
        /// Get or set the name of the type of the property.
        /// </summary>
        [XmlIgnore]
        public string Type
        {
            get { return this.EditingPropertyDefinition.Type; }
            set { this.EditingPropertyDefinition.Type = value; }
        }

        /// <summary>
        /// Get or set the namespace of the element type of the property.
        /// </summary>
        [XmlIgnore]
        public string ElementNamespace
        {
            get { return this.EditingPropertyDefinition.ElementNamespace; }
            set { this.EditingPropertyDefinition.ElementNamespace = value; }
        }

        /// <summary>
        /// Get or set the element type of the property.
        /// </summary>
        [XmlIgnore]
        public string ElementType
        {
            get { return this.EditingPropertyDefinition.ElementType; }
            set { this.EditingPropertyDefinition.ElementType = value; }
        }

        /// <summary>Get or set the description of the property.</summary>
        [XmlIgnore]
        public string Description
        {
            get { return this.EditingPropertyDefinition.Description; }
            set { this.EditingPropertyDefinition.Description = value; }
        }

        /// <summary>Get or set string representation of the default value of the property.</summary>
        [XmlIgnore]
        public string DefaultValue
        {
            get { return this.EditingPropertyDefinition.DefaultValue; }
            set { this.EditingPropertyDefinition.DefaultValue = value; }
        }

        /// <summary>
        /// The flag indicating whether to ignore this property for XML serialization.
        /// </summary>
        [XmlIgnore]
        public bool XmlIgnore
        {
            get { return this.EditingPropertyDefinition.XmlIgnore; }
            set { this.EditingPropertyDefinition.XmlIgnore = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable custom setter or not.
        /// </summary>
        [XmlIgnore]
        public bool CustomSetterEnabled
        {
            get { return this.EditingPropertyDefinition.CustomSetterEnabled; }
            set { this.EditingPropertyDefinition.CustomSetterEnabled = value; }
        }

        /// <summary>
        /// ユーザーが定義できるカスタムセッター.
        /// </summary>
        [XmlIgnore]
        public string CustomSetter
        {
            get { return this.EditingPropertyDefinition.CustomSetter; }
            set { this.EditingPropertyDefinition.CustomSetter = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable custom getter or not.
        /// </summary>
        [XmlIgnore]
        public bool CustomGetterEnabled
        {
            get { return this.EditingPropertyDefinition.CustomGetterEnabled; }
            set { this.EditingPropertyDefinition.CustomGetterEnabled = value; }
        }

        /// <summary>
        /// ユーザーが定義できるカスタムゲッター.
        /// </summary>
        [XmlIgnore]
        public string CustomGetter
        {
            get { return this.EditingPropertyDefinition.CustomGetter; }
            set { this.EditingPropertyDefinition.CustomGetter = value; }
        }

        /// <summary>
        /// カスタムプロパティ設定時にフィールド定義をするかどうかのフラグです.
        /// </summary>
        [XmlIgnore]
        public bool EnableFieldDeclaration
        {
            get { return this.EditingPropertyDefinition.EnableFieldDeclaration; }
            set { this.EditingPropertyDefinition.EnableFieldDeclaration = value; }
        }

        /// <summary>
        /// ReadXmlにおいて規定値をnullとして読み込む処理を挿入するかどうかのフラグです.
        /// </summary>
        [XmlIgnore]
        public bool EnableNullOriginal
        {
            get { return this.EditingPropertyDefinition.EnableNullOriginal; }
            set { this.EditingPropertyDefinition.EnableNullOriginal = value; }
        }

        /// <summary>
        /// Get or set the version converter.
        /// </summary>
        [XmlIgnore]
        public string VersionConverter
        {
            get { return this.EditingPropertyDefinition.VersionConverter; }
            set { this.EditingPropertyDefinition.VersionConverter = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to always apply the version converter.
        /// The version converter only generated to the source code at this version if
        /// this flag is false, otherwise, the version converter will be applied to all
        /// the versions after this version.
        /// </summary>
        [XmlIgnore]
        public bool AlwaysApplyVersionConverter
        {
            get { return this.EditingPropertyDefinition.AlwaysApplyVersionConverter; }
            set { this.EditingPropertyDefinition.AlwaysApplyVersionConverter = value; }
        }

        /// <summary>
        /// Get the name of the property with the first character in lower case.
        /// </summary>
        [XmlIgnore]
        public string NameFirstCharLower
        {
            get
            {
                if (string.IsNullOrEmpty(this.Name) == true)
                {
                    return string.Empty;
                }
                else if (this.Name.Length == 1)
                {
                    return this.Name.ToLowerInvariant();
                }

                return char.ToLowerInvariant(this.Name[0]) + this.Name.Substring(1);
            }
        }

        /// <summary>
        /// ジェネリックであるかどうかチェックします.
        /// </summary>
        [XmlIgnore]
        public bool IsGeneric
        {
            get
            {
                EditorTypeInfo info = TypeManager.FindEditorType(this.Namespace, this.Type);
                if (info != null)
                {
                    return info.IsGeneric;
                }

                return false;
            }
        }

        /// <summary>
        /// コレクションであるかどうかチェックします.
        /// </summary>
        [XmlIgnore]
        public bool IsDataModelCollection
        {
            get
            {
                if (this.IsCollection == false)
                {
                    return false;
                }

                EditorTypeInfo info = TypeManager.FindEditorType(this.ElementNamespace, this.ElementType);
                if (info != null)
                {
                    return info.IsDataModel;
                }

                return false;
            }
        }

        /// <summary>
        /// コレクションであるかどうかチェックします.
        /// </summary>
        [XmlIgnore]
        public bool IsCollection
        {
            get
            {
                EditorTypeInfo info = TypeManager.FindEditorType(this.Namespace, this.Type);
                if (info != null)
                {
                    return info.IsCollection;
                }

                return false;
            }
        }

        /// <summary>
        /// カスタムプロパティを持つかどうかチェックします.
        /// </summary>
        [XmlIgnore]
        public bool HasCustomProperty
        {
            get
            {
                return (!string.IsNullOrEmpty(this.CustomSetter) && this.CustomSetterEnabled)
                    || (!string.IsNullOrEmpty(this.CustomGetter) && this.CustomGetterEnabled);
            }
        }

        /// <summary>
        /// 属性を取得します.
        /// </summary>
        [XmlIgnore]
        public string Attribute
        {
            get
            {
                if (this.XmlIgnore)
                {
                    // インデントを含む形.
                    return "\r\n        [XmlIgnore]";
                }

                return string.Empty;
            }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChildren(this.Versions);
            base.Dispose();
        }

        /// <summary>
        /// Sort the version definitions so they will be ordered by the version number.
        /// </summary>
        public void SortVersionDefinitions()
        {
            this.versionDefs.Sort((a, b) => a.VersionIndex.CompareTo(b.VersionIndex));
        }

        /// <summary>
        /// Update the editing version to match the version set in the root definition.
        /// </summary>
        public void UpdateEditingVersion()
        {
            // Check if the editing version is changed in the EditorDataModelRootDefinition.
            var rootDef = this.FindParent<EditorDataModelRootDefinition>();
            if (rootDef == null)
            {
                this.latestEditedVersionIndex = this.Versions.Count - 1;
                this.editingVersionDef = this.Versions[this.latestEditedVersionIndex];
                this.editingPropertyDef = this.editingVersionDef.PropertyDefinition;
                return;
            }

            int index;
            if (this.EditingVersionLock == null)
            {
                index = rootDef.EditingVersionIndex;
            }
            else
            {
                index = rootDef.FindVersionIndex(this.EditingVersionLock.EditingVersion);
            }

            if (index < 0)
            {
                return;
            }

            if (this.latestEditedVersionIndex == index)
            {
                return;
            }

            // Find the version definition that is closest and is older than the
            // editing version.
            EditorPropertyVersionDefinition closestVersionDef = null;
            int versionDefIndex = -1;
            for (int i = 0; i < this.versionDefs.Count; ++i)
            {
                var versionDef = this.versionDefs[i];
                if (closestVersionDef == null ||
                    versionDef.VersionIndex > closestVersionDef.VersionIndex)
                {
                    if (versionDef.VersionIndex <= index)
                    {
                        closestVersionDef = versionDef;
                        versionDefIndex = i;
                    }
                }
            }

            // Save the editing version definition.
            this.editingVersionDef = closestVersionDef;

            // Update the property definition to use for the editing version.
            if (this.editingVersionDef == null)
            {
                this.editingPropertyDef = null;
            }
            else
            {
                if (this.editingVersionDef.PropertyDefinition == null &&
                    versionDefIndex > 0)
                {
                    // This version does not have the property definition,
                    // use the property definition of the version before
                    // the editing version.
                    this.editingPropertyDef =
                        this.versionDefs[versionDefIndex - 1].PropertyDefinition;
                }
                else
                {
                    this.editingPropertyDef =
                        this.editingVersionDef.PropertyDefinition;
                }
            }
        }
    }

    /// <summary>
    /// ソースコード生成のための一時クラスです.
    /// Temporary class for source code generation.
    /// </summary>
    public class EditorDataModelSuperClassDefinition
    {
        /// <summary>
        /// コンストラクタ.
        /// </summary>
        public EditorDataModelSuperClassDefinition()
        {
            this.Name = string.Empty;
        }

        /// <summary>
        /// クラス名を設定/取得します.
        /// </summary>
        public string Name { get; set; }
    }
}
