﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Windows.Forms;
using System.Xml.Serialization;

using EffectMaker.DataModelMaker.Application.Properties;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.Generators;

using EffectMaker.DataModelMaker.UIControls.BinaryDataEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;
using EffectMaker.DataModelMaker.UIControls.VersionEditor;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

namespace EffectMaker.DataModelMaker.Application
{
    /// <summary>
    /// The class for the main window of the application.
    /// </summary>
    public partial class MainForm : Form
    {
        /// <summary>The current active content view.</summary>
        private ContentViews activeContentView = ContentViews.None;

        /// <summary>
        /// Constructor.
        /// </summary>
        public MainForm()
        {
            this.InitializeComponent();

            // Set up MyDocument as the last opened folder.
            if (string.IsNullOrEmpty(Settings.Default.LastOpendProjectFolderPath) == true)
            {
                Settings.Default.LastOpendProjectFolderPath =
                    Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            }

            // Set up MyDocument as the last exported folder.
            if (string.IsNullOrEmpty(Settings.Default.LastExportedFolderPath) == true)
            {
                Settings.Default.LastExportedFolderPath =
                    Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            }
        }

        /// <summary>
        /// Enum for the content views.
        /// </summary>
        private enum ContentViews
        {
            /// <summary>No content view is active.</summary>
            None,

            /// <summary>The editor data model view.</summary>
            EditorDataModelView,

            /// <summary>The runtime data model view.</summary>
            RuntimeDataModelView,

            /// <summary>The editor data model version view.</summary>
            EditorDataModelVersionView,

            /// <summary>The binary data view.</summary>
            BinaryDataView,

            /// <summary>The conversion editor view.</summary>
            ConversionView,
        }

        /// <summary>
        /// Handle Load event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnLoad(EventArgs e)
        {
            // Set window location.
            if (Settings.Default.MainFormLocation != null)
            {
                this.Location = Settings.Default.MainFormLocation;
            }

            // Set window size.
            if (Settings.Default.MainFormSize != null)
            {
                this.Size = Settings.Default.MainFormSize;
            }

            base.OnLoad(e);
        }

        /// <summary>
        /// Handle FormClosing event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            Settings.Default.MainFormLocation = this.Location;

            if (this.WindowState == FormWindowState.Normal)
            {
                Settings.Default.MainFormSize = this.Size;
            }
            else
            {
                Settings.Default.MainFormSize = this.RestoreBounds.Size;
            }

            base.OnFormClosing(e);
        }

        /// <summary>
        /// Handle main menu item "Create New Project".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemNewProject(object sender, EventArgs e)
        {
            // Create the definitions.
            Program.WorkspaceDefinition = new WorkspaceDefinition()
            {
                EditorDataModelDefinition = new EditorDataModelRootDefinition(),
                RuntimeDataModelDefinition = new RuntimeDataModelRootDefinition()
            };

            // Create view models for the definitions.
            Program.WorkspaceVM = new WorkspaceViewModel(null, Program.WorkspaceDefinition);

            Program.WorkspaceVM.RootEditorDataModelDefinition =
                new RootEditorDataModelDefinitionViewModel(
                    Program.WorkspaceVM,
                    Program.WorkspaceDefinition.EditorDataModelDefinition);

            Program.WorkspaceVM.RootRuntimeDataModelDefinition =
                new RootRuntimeDataModelDefinitionViewModel(
                    Program.WorkspaceVM,
                    Program.WorkspaceDefinition.RuntimeDataModelDefinition);

            Program.WorkspaceVM.RootBinaryDataDefinition =
                new RootBinaryDataDefinitionViewModel(
                    Program.WorkspaceVM,
                    Program.WorkspaceDefinition.RuntimeDataModelDefinition);

            // Show editor data model view.
            this.ShowEditorDataModelView();
        }

        /// <summary>
        /// Handle main menu item "Save Project".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemSaveProject(object sender, EventArgs e)
        {
            if (Program.WorkspaceDefinition == null)
            {
                MessageBox.Show(
                    "No project to save.",
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            string path = Program.WorkspaceDefinition.FilePath;
            if (string.IsNullOrEmpty(path) == true)
            {
                // The workspace hasn't been saved yet, try save as.
                this.OnMainMenuItemSaveAsProject(sender, e);
                return;
            }

            string fileName = Path.GetFileNameWithoutExtension(path);
            string folderPath = Path.GetDirectoryName(path);

            if (Directory.Exists(folderPath) == false)
            {
                Directory.CreateDirectory(folderPath);
            }

            // Save the folder path.
            Settings.Default.LastOpendProjectFolderPath = folderPath;

            // Save the editor data model definition file.
            string myFileName = Program.WorkspaceDefinition.EditorDataModelDefinitionFilePath;
            if (string.IsNullOrEmpty(myFileName) == true)
            {
                myFileName = fileName + "_editor.xml";
            }

            string myFilePath = Path.GetFullPath(Path.Combine(folderPath, myFileName));

            try
            {
                using (var stream =
                    new FileStream(myFilePath, FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(EditorDataModelRootDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition.EditorDataModelDefinition);
                }

                Program.WorkspaceDefinition.EditorDataModelDefinitionFilePath = myFileName;
            }
            catch (Exception ex)
            {
                string msg = string.Format(
                    "Failed saving the editor data model definition file to {0}\nError message: {1}",
                    myFilePath,
                    ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            // Save the runtime data model definition file.
            myFileName = Program.WorkspaceDefinition.RuntimeDataModelDefinitionFilePath;
            if (string.IsNullOrEmpty(myFileName) == true)
            {
                myFileName = fileName + "_runtime.xml";
            }

            myFilePath = Path.GetFullPath(Path.Combine(folderPath, myFileName));

            try
            {
                using (var stream =
                    new FileStream(Path.GetFullPath(myFilePath), FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(RuntimeDataModelRootDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition.RuntimeDataModelDefinition);
                }

                Program.WorkspaceDefinition.RuntimeDataModelDefinitionFilePath = myFileName;
            }
            catch (Exception ex)
            {
                string msg = string.Format(
                    "Failed saving the runtime data model definition file to {0}\nError message: {1}",
                    myFilePath,
                    ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            // Save the project file.
            try
            {
                using (var stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(WorkspaceDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition);
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format("Failed saving the project file to {0}\nError message: {1}", path, ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }
        }

        /// <summary>
        /// Handle main menu item "Save Project As...".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemSaveAsProject(object sender, EventArgs e)
        {
            if (Program.WorkspaceDefinition == null)
            {
                MessageBox.Show(
                    "No project to save.",
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            var dialog = new SaveFileDialog()
            {
                Title = "Save Project",
                AutoUpgradeEnabled = true,
                InitialDirectory = Settings.Default.LastOpendProjectFolderPath,
                Filter = "XML Files (*.xml)|*.xml"
            };

            DialogResult result = dialog.ShowDialog(this);
            if (result != DialogResult.OK)
            {
                return;
            }

            string path = Path.GetFullPath(dialog.FileName);
            string fileName = Path.GetFileNameWithoutExtension(dialog.FileName);
            string folderPath = Path.GetDirectoryName(path);

            // Save the folder path.
            Settings.Default.LastOpendProjectFolderPath = folderPath;

            // Save the editor data model definition file.
            string myFileName = Program.WorkspaceDefinition.EditorDataModelDefinitionFilePath;
            if (string.IsNullOrEmpty(myFileName) == true)
            {
                myFileName = fileName + "_editor.xml";
            }

            string myFilePath = Path.GetFullPath(Path.Combine(folderPath, myFileName));

            try
            {
                using (var stream =
                    new FileStream(myFilePath, FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(EditorDataModelRootDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition.EditorDataModelDefinition);
                }

                Program.WorkspaceDefinition.EditorDataModelDefinitionFilePath = myFileName;
            }
            catch (Exception ex)
            {
                string msg = string.Format(
                    "Failed saving the editor data model definition file to {0}\nError message: {1}",
                    myFilePath,
                    ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            // Save the runtime data model definition file.
            myFileName = Program.WorkspaceDefinition.RuntimeDataModelDefinitionFilePath;
            if (string.IsNullOrEmpty(myFileName) == true)
            {
                myFileName = fileName + "_runtime.xml";
            }

            myFilePath = Path.GetFullPath(Path.Combine(folderPath, myFileName));

            try
            {
                using (var stream =
                    new FileStream(Path.GetFullPath(myFilePath), FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(RuntimeDataModelRootDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition.RuntimeDataModelDefinition);
                }

                Program.WorkspaceDefinition.RuntimeDataModelDefinitionFilePath = myFileName;
            }
            catch (Exception ex)
            {
                string msg = string.Format(
                    "Failed saving the runtime data model definition file to {0}\nError message: {1}",
                    myFilePath,
                    ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            // Save the project file.
            try
            {
                using (var stream = new FileStream(path, FileMode.Create, FileAccess.Write))
                {
                    var serializer = new XmlSerializer(typeof(WorkspaceDefinition));
                    serializer.Serialize(stream, Program.WorkspaceDefinition);
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format("Failed saving the project file to {0}\nError message: {1}", path, ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            // Save the file path to the workspace definition.
            Program.WorkspaceDefinition.FilePath = path;
        }

        /// <summary>
        /// Handle main menu item "Load Project".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemLoadProject(object sender, EventArgs e)
        {
            var dialog = new OpenFileDialog()
            {
                Title = "Load Project",
                AutoUpgradeEnabled = true,
                InitialDirectory = Settings.Default.LastOpendProjectFolderPath,
                Multiselect = false,
                Filter = "XML Files (*.xml)|*.xml|All Files (*.*)|*.*"
            };

            DialogResult result = dialog.ShowDialog(this);
            if (result != DialogResult.OK)
            {
                return;
            }

            // First close the project.
            this.OnMainMenuItemCloseProject(this, EventArgs.Empty);

            string path = Path.GetFullPath(dialog.FileName);

            // Save the folder path.
            Settings.Default.LastOpendProjectFolderPath = Path.GetDirectoryName(path);

            // Load project file.
            WorkspaceDefinition def = null;
            try
            {
                using (var stream = new FileStream(path, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(WorkspaceDefinition));
                    def = serializer.Deserialize(stream) as WorkspaceDefinition;
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format("Failed loading the project file from {0}\nError message: {1}", path, ex.Message);
                MessageBox.Show(
                    msg,
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            if (def == null)
            {
                return;
            }

            // Save the workspace definition.
            Program.WorkspaceDefinition = def;

            // Save the file path to the workspace definition.
            def.FilePath = path;

            // Load the definition files.
            Program.WorkspaceVM = new WorkspaceViewModel(null, Program.WorkspaceDefinition);
            if (Program.WorkspaceVM.LoadFromFile(Path.GetDirectoryName(path)) == false)
            {
                return;
            }

            // Show editor data model view.
            this.ShowEditorDataModelView();
        }

        /// <summary>
        /// Handle main menu item "Close Project".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemCloseProject(object sender, EventArgs e)
        {
            if (Program.WorkspaceDefinition != null)
            {
                Program.WorkspaceDefinition.Dispose();
                Program.WorkspaceDefinition = null;
            }

            if (Program.WorkspaceVM != null)
            {
                Program.WorkspaceVM.Dispose();
                Program.WorkspaceVM = null;
            }

            this.ClearContentPanel();

            this.activeContentView = ContentViews.None;

            WorkspaceManager.OutputRegisteredDefinitions();
            TypeManager.OutputRegisteredTypes();
        }

        /// <summary>
        /// Handle main menu item "Export".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemExport(object sender, EventArgs e)
        {
            if (Program.WorkspaceDefinition == null)
            {
                MessageBox.Show(
                    "No project to export.",
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);

                return;
            }

            var dialog = new FolderBrowserDialog()
            {
                SelectedPath = Properties.Settings.Default.LastExportedFolderPath,
                ShowNewFolderButton = true,
                Description = "Export"
            };

            if (dialog.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            // Save the selected folder path to config.
            Properties.Settings.Default.LastExportedFolderPath = dialog.SelectedPath;

            // Compose editor data model output path.
            string editorDataModelPath = Path.Combine(dialog.SelectedPath, "EditorDataModels");
            if (Directory.Exists(editorDataModelPath) == false)
            {
                Directory.CreateDirectory(editorDataModelPath);
            }

            // Create the generators.
            var editorDataModelGenerator =
                new DataModelGenerator(Program.WorkspaceDefinition.EditorDataModelDefinition);

            int progressSteps = editorDataModelGenerator.EvaluateTotalProgressSteps();
            editorDataModelGenerator.ProgressAdvanced += (s, evt) =>
            {
                this.mainProgressBar.PerformStep();
            };

            var runtimeDataModelGenerator =
                new RuntimeDataModelGenerator(Program.WorkspaceDefinition.RuntimeDataModelDefinition);

            var converterGenerator =
                new BinaryConversionInfoGenerator(Program.WorkspaceDefinition.RuntimeDataModelDefinition);

            // Generate editor data models.
            this.mainProgressBar.Visible = true;
            this.mainProgressBar.Minimum = 1;
            this.mainProgressBar.Maximum = progressSteps;
            this.mainProgressBar.Value = 1;
            this.mainProgressBar.Step = 1;

            editorDataModelGenerator.Generate(editorDataModelPath);

            this.mainProgressBar.Visible = false;

            // ランタイムデータモデルをバリデート
            var runtimeDataModelValidator =
                new RuntimeDataModelValidator(Program.WorkspaceDefinition.RuntimeDataModelDefinition);
            runtimeDataModelValidator.Validate();

            // Compose runtime data model output path.
            string runtimeDataModelPath = Path.Combine(dialog.SelectedPath, "RuntimeDataModels");
            if (Directory.Exists(runtimeDataModelPath) == false)
            {
                Directory.CreateDirectory(runtimeDataModelPath);
            }

            // Generate runtime data models.
            runtimeDataModelGenerator.Generate(runtimeDataModelPath);

            // Compose converter output path.
            string converterDataModelPath = Path.Combine(dialog.SelectedPath, "ConverterSourceFiles");
            if (Directory.Exists(converterDataModelPath) == false)
            {
                Directory.CreateDirectory(converterDataModelPath);
            }

            // Generate conversion.
            converterGenerator.Generate(converterDataModelPath);

            // XPath テーブルを生成
            {
                var xpathTableGenerator = new XPathTableGenerator(Program.WorkspaceDefinition.EditorDataModelDefinition);

                string xpathTablePath = Path.Combine(dialog.SelectedPath, "XPathTable");
                if (Directory.Exists(xpathTablePath) == false)
                {
                    Directory.CreateDirectory(xpathTablePath);
                }

                xpathTableGenerator.Generate(xpathTablePath);
            }
        }

        /// <summary>
        /// Handle main menu item "Quit Application".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMainMenuItemQuitApplication(object sender, EventArgs e)
        {
            System.Windows.Forms.Application.Exit();
        }

        /// <summary>
        /// Handle main menu item "Edit" DropDownOpening event.
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnEditMenuItemOpening(object sender, EventArgs e)
        {
            switch (this.activeContentView)
            {
                case ContentViews.EditorDataModelView:
                    {
                        this.editorVersionMenuItem.Visible = true;
                        this.createEditorDataModelMenuItem.Visible = true;
                        this.createRuntimeDataModelMenuItem.Visible = false;
                        this.createBinaryDataMenuItem.Visible = false;
                        this.deleteEditorDataModelMenuItem.Visible = true;
                        this.deleteRuntimeDataModelMenuItem.Visible = false;
                        this.deleteBinaryDataMenuItem.Visible = false;
                        break;
                    }

                case ContentViews.EditorDataModelVersionView:
                    {
                        this.editorVersionMenuItem.Visible = true;
                        this.createEditorDataModelMenuItem.Visible = true;
                        this.createRuntimeDataModelMenuItem.Visible = false;
                        this.createBinaryDataMenuItem.Visible = false;
                        this.deleteEditorDataModelMenuItem.Visible = true;
                        this.deleteRuntimeDataModelMenuItem.Visible = false;
                        this.deleteBinaryDataMenuItem.Visible = false;
                        break;
                    }

                case ContentViews.RuntimeDataModelView:
                    {
                        this.editorVersionMenuItem.Visible = false;
                        this.createEditorDataModelMenuItem.Visible = false;
                        this.createRuntimeDataModelMenuItem.Visible = true;
                        this.createBinaryDataMenuItem.Visible = false;
                        this.deleteEditorDataModelMenuItem.Visible = false;
                        this.deleteRuntimeDataModelMenuItem.Visible = true;
                        this.deleteBinaryDataMenuItem.Visible = false;
                        break;
                    }

                case ContentViews.BinaryDataView:
                    {
                        this.editorVersionMenuItem.Visible = false;
                        this.createEditorDataModelMenuItem.Visible = false;
                        this.createRuntimeDataModelMenuItem.Visible = false;
                        this.createBinaryDataMenuItem.Visible = true;
                        this.deleteEditorDataModelMenuItem.Visible = false;
                        this.deleteRuntimeDataModelMenuItem.Visible = false;
                        this.deleteBinaryDataMenuItem.Visible = true;
                        break;
                    }

                case ContentViews.ConversionView:
                    {
                        this.editorVersionMenuItem.Visible = false;
                        this.createEditorDataModelMenuItem.Visible = false;
                        this.createRuntimeDataModelMenuItem.Visible = false;
                        this.createBinaryDataMenuItem.Visible = true;
                        this.deleteEditorDataModelMenuItem.Visible = false;
                        this.deleteRuntimeDataModelMenuItem.Visible = false;
                        this.deleteBinaryDataMenuItem.Visible = false;
                        break;
                    }

                default:
                    {
                        this.editorVersionMenuItem.Visible = false;
                        this.createEditorDataModelMenuItem.Visible = false;
                        this.createRuntimeDataModelMenuItem.Visible = false;
                        this.createBinaryDataMenuItem.Visible = false;
                        this.deleteEditorDataModelMenuItem.Visible = false;
                        this.deleteRuntimeDataModelMenuItem.Visible = false;
                        this.deleteBinaryDataMenuItem.Visible = false;
                        break;
                    }
            }
        }

        /// <summary>
        /// Handle main menu item "Edit editor versions".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnEditEditorVersions(object sender, EventArgs e)
        {
            this.ShowEditorVersionDialog();
        }

        /// <summary>
        /// Handle main menu item "Create editor data model".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnCreateEditorDataModel(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var myViewModel = Program.WorkspaceVM.RootEditorDataModelDefinition;
            if (myViewModel == null)
            {
                return;
            }

            var dialog = new CreateDataModelDialog()
            {
                DataSource = myViewModel,
                StartPosition = FormStartPosition.CenterParent
            };

            DialogResult result = dialog.ShowDialog(this);
            if (result != DialogResult.OK)
            {
                return;
            }

            if (myViewModel.SelectedDataModel == null)
            {
                return;
            }

            var editor = new DataModelEditorDialog()
            {
                DataSource = myViewModel.SelectedDataModel,
                ShouldDeleteDataSourceOnCancel = true,
                StartPosition = FormStartPosition.CenterParent
            };

            editor.ShowDialog(this);
        }

        /// <summary>
        /// Handle main menu item "Create runtime data model".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnCreateRuntimeDataModel(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var myViewModel = Program.WorkspaceVM.RootRuntimeDataModelDefinition;
            if (myViewModel == null)
            {
                return;
            }

            var dialog = new CreateDataModelDialog()
            {
                DataSource = myViewModel,
                StartPosition = FormStartPosition.CenterParent
            };

            DialogResult result = dialog.ShowDialog(this);
            if (result != DialogResult.OK)
            {
                return;
            }

            if (myViewModel.SelectedDataModel == null)
            {
                return;
            }

            var editor = new DataModelEditorDialog()
            {
                DataSource = myViewModel.SelectedDataModel,
                ShouldDeleteDataSourceOnCancel = true,
                StartPosition = FormStartPosition.CenterParent
            };

            editor.ShowDialog(this);
        }

        /// <summary>
        /// Handle main menu item "Create binary data".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnCreateBinaryData(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var myViewModel = Program.WorkspaceVM.RootBinaryDataDefinition.CreateBinaryData();
            if (myViewModel == null)
            {
                return;
            }

            var dialog = new BinaryDataEditorDialog()
            {
                DataSource = myViewModel,
                ShouldDeleteDataSourceOnCancel = true,
                StartPosition = FormStartPosition.CenterParent
            };

            dialog.ShowDialog(this);
        }

        /// <summary>
        /// Handle main menu item "Delete selected editor data model".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDeleteSelectedEditorDataModel(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var selectedDataModelVM =
                Program.WorkspaceVM.RootEditorDataModelDefinition.SelectedDataModel;
            if (selectedDataModelVM == null)
            {
                MessageBox.Show(
                    "Please select the data model to delete from the list on the left side.",
                    "Information",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }

            DialogResult confirmationResult = MessageBox.Show(
                "Are you sure you want to delete the data model '" + selectedDataModelVM.DataModelName + "'?",
                "Delete Binary Data",
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question);
            if (confirmationResult != DialogResult.Yes)
            {
                return;
            }

            // Delete the data model.
            Program.WorkspaceVM.RootEditorDataModelDefinition.DeleteDataModel(selectedDataModelVM);
        }

        /// <summary>
        /// Handle main menu item "Delete selected runtime data model".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDeleteSelectedRuntimeDataModel(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var selectedDataModelVM =
                Program.WorkspaceVM.RootRuntimeDataModelDefinition.SelectedDataModel;
            if (selectedDataModelVM == null)
            {
                MessageBox.Show(
                    "Please select the data model to delete from the list on the left side.",
                    "Information",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }

            DialogResult confirmationResult = MessageBox.Show(
                "Are you sure you want to delete the data model '" + selectedDataModelVM.DataModelName + "'?",
                "Delete Binary Data",
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question);
            if (confirmationResult != DialogResult.Yes)
            {
                return;
            }

            // Delete the data model.
            Program.WorkspaceVM.RootRuntimeDataModelDefinition.DeleteDataModel(selectedDataModelVM);
        }

        /// <summary>
        /// Handle main menu item "Delete selected binary data".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDeleteSelectedBinaryData(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var selectedBinaryDataVM =
                Program.WorkspaceVM.RootBinaryDataDefinition.SelectedBinaryDataViewModel;
            if (selectedBinaryDataVM == null)
            {
                MessageBox.Show(
                    "Please select the binary data to delete from the list on the left side.",
                    "Information",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }

            DialogResult confirmationResult = MessageBox.Show(
                "Are you sure you want to delete the binary data '" + selectedBinaryDataVM.Name + "'?",
                "Delete Binary Data",
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question);
            if (confirmationResult != DialogResult.Yes)
            {
                return;
            }

            // Delete the binary data.
            Program.WorkspaceVM.RootBinaryDataDefinition.DeleteBinaryData(selectedBinaryDataVM);
        }

        /// <summary>
        /// Handle main menu item "View" DropDownOpening event.
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnViewMenuItemOpening(object sender, EventArgs e)
        {
            if (Program.WorkspaceVM == null)
            {
                this.editorDataModelViewMenuItem.Enabled = false;
                this.runtimeDataModelViewMenuItem.Enabled = false;
                this.editorDataModelVersionMenuItem.Enabled = false;
                this.binaryDataViewMenuItem.Enabled = false;
                this.conversionViewMenuItem.Enabled = false;
            }
            else
            {
                this.editorDataModelViewMenuItem.Enabled = true;
                this.runtimeDataModelViewMenuItem.Enabled = true;
                this.editorDataModelVersionMenuItem.Enabled = true;
                this.binaryDataViewMenuItem.Enabled = true;
                this.conversionViewMenuItem.Enabled = true;
            }
        }

        /// <summary>
        /// Handle main menu item "Show Editor Data Model View".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnShowEditorDataModelView(object sender, EventArgs e)
        {
            this.ShowEditorDataModelView();
        }

        /// <summary>
        /// Handle main menu item "Show Runtime Data Model View".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnShowRuntimeDataModelView(object sender, EventArgs e)
        {
            this.ShowRuntimeDataModelView();
        }

        /// <summary>
        /// Handle main menu item "Show Editor Data Model Version View".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnShowEditorDataModelVersionView(object sender, EventArgs e)
        {
            this.ShowEditorDataModelVersionView();
        }

        /// <summary>
        /// Handle main menu item "Show Binary Data View".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnShowBinaryDataView(object sender, EventArgs e)
        {
            this.ShowBinaryDataView();
        }

        /// <summary>
        /// Handle main menu item "Show Conversion View".
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnShowConversionView(object sender, EventArgs e)
        {
            this.ShowConversionView();
        }

        /// <summary>
        /// Show editor version dialog.
        /// </summary>
        private void ShowEditorVersionDialog()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            var dialog = new EditorVersionEditorDialog()
            {
                DataContext = Program.WorkspaceVM.RootEditorDataModelDefinition,
            };

            dialog.Show(this);
        }

        /// <summary>
        /// Show editor data model view.
        /// </summary>
        private void ShowEditorDataModelView()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            // First clear all the content views.
            this.ClearContentPanel();

            // Create the editor data model view.
            var view = new EffectMaker.DataModelMaker.UIControls.DataModelView.DataModelView();
            view.Dock = DockStyle.Fill;
            view.DataContext = Program.WorkspaceVM.RootEditorDataModelDefinition;

            // Add the editor data model view to the panel.
            this.contentPanel.Controls.Add(view);

            view.SplitterDistance =
                Properties.Settings.Default.EditorDataModelViewSplitterDistance;

            view.SplitterMoved += (s, e) =>
            {
                Properties.Settings.Default.EditorDataModelViewSplitterDistance = e.SplitX;
            };

            // Set up the active content view.
            this.activeContentView = ContentViews.EditorDataModelView;
        }

        /// <summary>
        /// Show runtime data model view.
        /// </summary>
        private void ShowRuntimeDataModelView()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            // First clear all the content views.
            this.ClearContentPanel();

            // Create the runtime data model view.
            var view = new EffectMaker.DataModelMaker.UIControls.DataModelView.DataModelView();
            view.Dock = DockStyle.Fill;
            view.DataContext = Program.WorkspaceVM.RootRuntimeDataModelDefinition;

            // Add the runtime data model view to the panel.
            this.contentPanel.Controls.Add(view);

            view.SplitterDistance =
                Properties.Settings.Default.RuntimeDataModelViewSplitterDistance;

            view.SplitterMoved += (s, e) =>
            {
                Properties.Settings.Default.RuntimeDataModelViewSplitterDistance = e.SplitX;
            };

            // Set up the active content view.
            this.activeContentView = ContentViews.RuntimeDataModelView;
        }

        /// <summary>
        /// Show editor data model version view.
        /// </summary>
        private void ShowEditorDataModelVersionView()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            // First clear all the content views.
            this.ClearContentPanel();

            // Create the editor data model view.
            var view = new EffectMaker.DataModelMaker.UIControls.DataModelVersionView.DataModelVersionView();
            view.Dock = DockStyle.Fill;
            view.DataContext = Program.WorkspaceVM.RootEditorDataModelDefinition;

            // Add the editor data model view to the panel.
            this.contentPanel.Controls.Add(view);

            view.SplitterDistance =
                Properties.Settings.Default.EditorDataModelVersionViewSplitterDistance;

            view.SplitterMoved += (s, e) =>
            {
                Properties.Settings.Default.EditorDataModelVersionViewSplitterDistance = e.SplitX;
            };

            // Set up the active content view.
            this.activeContentView = ContentViews.EditorDataModelVersionView;
        }

        /// <summary>
        /// Show binary data view.
        /// </summary>
        private void ShowBinaryDataView()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            // First clear all the content views.
            this.ClearContentPanel();

            // Create the runtime data model view.
            var view = new EffectMaker.DataModelMaker.UIControls.BinaryDataView.BinaryDataView();
            view.Dock = DockStyle.Fill;
            view.DataContext = Program.WorkspaceVM.RootBinaryDataDefinition;

            // Add the runtime data model view to the panel.
            this.contentPanel.Controls.Add(view);

            view.LeftSplitterDistance =
                Properties.Settings.Default.BinaryDataViewLeftSplitterDistance;
            view.RightSplitterDistance =
                Properties.Settings.Default.BinaryDataViewRightSplitterDistance;

            view.LeftSplitterMoved += (s, e) =>
            {
                Properties.Settings.Default.BinaryDataViewLeftSplitterDistance = e.SplitX;
            };

            view.RightSplitterMoved += (s, e) =>
            {
                Properties.Settings.Default.BinaryDataViewRightSplitterDistance = e.SplitX;
            };

            // Set up the active content view.
            this.activeContentView = ContentViews.BinaryDataView;
        }

        /// <summary>
        /// Show conversion view.
        /// </summary>
        private void ShowConversionView()
        {
            if (Program.WorkspaceVM == null)
            {
                return;
            }

            // First clear all the content views.
            this.ClearContentPanel();

            // Set up the active content view.
            this.activeContentView = ContentViews.ConversionView;

            // Create the runtime data model view.
            var view = new EffectMaker.DataModelMaker.UIControls.ConversionView.ConversionView();
            view.Dock = DockStyle.Fill;
            view.DataContext = Program.WorkspaceVM.RootBinaryDataDefinition;

            // Add the runtime data model view to the panel.
            this.contentPanel.Controls.Add(view);

            view.SplitterDistance =
                Properties.Settings.Default.ConversionViewSplitterDistance;

            view.SplitterMoved += (s, e) =>
                {
                    Properties.Settings.Default.ConversionViewSplitterDistance = e.SplitX;
                };

            // Set up the active content view.
            this.activeContentView = ContentViews.ConversionView;
        }

        /// <summary>
        /// Clear all the controls in the content panel.
        /// </summary>
        private void ClearContentPanel()
        {
            foreach (Control control in this.contentPanel.Controls)
            {
                control.Dispose();
            }

            this.contentPanel.Controls.Clear();
        }
    }
}
