﻿// ========================================================================
// <copyright file="UIntUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    /// <summary>
    /// uint 関連のユーティリティです。
    /// </summary>
    public static class UIntUtility
    {
        /// <summary>
        /// uint の値が偶数かどうかを示します。
        /// </summary>
        /// <param name="number">uint の値です。</param>
        /// <returns>偶数の場合は true を返します。</returns>
        public static bool IsEven(uint number)
        {
            return (number % 2) == 0;
        }

        /// <summary>
        /// 値を指定された範囲内に制限します。
        /// </summary>
        /// <param name="value">クランプする値です。</param>
        /// <param name="min">最小値です。</param>
        /// <param name="max">最大値です。</param>
        /// <returns>クランプされた値です。</returns>
        public static uint Clamp(uint value, uint min, uint max)
        {
            Ensure.Operation.True(min <= max);

            if (value < min)
            {
                return min;
            }

            if (max < value)
            {
                return max;
            }

            return value;
        }

        /// <summary>
        /// 値が指定された範囲内にあるか調べます。
        /// </summary>
        /// <param name="value">範囲チェックする値です。</param>
        /// <param name="min">最小値です。</param>
        /// <param name="max">最大値です。</param>
        /// <returns>範囲内であれば true を返します。</returns>
        public static bool Contains(uint value, uint min, uint max)
        {
            Ensure.Operation.True(min <= max);

            if (value < min)
            {
                return false;
            }

            if (max < value)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 値にアライメントを加算します。
        /// </summary>
        /// <param name="value">元の値です。</param>
        /// <param name="align">アライメント値です。</param>
        /// <returns>元の値にアライメントの倍数になるように値を加算した結果です。</returns>
        public static uint Align(uint value, uint align)
        {
            // 2のべき乗であるかどうかを判定します。
            Ensure.Argument.True((align & (align - 1)) == 0);

            return (value + (align - 1)) & ~(align - 1);
        }

        /// <summary>
        /// targetの下位bitsビットをvalueの値で上書きして返します。
        /// </summary>
        /// <param name="target">上書きされる値です。</param>
        /// <param name="value">上書きする値です。</param>
        /// <param name="bits">ビット幅です。</param>
        /// <returns>上書き後の値です。</returns>
        public static uint SetLSBs(uint target, uint value, int bits)
        {
            uint mask = (uint)(1 << bits) - 1;

            target &= ~mask;
            target |= value & mask;

            return target;
        }

        /// <summary>
        /// valueの下位bitsビットを取得します。
        /// </summary>
        /// <param name="value">取得元の値です。</param>
        /// <param name="bits">ビット幅です。</param>
        /// <returns>取得した値です。</returns>
        public static uint GetLSBs(uint value, int bits)
        {
            uint mask = (uint)(1 << bits) - 1;
            return value & mask;
        }

        /// <summary>
        /// targetの上位bitsビットをvalueの値で上書きして返します。
        /// </summary>
        /// <param name="target">上書きされる値です。</param>
        /// <param name="value">上書きする値です。</param>
        /// <param name="bits">ビット幅です。</param>
        /// <returns>上書き後の値です。</returns>
        public static uint SetMSBs(uint target, uint value, int bits)
        {
            uint mask = 0xFFFFFFFFU << (32 - bits);

            target &= ~mask;
            target |= value << (32 - bits);

            return target;
        }

        /// <summary>
        /// valueの上位bitsビットを取得します。
        /// </summary>
        /// <param name="value">取得元の値です。</param>
        /// <param name="bits">ビット幅です。</param>
        /// <returns>取得した値です。</returns>
        public static uint GetMSBs(uint value, int bits)
        {
            uint mask = 0xFFFFFFFFU << (32 - bits);
            return (value & mask) >> (32 - bits);
        }
    }
}
