﻿// ========================================================================
// <copyright file="Pointer.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.IO;
    using System.Text;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// バイト列へのポインタを示すクラスです。
    /// </summary>
    public class Pointer
    {
        private long position = 0;
        private byte[] byteStream = null;
        private MemoryStream stream;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="stream">指し示すバイト列です。</param>
        public Pointer(byte[] stream)
        {
            this.byteStream = stream;
            this.stream = new MemoryStream(this.byteStream);
            this.position = 0;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="stream">指し示すバイト列です。</param>
        /// <param name="position">指し示すポイントのオフセット情報です。</param>
        public Pointer(byte[] stream, long position)
        {
            this.byteStream = stream;
            this.stream = new MemoryStream(this.byteStream);
            this.position = position;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="pointer">コピー元のポインタです。</param>
        public Pointer(Pointer pointer)
        {
            this.Set(pointer);
        }

        /// <summary>
        /// 加算演算子のオーバーロードです。
        /// </summary>
        /// <param name="pointer">加算元のポインタです。</param>
        /// <param name="offset">加算するアドレスオフセットです。</param>
        /// <returns>アドレスを加算したポインタを返します。</returns>
        public static Pointer operator +(Pointer pointer, long offset)
        {
            Pointer newPointer = new Pointer(pointer);
            newPointer.Add(offset);
            return newPointer;
        }

        /// <summary>
        /// 加算演算子のオーバーロードです。
        /// </summary>
        /// <param name="pointer">加算元のポインタです。</param>
        /// <param name="offset">加算するアドレスオフセットです。</param>
        /// <returns>アドレスを加算したポインタを返します。</returns>
        public static Pointer operator +(Pointer pointer, ulong offset)
        {
            Pointer newPointer = new Pointer(pointer);
            newPointer.Add((long)offset);
            return newPointer;
        }

        /// <summary>
        /// 減算演算子のオーバーロードです。
        /// </summary>
        /// <param name="pointer">加算元のポインタです。</param>
        /// <param name="offset">減算するアドレスオフセットです。</param>
        /// <returns>アドレスを減算したポインタを返します。</returns>
        public static Pointer operator -(Pointer pointer, long offset)
        {
            Pointer newPointer = new Pointer(pointer);
            newPointer.Add(-offset);
            return newPointer;
        }

        /// <summary>
        /// 減算演算子のオーバーロードです。
        /// </summary>
        /// <param name="pointer">加算元のポインタです。</param>
        /// <param name="offset">減算するアドレスオフセットです。</param>
        /// <returns>アドレスを減算したポインタを返します。</returns>
        public static Pointer operator -(Pointer pointer, ulong offset)
        {
            Pointer newPointer = new Pointer(pointer);
            newPointer.Add(-(long)offset);
            return newPointer;
        }

        /// <summary>
        /// 他のポインタの値をコピーするメソッドです。
        /// </summary>
        /// <param name="pointer">コピー元のポインタです。</param>
        public void Set(Pointer pointer)
        {
            this.byteStream = pointer.byteStream;
            this.stream = new MemoryStream(this.byteStream);
            this.position = pointer.position;
        }

        /// <summary>
        /// クローンメソッドです。
        /// </summary>
        /// <returns>値をコピーした新しいインスタンスを生成します。</returns>
        public Pointer Clone()
        {
            return new Pointer(this);
        }

        /// <summary>
        /// アドレスを進めます。
        /// </summary>
        /// <param name="offset">オフセット情報です。</param>
        public void Add(long offset)
        {
            this.position += offset;
            Ensure.Operation.True(this.position >= 0);
            Ensure.Operation.True(this.position < this.byteStream.Length);
        }

        /// <summary>
        /// ポインタから bool の値を取得します。
        /// </summary>
        /// <returns> bool の値です。</returns>
        public bool PeekBoolean()
        {
            return this.byteStream[this.position] != 0;
        }

        /// <summary>
        /// ポインタから char の値を取得します。
        /// </summary>
        /// <returns> char の値です。</returns>
        public char PeekChar()
        {
            return BitConverter.ToChar(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから byte の値を取得します。
        /// </summary>
        /// <returns> byte の値です。</returns>
        public byte PeekByte()
        {
            return this.byteStream[this.position];
        }

        /// <summary>
        /// ポインタから sbyte の値を取得します。
        /// </summary>
        /// <returns> sbyte の値です。</returns>
        public sbyte PeekSByte()
        {
            return (sbyte)this.byteStream[this.position];
        }

        /// <summary>
        /// ポインタから short の値を取得します。
        /// </summary>
        /// <returns> short の値です。</returns>
        public short PeekInt16()
        {
            return BitConverter.ToInt16(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから ushort の値を取得します。
        /// </summary>
        /// <returns> ushort の値です。</returns>
        public ushort PeekUInt16()
        {
            return BitConverter.ToUInt16(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから int の値を取得します。
        /// </summary>
        /// <returns> int の値です。</returns>
        public int PeekInt32()
        {
            return BitConverter.ToInt32(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから uint の値を取得します。
        /// </summary>
        /// <returns> uint の値です。</returns>
        public uint PeekUInt32()
        {
            return BitConverter.ToUInt32(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから long の値を取得します。
        /// </summary>
        /// <returns> long の値です。</returns>
        public long PeekInt64()
        {
            return BitConverter.ToInt64(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから ulong の値を取得します。
        /// </summary>
        /// <returns> ulong の値です。</returns>
        public ulong PeekUInt64()
        {
            return BitConverter.ToUInt64(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから float の値を取得します。
        /// </summary>
        /// <returns> float の値です。</returns>
        public float PeekSingle()
        {
            return BitConverter.ToSingle(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから double の値を取得します。
        /// </summary>
        /// <returns> double の値です。</returns>
        public double PeekDouble()
        {
            return BitConverter.ToDouble(this.byteStream, (int)this.position);
        }

        /// <summary>
        /// ポインタから string の値を取得します。
        /// </summary>
        /// <param name="encoding">エンコーディングの指定です。</param>
        /// <returns> string の値です。</returns>
        public string PeekString(Encoding encoding)
        {
            int size = 0;

            while (this.byteStream[this.position + size] != 0)
            {
                ++size;
            }

            return encoding.GetString(this.byteStream, (int)this.position, size);
        }

        /// <summary>
        /// ポインタから ASCII として string の値を取得します。
        /// </summary>
        /// <returns> string の値です。</returns>
        public string PeekString()
        {
            return this.PeekString(Encoding.ASCII);
        }

        /// <summary>
        /// ポインタから bool の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> bool の値です。</returns>
        public bool ReadBoolean()
        {
            bool result = this.PeekBoolean();
            this.Add(sizeof(byte));
            return result;
        }

        /// <summary>
        /// ポインタから char の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> char の値です。</returns>
        public char ReadChar()
        {
            char result = this.PeekChar();
            this.Add(sizeof(byte));
            return result;
        }

        /// <summary>
        /// ポインタから byte の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> byte の値です。</returns>
        public byte ReadByte()
        {
            byte result = this.PeekByte();
            this.Add(sizeof(byte));
            return result;
        }

        /// <summary>
        /// ポインタから sbyte の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> sbyte の値です。</returns>
        public sbyte ReadSByte()
        {
            sbyte result = this.PeekSByte();
            this.Add(sizeof(sbyte));
            return result;
        }

        /// <summary>
        /// ポインタから short の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> short の値です。</returns>
        public short ReadInt16()
        {
            short result = this.PeekInt16();
            this.Add(sizeof(short));
            return result;
        }

        /// <summary>
        /// ポインタから ushort の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> ushort の値です。</returns>
        public ushort ReadUInt16()
        {
            ushort result = this.PeekUInt16();
            this.Add(sizeof(ushort));
            return result;
        }

        /// <summary>
        /// ポインタから int の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> int の値です。</returns>
        public int ReadInt32()
        {
            int result = this.PeekInt32();
            this.Add(sizeof(int));
            return result;
        }

        /// <summary>
        /// ポインタから uint の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> uint の値です。</returns>
        public uint ReadUInt32()
        {
            uint result = this.PeekUInt32();
            this.Add(sizeof(uint));
            return result;
        }

        /// <summary>
        /// ポインタから long の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> long の値です。</returns>
        public long ReadInt64()
        {
            long result = this.PeekInt64();
            this.Add(sizeof(long));
            return result;
        }

        /// <summary>
        /// ポインタから ulong の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> ulong の値です。</returns>
        public ulong ReadUInt64()
        {
            ulong result = this.PeekUInt64();
            this.Add(sizeof(ulong));
            return result;
        }

        /// <summary>
        /// ポインタから float の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> float の値です。</returns>
        public float ReadSingle()
        {
            float result = this.PeekSingle();
            this.Add(sizeof(float));
            return result;
        }

        /// <summary>
        /// ポインタから double の値を取得し、ポインタを進めます。
        /// </summary>
        /// <returns> double の値です。</returns>
        public double ReadDouble()
        {
            double result = this.PeekDouble();
            this.Add(sizeof(double));
            return result;
        }
    }
}
