﻿// ========================================================================
// <copyright file="Vector4.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.ComponentModel;
    using System.Diagnostics;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// 4 次元のベクトルを表します。
    /// </summary>
    public sealed class Vector4 : ObservableObject, ICloneable, ISettable, IEquatable<Vector4>, IVector
    {
        //-----------------------------------------------------------------
        private float x;
        private float y;
        private float z;
        private float w;

        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Vector4()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のベクトルです。</param>
        public Vector4(Vector4 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// X,Y,Z,W座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="x">X座標です。</param>
        /// <param name="y">Y座標です。</param>
        /// <param name="z">Z座標です。</param>
        /// <param name="w">W座標です。</param>
        public Vector4(float x, float y, float z, float w)
        {
            this.Set(x, y, z, w);
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="source">X,Y,Z座標の配列です。</param>
        public Vector4(float[] source)
        {
            this.Set(source);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        //-----------------------------------------------------------------

        /// <summary>
        /// 次元数を取得します。
        /// </summary>
        public int Dimension
        {
            get
            {
                return 4;
            }
        }

        /// <summary>
        /// X 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float X
        {
            get
            {
                return this.x;
            }

            set
            {
                this.x = value;
                OnPropertyChanged("X");
            }
        }

        /// <summary>
        /// Y 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float Y
        {
            get
            {
                return this.y;
            }

            set
            {
                this.y = value;
                OnPropertyChanged("Y");
            }
        }

        /// <summary>
        /// Z 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float Z
        {
            get
            {
                return this.z;
            }

            set
            {
                this.z = value;
                OnPropertyChanged("Z");
            }
        }

        /// <summary>
        /// W 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float W
        {
            get
            {
                return this.w;
            }

            set
            {
                this.w = value;
                OnPropertyChanged("W");
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ベクトルの長さを取得します。
        /// </summary>
        /// <returns>ベクトルの長さです。</returns>
        public float Length
        {
            get { return (float)Math.Sqrt(this.LengthSquared); }
        }

        /// <summary>
        /// 長さの 2 乗を取得します。
        /// </summary>
        /// <returns>ベクトルの長さの２乗です。</returns>
        public float LengthSquared
        {
            get { return (this.X * this.X) + (this.Y * this.Y) + (this.Z * this.Z) + (this.W * this.W); }
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します。
        /// </summary>
        /// <param name="i">インデックスです。</param>
        /// <returns>成分です。</returns>
        public float this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    case 1:
                        return this.Y;
                    case 2:
                        return this.Z;
                    case 3:
                        return this.W;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    case 1:
                        this.Y = value;
                        break;
                    case 2:
                        this.Z = value;
                        break;
                    case 3:
                        this.W = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        //-----------------------------------------------------------------
        // オペレータ
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つのベクトルを加算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを加算した結果のベクトルです。</returns>
        public static Vector4 operator +(Vector4 lhs, Vector4 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector4(lhs.X + rhs.X, lhs.Y + rhs.Y, lhs.Z + rhs.Z, lhs.W + rhs.W);
        }

        /// <summary>
        /// 2 つのベクトルを減算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを減算した結果のベクトルです。</returns>
        public static Vector4 operator -(Vector4 lhs, Vector4 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector4(lhs.X - rhs.X, lhs.Y - rhs.Y, lhs.Z - rhs.Z, lhs.W - rhs.W);
        }

        /// <summary>
        /// ベクトルにスカラーを乗算します。
        /// </summary>
        /// <param name="vec">演算の左のベクトルです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>vecとscalarを乗算した結果のベクトルです。</returns>
        public static Vector4 operator *(Vector4 vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);

            return new Vector4(vec.X * scalar, vec.Y * scalar, vec.Z * scalar, vec.W * scalar);
        }

        /// <summary>
        /// ベクトルをスカラーで除算します。
        /// </summary>
        /// <param name="vec">演算の左のベクトルです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>vecとscalarを除算した結果のベクトルです。</returns>
        public static Vector4 operator /(Vector4 vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Operation.DividerNotZero(scalar);

            float invScale = 1.0f / scalar;
            return new Vector4(
                vec.X * invScale, vec.Y * invScale, vec.Z * invScale, vec.W * invScale);
        }

        /// <summary>
        /// 2 つのベクトルの内積を計算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsの外積を計算した結果のスカラーです。</returns>
        public static float Dot(Vector4 lhs, Vector4 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return (lhs.X * rhs.X) + (lhs.Y * rhs.Y) + (lhs.Z * rhs.Z) + (lhs.W * rhs.W);
        }

        /// <summary>
        /// 指定したベクトルを正規化したベクトルを作成します。
        /// </summary>
        /// <remarks>
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// DivideByZeroException　例外を投げます。
        /// </remarks>
        /// <param name="vec">指定のベクトルです。</param>
        /// <returns>正規化したベクトルです。</returns>
        public static Vector4 Normalize(Vector4 vec)
        {
            Vector4 result = new Vector4(vec);
            result.Normalize();
            return result;
        }

        /// <summary>
        /// 正規化したベクトルを作成します。
        /// ベクトル vec の長さが FloatUtility.Epsilon 以下の場合は
        /// 指定したベクトル substitute に設定します。
        /// </summary>
        /// <param name="vec">元のベクトルです。</param>
        /// <param name="substitute">ベクトルの長さが FloatUtility.Epsilon 以下の場合に
        /// 代わりに設定するベクトルです。</param>
        /// <returns>ベクトルの長さを正規化できた場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool SafeNormalize(Vector4 vec, Vector4 substitute)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Argument.NotNull(substitute);

            float lengthSquared = vec.LengthSquared;
            if (vec.IsZero(lengthSquared))
            {
                vec.Set(substitute);
                return false;
            }

            vec.Normalize(lengthSquared);
            return true;
        }

        /// <summary>
        /// X,Y,Z,W座標を格納した配列を返します。
        /// </summary>
        /// <returns>X,Y,Z,W座標の配列です。</returns>
        public float[] ToArray()
        {
            return new float[] { this.X, this.Y, this.Z, this.W };
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            Ensure.Argument.True(source is IVector);
            this.Set(source as IVector);
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。同じ次元でなければなりません。</param>
        public void Set(IVector source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Dimension);

            this.Set(source[0], source[1], source[2], source[3]);
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public void Set(Vector4 source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.X, source.Y, source.Z, source.W);
        }

        /// <summary>
        /// X,Y,Z,W座標を指定の値に設定します。
        /// </summary>
        /// <param name="x">X座標です。</param>
        /// <param name="y">Y座標です。</param>
        /// <param name="z">Z座標です。</param>
        /// <param name="w">W座標です。</param>
        public void Set(float x, float y, float z, float w)
        {
            this.X = x;
            this.Y = y;
            this.Z = z;
            this.W = w;
        }

        /// <summary>
        /// X,Y,Z,W座標を指定の値に設定します。
        /// </summary>
        /// <param name="source">X,Y,Z,W座標の配列です。</param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Length);

            this.Set(source[0], source[1], source[2], source[3]);
        }

        /// <summary>
        /// 成分を 0 に設定します。
        /// </summary>
        public void SetZero()
        {
            this.X = 0f;
            this.Y = 0f;
            this.Z = 0f;
            this.W = 0f;
        }

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Vector4(this);
        }

        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// ベクトルの長さが FloatUtility.Epsilon 以下か示します。
        /// </summary>
        /// <returns>ベクトルの長さが FloatUtility.Epsilon 以下ならば true です。
        /// それ以外の場合は false です。</returns>
        public bool IsZero()
        {
            return this.IsZero(this.LengthSquared);
        }

        /// <summary>
        /// 指定したベクトルとの内積を計算します。
        /// </summary>
        /// <param name="vec">掛けるベクトルです。</param>
        /// <returns>vecとの内積を計算した結果のスカラーです。</returns>
        public float Dot(Vector4 vec)
        {
            return Dot(this, vec);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 正規化したベクトルに設定します。
        /// </summary>
        /// <remarks>
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// DivideByZeroException　例外を投げます。
        /// </remarks>
        public void Normalize()
        {
            float lengthSquared = this.LengthSquared;
            if (this.IsZero(lengthSquared))
            {
                throw new DivideByZeroException();
            }

            Normalize(lengthSquared);
        }

        /// <summary>
        /// 正規化したベクトルに設定します。
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// 指定したベクトルに設定します。
        /// </summary>
        /// <param name="substitute">ベクトルの長さが FloatUtility.Epsilon 以下の場合に
        /// 代わりに設定するベクトルです。</param>
        /// <returns>ベクトルの長さを正規化できた場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public bool SafeNormalize(Vector4 substitute)
        {
            return SafeNormalize(this, substitute);
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Vector4);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Vector4 other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                (this.X == other.X) &&
                (this.Y == other.Y) &&
                (this.Z == other.Z) &&
                (this.W == other.W);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Vector4 other, float tolerance)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                FloatUtility.NearlyEqual(this.X, other.X, tolerance) &&
                FloatUtility.NearlyEqual(this.Y, other.Y, tolerance) &&
                FloatUtility.NearlyEqual(this.Z, other.Z, tolerance) &&
                FloatUtility.NearlyEqual(this.W, other.W, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode() ^
                this.Y.GetHashCode() ^
                this.Z.GetHashCode() ^
                this.W.GetHashCode();
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("[{0} {1} {2} {3}]", this.X, this.Y, this.Z, this.W);
        }

        // 長さの 2 乗を受け取り、長さがほぼゼロかどうか判定します。
        private bool IsZero(float lengthSquared)
        {
            Debug.Assert(lengthSquared == this.LengthSquared, "Unexpected case!");

            float epsilonSquared = FloatUtility.Epsilon * FloatUtility.Epsilon;
            return lengthSquared <= epsilonSquared;
        }

        // 長さの 2 乗を受け取り長さを１に正規化
        private void Normalize(float lengthSquared)
        {
            Debug.Assert(lengthSquared == this.LengthSquared, "Unexpected case!");

            if (lengthSquared == 1f)
            {
                return;
            }

            float inverseLength = 1f / (float)Math.Sqrt(lengthSquared);
            this.Set(this * inverseLength);
        }
    }
}
