﻿// ========================================================================
// <copyright file="Matrix3DTranslateFunction.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System.Diagnostics;

    /// <summary>
    /// 行列の３次元の平行移動に関する演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix3DTranslateFunction<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// x 軸、y 軸、z 軸に沿って平行移動する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="x">x 軸に沿った移動量です。</param>
        /// <param name="y">y 軸に沿った移動量です。</param>
        /// <param name="z">z 軸に沿った移動量です。</param>
        public static void SetTranslate(TMatrix mtx, float x, float y, float z)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            mtx.SetIdentity();
            mtx[0, 3] = x;
            mtx[1, 3] = y;
            mtx[2, 3] = z;
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Debug.Assert(mtx.RowCount >= 3, "Unexpected case!");
            Debug.Assert(mtx.ColumnCount >= 4, "Unexpected case!");
        }
    }
}
