﻿// <auto-generated />
// ========================================================================
// <copyright file="Geometry.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Geometry_
{
    using System;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// </summary>
    public enum IntersectionResult
    {
        /// <summary>
        /// 交差しない。
        /// </summary>
         None = 0,

        /// <summary>
        /// 一点で交差。
        /// </summary>
        Intersect1 = 1,

        /// <summary>
        /// 二点で交差。
        /// </summary>
        Intersect2 = 2,

        /// <summary>
        /// </summary>
        Outside = 0,

        /// <summary>
        /// </summary>
        Inside = 1,

        /// <summary>
        /// </summary>
        Intersect = 2,

        /// <summary>
        /// </summary>
        Line3OnPlane = 2,

        /// <summary>
        /// </summary>
        Ray3OnPlane = 2,

        /// <summary>
        /// </summary>
        Segment3OnPlane = 2,
    }

    /// <summary>
    /// 平面です。
    /// 平面は以下の式で定義されます。
    /// <code>
    /// Normal.X * x + Normal.Y * y + Normal.Z * z + Constant = 0
    /// </code>
    /// 平面の陰関数表現をパラメータとして持っています。
    /// 法線ベクトルNormalは正規化されている必要があります。
    /// </summary>
    public class Plane : IEquatable<Plane>
    {
        /// <summary>
        /// デフォルトコンストラクタです。
        /// プロパティの初期化は行いません。
        /// </summary>
        public Plane()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元です。</param>
        public Plane(Plane source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="normal">法線ベクトルです。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public Plane(Vector3 normal, float constant, bool isNormalized)
        {
            this.Set(normal, constant, isNormalized);
        }

        /// <summary>
        /// コンストラクタです。
        /// 正規化を行います。
        /// </summary>
        /// <param name="normal">法線ベクトルです。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        public Plane(Vector3 normal, float constant)
        {
            this.Set(normal, constant);
        }

        /// <summary>
        /// コンストラクタです。
        /// 正規化を行います。
        /// </summary>
        /// <param name="source">成分を格納した配列です [Normal.X, Normal.Y, Normal.Z, Constant]。</param>
        public Plane(float[] source)
        {
            this.Set(source[0], source[1], source[2], source[3]);
        }

        /// <summary>
        /// コンストラクタです。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="source">成分を格納した配列です [Normal.X, Normal.Y, Normal.Z, Constant]。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public Plane(float[] source, bool isNormalized)
        {
            this.Set(source, isNormalized);
        }

        /// <summary>
        /// コンストラクタです。
        /// 正規化を行います。
        /// </summary>
        /// <param name="nx">法線ベクトルの x 成分です。</param>
        /// <param name="ny">法線ベクトルの y 成分です。</param>
        /// <param name="nz">法線ベクトルの z 成分です。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        public Plane(float nx, float ny, float nz, float constant)
        {
            this.Set(nx, ny, nz, constant);
        }

        /// <summary>
        /// コンストラクタです。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="nx">法線ベクトルの x 成分です。</param>
        /// <param name="ny">法線ベクトルの y 成分です。</param>
        /// <param name="nz">法線ベクトルの z 成分です。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public Plane(float nx, float ny, float nz, float constant, bool isNormalized)
        {
            this.Set(nx, ny, nz, constant, isNormalized);
        }

        /// <summary>
        /// コンストラクタです。
        /// 与えられた点を通る平面を設定します。
        /// p0, p1, p2 は上からみて時計回りとして解釈されます。
        /// </summary>
        /// <param name="p0">平面上の点0です。</param>
        /// <param name="p1">平面上の点1です。</param>
        /// <param name="p2">平面上の点2です。</param>
        public Plane(Vector3 p0, Vector3 p1, Vector3 p2)
        {
            this.Set(p0, p1, p2);
        }

        /// <summary>
        /// 平面の法線です。
        /// </summary>
        public Vector3 Normal { get; private set; }

        /// <summary>
        /// 平面から原点までの法線方向の距離です。
        /// </summary>
        public float Constant { get; private set; }

        /// <summary>
        /// 点の平面に対する位置を求めます。
        /// 返り値が正の場合は点は平面の法線方向に存在します。
        /// 逆に負の場合は平面の法線の逆方向に存在します。
        /// 返り値の絶対値は点と平面の距離となります。
        /// </summary>
        /// <param name="p">テストする点です。</param>
        /// <returns>
        /// 平面の陰関数表現に p を代入した値が返ります。
        /// </returns>
        public float Test(Vector3 p)
        {
            return this.Constant + this.Normal.Dot(p);
        }

        /// <summary>
        /// 平面を設定します。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="normal">法線ベクトルです。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public void Set(Vector3 normal, float constant, bool isNormalized)
        {
            Ensure.Argument.NotNull(normal);

            this.Normal = normal;
            this.Constant = constant;
            if (!isNormalized)
            {
                this.Normalize();
            }
        }

        /// <summary>
        /// 平面を設定します。
        /// 正規化を行います。
        /// </summary>
        /// <param name="normal">法線ベクトルです。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        public void Set(Vector3 normal, float constant)
        {
            this.Set(normal, constant, false);
        }

        /// <summary>
        /// 平面を設定します。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="nx">法線ベクトルの x 成分です。</param>
        /// <param name="ny">法線ベクトルの y 成分です。</param>
        /// <param name="nz">法線ベクトルの z 成分です。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public void Set(float nx, float ny, float nz, float constant, bool isNormalized)
        {
            this.Set(new Vector3(nx, ny, nz), constant, isNormalized);
        }

        /// <summary>
        /// 平面を設定します。
        /// 正規化を行います。
        /// </summary>
        /// <param name="nx">法線ベクトルの x 成分です。</param>
        /// <param name="ny">法線ベクトルの y 成分です。</param>
        /// <param name="nz">法線ベクトルの z 成分です。</param>
        /// <param name="constant">平面の単位法線ベクトルと平面上の点の内積の符号を反転させた値です。</param>
        public void Set(float nx, float ny, float nz, float constant)
        {
            this.Set(nx, ny, nz, constant, false);
        }

        /// <summary>
        /// 平面を設定をコピーします。
        /// </summary>
        /// <param name="source">コピー元です。</param>
        public void Set(Plane source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.Normal, source.Constant, true);
        }

        /// <summary>
        /// 平面を設定します。
        /// isNormalized に false を指定した場合には正規化を行います。
        /// </summary>
        /// <param name="source">成分を格納した配列です [Normal.X, Normal.Y, Normal.Z, Constant]。</param>
        /// <param name="isNormalized">法線ベクトルが正規化されているときは true を指定します。</param>
        public void Set(float[] source, bool isNormalized)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 4);
            this.Set(source[0], source[1], source[2], source[3], isNormalized);
        }

        /// <summary>
        /// 平面を設定します。
        /// 正規化を行います。
        /// </summary>
        /// <param name="source">成分を格納した配列です [Normal.X, Normal.Y, Normal.Z, Constant]。</param>
        public void Set(float[] source)
        {
            this.Set(source, false);
        }

        /// <summary>
        /// 平面上の3点を指定して平面を設定します。
        /// 表面を見たときに3点が時計まわりとなります。
        /// </summary>
        /// <param name="p0">平面上の点 0 です。</param>
        /// <param name="p1">平面上の点 1 です。</param>
        /// <param name="p2">平面上の点 2 です。</param>
        public void Set(Vector3 p0, Vector3 p1, Vector3 p2)
        {
            Ensure.Argument.NotNull(p0);
            Ensure.Argument.NotNull(p1);
            Ensure.Argument.NotNull(p2);

            Vector3 v0 = p2 - p0;
            Vector3 v1 = p1 - p0;
            Vector3 v2 = Vector3.Cross(v0, v1);
            v2.Normalize();
            this.Set(v2, -v2.Dot(p0), true);
        }

        /// <summary>
        /// 平面が球と交差するか調べます。
        /// </summary>
        /// <param name="sphere">球です。</param>
        /// <returns></returns>
        public bool Intersection(Sphere sphere)
        {
            Ensure.Argument.NotNull(sphere);

            // 球の中心と平面との距離を求めます。
            float distance = this.Normal.Dot(sphere.Center) + this.Constant;

            // 距離が球の半径以内なら交差します。
            return Math.Abs(distance) <= sphere.Radius;
        }

        /// <summary>
        /// 平面がAABBと交差するか調べます。
        /// </summary>
        /// <param name="aabb">AABBです。</param>
        /// <returns>交差する場合に true を返します。</returns>
        public bool Intersection(AABB aabb)
        {
            Ensure.Argument.NotNull(aabb);

            //// RCD p.164

            Vector3 center = (aabb.Minimum + aabb.Maximum) * 0.5f;
            Vector3 extent = aabb.Maximum - center; // ここでEの要素は全て正である。

            // AABBの中心からAABBの各点へのベクトルとNとの内積で最大のものはrで最小のものは-rである。
            // つまり法線Nを持ち、AABBの中心を通る平面からr以内はAABBの内部である。
            var r =
                (extent.X * Math.Abs(this.Normal.X)) +
                (extent.Y * Math.Abs(this.Normal.Y)) +
                (extent.Z * Math.Abs(this.Normal.Z));

            // 平面とAABBの中心の距離
            float s = this.Normal.Dot(center) + this.Constant;

            // 平面とAABBの中心の距離がr以内ならば交差する。
            return Math.Abs(s) <= r;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Plane);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Plane other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Normal == other.Normal && this.Constant == other.Constant;
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return this.Normal.GetHashCode() ^ this.Constant.GetHashCode();
        }

        /// <summary>
        /// 正規化を行います。
        /// </summary>
        private void Normalize()
        {
            float length = this.Normal.Length;
            this.Normal = Vector3.Normalize(this.Normal);
            this.Constant /= length;
        }
    }

    /// <summary>
    /// 球状の空間を定義します。
    /// <code>
    /// |P - Center| ≦ Radius
    /// </code>
    /// </summary>
    public class Sphere : IEquatable<Sphere>
    {
        /// <summary>
        /// </summary>
        public Sphere()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Sphere(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="center"></param>
        /// <param name="radius">半径です。</param>
        public Sphere(Vector3 center, float radius)
        {
            this.Set(center, radius);
        }

        /// <summary>
        /// </summary>
        public Vector3 Center { get; private set; }

        /// <summary>
        /// </summary>
        public float Radius { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="center"></param>
        /// <param name="radius">半径です。</param>
        public void Set(Vector3 center, float radius)
        {
            this.Center = center;
            this.Radius = radius;
        }

        /// <summary>
        /// </summary>
        /// <param name="cx"></param>
        /// <param name="cy"></param>
        /// <param name="cz"></param>
        /// <param name="radius">半径です。</param>
        public void Set(float cx, float cy, float cz, float radius)
        {
            this.Set(new Vector3(cx, cy, cz), radius);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Sphere source)
        {
            this.Set(source.Center, source.Radius);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 4);

            this.Set(source[0], source[1], source[2], source[3]);
        }

        /// <summary>
        /// </summary>
        /// <param name="plane">平面です。</param>
        /// <returns></returns>
        public bool Intersection(Plane plane)
        {
            Ensure.Argument.NotNull(plane);

            return plane.Intersection(this);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Intersection(Sphere other)
        {
            Ensure.Argument.NotNull(other);

            // EMG p.536
            Vector3 centerDiff = this.Center - other.Center;
            float radiusSum = this.Radius + other.Radius;

            return centerDiff.LengthSquared <= radiusSum * radiusSum;
        }

        /// <summary>
        /// </summary>
        /// <param name="aabb"></param>
        /// <returns></returns>
        public bool Intersection(AABB aabb)
        {
            Ensure.Argument.NotNull(aabb);

            // GTCG p.645
            float distSq = 0f;

            if (this.Center.X < aabb.Minimum.X)
            {
                distSq += (this.Center.X - aabb.Minimum.X) * (this.Center.X - aabb.Minimum.X);
            }
            else if (this.Center.X > aabb.Maximum.X)
            {
                distSq += (this.Center.X - aabb.Maximum.X) * (this.Center.X - aabb.Maximum.X);
            }

            if (this.Center.Y < aabb.Minimum.Y)
            {
                distSq += (this.Center.Y - aabb.Minimum.Y) * (this.Center.Y - aabb.Minimum.Y);
            }
            else if (this.Center.Y > aabb.Maximum.Y)
            {
                distSq += (this.Center.Y - aabb.Maximum.Y) * (this.Center.Y - aabb.Maximum.Y);
            }

            if (this.Center.Z < aabb.Minimum.Z)
            {
                distSq += (this.Center.Z - aabb.Minimum.Z) * (this.Center.Z - aabb.Minimum.Z);
            }
            else if (this.Center.Z > aabb.Maximum.Z)
            {
                distSq += (this.Center.Z - aabb.Maximum.Z) * (this.Center.Z - aabb.Maximum.Z);
            }

            return distSq <= this.Radius * this.Radius;
        }

        /// <summary>
        /// </summary>
        /// <param name="frustum"></param>
        /// <returns></returns>
        public bool Intersection(Frustum frustum)
        {
            Ensure.Argument.NotNull(frustum);

            return frustum.Intersection(this);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Sphere);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Sphere other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Center == other.Center && this.Radius == other.Radius;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
             return this.Center.GetHashCode() ^ this.Radius.GetHashCode();
        }

        /// <summary>
        /// 点の集合を包含する球をセットします。
        /// </summary>
        /// <param name="points"></param>
        public void Set(Vector3[] points)
        {
            Ensure.Argument.ArrayNotEmpty(points);
            Ensure.Argument.NotNull(points[0]);

            // EMG p.533
            AABB tmp = new AABB();
            tmp.Set(points);

            this.Center = (tmp.Minimum + tmp.Maximum) * 0.5f;

            float maxDistance = (this.Center - points[0]).LengthSquared;
            foreach (var point in points)
            {
                Ensure.Argument.NotNull(point);
                float dist = (this.Center - point).LengthSquared;
                if (dist > maxDistance)
                {
                    maxDistance = dist;
                }
            }

            this.Radius = (float)Math.Sqrt(maxDistance);
        }
    }

    /// <summary>
    /// xyz軸に沿ったボックスです。
    /// 2つの点MinimumとMaximumで定義されます。
    /// Minimumのそれぞれの座標はMaximumの座標よりも小さい必要があります。
    /// </summary>
    public class AABB : IEquatable<AABB>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AABB()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public AABB(AABB source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public AABB(float[] source, bool isNormalized)
        {
            this.Set(source, isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public AABB(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="minimum"></param>
        /// <param name="maximum"></param>
        /// <param name="isNormalized"></param>
        public AABB(Vector3 minimum, Vector3 maximum, bool isNormalized)
        {
            this.Set(minimum, maximum, isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="minimum"></param>
        /// <param name="maximum"></param>
        public AABB(Vector3 minimum, Vector3 maximum)
        {
            this.Set(minimum, maximum);
        }

        /// <summary>
        /// </summary>
        public Vector3 Minimum { get; private set; }

        /// <summary>
        /// </summary>
        public Vector3 Maximum { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="minimum"></param>
        /// <param name="maximum"></param>
        /// <param name="isNormalized"></param>
        public void Set(Vector3 minimum, Vector3 maximum, bool isNormalized)
        {
            Ensure.Argument.NotNull(minimum);
            Ensure.Argument.NotNull(maximum);

            this.Minimum = minimum;
            this.Maximum = maximum;
            if (!isNormalized)
            {
                this.Normalize();
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="minimum"></param>
        /// <param name="maximum"></param>
        public void Set(Vector3 minimum, Vector3 maximum)
        {
            this.Set(minimum, maximum, false);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public void Set(float[] source, bool isNormalized)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 6);

            this.Set(
                new Vector3(source[0], source[1], source[2]), // minimum
                new Vector3(source[3], source[4], source[5]), // maximum
                isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            this.Set(source, true);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(AABB source)
        {
            this.Set(source.Minimum, source.Maximum, true);
        }

        /// <summary>
        /// </summary>
        /// <param name="points"></param>
        public void Set(Vector3[] points)
        {
            Ensure.Argument.ArrayNotEmpty(points);
            Ensure.Argument.NotNull(points[0]);

            // EMG p.540
            this.Minimum = points[0];
            this.Maximum = points[0];

            foreach (var point in points)
            {
                Ensure.Argument.NotNull(point);

                if (point.X < this.Minimum.X)
                {
                    this.Minimum.X = point.X;
                }
                else if (point.X > this.Maximum.X)
                {
                    this.Maximum.X = point.X;
                }

                if (point.Y < this.Minimum.Y)
                {
                    this.Minimum.Y = point.Y;
                }
                else if (point.Y > this.Maximum.Y)
                {
                    this.Maximum.Y = point.Y;
                }

                if (point.Z < this.Minimum.Z)
                {
                    this.Minimum.Z = point.Z;
                }
                else if (point.Z > this.Maximum.Z)
                {
                    this.Maximum.Z = point.Z;
                }
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="box">AABBです。</param>
        /// <param name="transform"></param>
        public void Set(AABB box, Matrix34 transform)
        {
            float x0, y0, z0;
            float x1, y1, z1;

            float a0, a1;
            float b0, b1;

            // 新しいX軸における最小値と最大値を求める
            x0 = (transform.M00 * box.Minimum.X) + transform.M03;
            x1 = (transform.M00 * box.Maximum.X) + transform.M03;
            a0 = transform.M01 * box.Minimum.Y;
            a1 = transform.M01 * box.Maximum.Y;
            b0 = transform.M02 * box.Minimum.Z;
            b1 = transform.M02 * box.Maximum.Z;

            if (x0 > x1)
            {
                float tmp = x0;
                x0 = x1;
                x1 = tmp;
            }

            if (a0 < a1)
            {
                x0 += a0;
                x1 += a1;
            }
            else
            {
                x0 += a1;
                x1 += a0;
            }

            if (b0 < b1)
            {
                x0 += b0;
                x1 += b1;
            }
            else
            {
                x0 += b1;
                x1 += b0;
            }

            // 新しいY軸における最小値と最大値を求める
            y0 = (transform.M10 * box.Minimum.X) + transform.M13;
            y1 = (transform.M10 * box.Maximum.X) + transform.M13;
            a0 = transform.M11 * box.Minimum.Y;
            a1 = transform.M11 * box.Maximum.Y;
            b0 = transform.M12 * box.Minimum.Z;
            b1 = transform.M12 * box.Maximum.Z;

            if (y0 > y1)
            {
                float tmp = y0;
                y0 = y1;
                y1 = tmp;
            }

            if (a0 < a1)
            {
                y0 += a0;
                y1 += a1;
            }
            else
            {
                y0 += a1;
                y1 += a0;
            }

            if (b0 < b1)
            {
                y0 += b0;
                y1 += b1;
            }
            else
            {
                y0 += b1;
                y1 += b0;
            }

            // 新しいZ軸における最小値と最大値を求める
            z0 = (transform.M20 * box.Minimum.X) + transform.M23;
            z1 = (transform.M20 * box.Maximum.X) + transform.M23;
            a0 = transform.M21 * box.Minimum.Y;
            a1 = transform.M21 * box.Maximum.Y;
            b0 = transform.M22 * box.Minimum.Z;
            b1 = transform.M22 * box.Maximum.Z;

            if (z0 > z1)
            {
                float tmp = z0;
                z0 = z1;
                z1 = tmp;
            }

            if (a0 < a1)
            {
                z0 += a0;
                z1 += a1;
            }
            else
            {
                z0 += a1;
                z1 += a0;
            }

            if (b0 < b1)
            {
                z0 += b0;
                z1 += b1;
            }
            else
            {
                z0 += b1;
                z1 += b0;
            }

            this.Minimum = new Vector3(x0, y0, z0);
            this.Maximum = new Vector3(x1, y1, z1);
        }

        /// <summary>
        /// </summary>
        /// <param name="plane">平面です。</param>
        /// <returns></returns>
        public bool Intersection(Plane plane)
        {
            Ensure.Argument.NotNull(plane);

            return plane.Intersection(this);
        }

        /// <summary>
        /// </summary>
        /// <param name="sphere">球です。</param>
        /// <returns></returns>
        public bool Intersection(Sphere sphere)
        {
            Ensure.Argument.NotNull(sphere);

            return sphere.Intersection(this);
        }

        /// <summary>
        /// ２つのAABBの空間が重なるか調べる。
        /// </summary>
        /// <param name="other">判定対象です。</param>
        /// <returns>重なる場合には true を、そうでなければ false を返す。</returns>
        public bool Intersection(AABB other)
        {
            // GTCG p.638
            if (this.Minimum.X > other.Maximum.X || other.Minimum.X > this.Maximum.X ||
                this.Minimum.Y > other.Maximum.Y || other.Minimum.Y > this.Maximum.Y ||
                this.Minimum.Z > other.Maximum.Z || other.Minimum.Z > this.Maximum.Z)
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="frustum"></param>
        /// <returns></returns>
        public bool Intersection(Frustum frustum)
        {
            Ensure.Argument.NotNull(frustum);

            return frustum.Intersection(this);
        }

        /// <summary>
        /// </summary>
        /// <param name="frustum"></param>
        /// <returns></returns>
        public IntersectionResult IntersectEx(Frustum frustum)
        {
            Ensure.Argument.NotNull(frustum);

            return frustum.IntersectEx(this);
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as AABB);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(AABB other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Minimum == other.Minimum && this.Maximum == other.Maximum;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Minimum.GetHashCode() ^ this.Maximum.GetHashCode();
        }

        /// <summary>
        /// </summary>
        private void Normalize()
        {
            Vector3 minimum = this.Minimum;
            Vector3 maximum = this.Maximum;

            this.Minimum = new Vector3(
                Math.Min(minimum.X, maximum.X),
                Math.Min(minimum.Y, maximum.Y),
                Math.Min(minimum.Z, maximum.Z));
            this.Maximum = new Vector3(
                Math.Max(minimum.X, maximum.X),
                Math.Max(minimum.Y, maximum.Y),
                Math.Max(minimum.Z, maximum.Z));
        }
    }

    /// <summary>
    /// パラメトリック形式の線(3D)です。
    /// <code>
    /// Line3: Origin + Direction * t
    /// </code>
    /// Origin は起点で Direction は線の方向です。
    /// Direction は正規化されていなければなりません。
    /// </summary>
    public class Line3 : IEquatable<Line3>
    {
        /// <summary>
        /// </summary>
        public Line3()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Line3(Line3 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public Line3(float[] source, bool isNormalized)
        {
            this.Set(source, isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Line3(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        /// <param name="isNormalized"></param>
        public Line3(Vector3 origin, Vector3 direction, bool isNormalized)
        {
            this.Set(origin, direction, isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        public Line3(Vector3 origin, Vector3 direction)
        {
            this.Set(origin, direction);
        }

        /// <summary>
        /// </summary>
        public Vector3 Origin { get; private set; }

        /// <summary>
        /// </summary>
        public Vector3 Direction { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        /// <param name="isNormalized"></param>
        public void Set(Vector3 origin, Vector3 direction, bool isNormalized)
        {
            Ensure.Argument.NotNull(origin);
            Ensure.Argument.NotNull(direction);

            this.Origin = origin;
            this.Direction = direction;
            if (!isNormalized)
            {
                this.Normalize();
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        public void Set(Vector3 origin, Vector3 direction)
        {
            this.Set(origin, direction, false);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public void Set(float[] source, bool isNormalized)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 6);

            this.Set(
                new Vector3(source[0], source[1], source[2]), // origin
                new Vector3(source[3], source[4], source[5]), // direction
                isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            this.Set(source, false);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Line3 source)
        {
            this.Set(source.Origin, source.Direction, true);
        }

        /// <summary>
        /// 線分を、線分の始点を始点とする線に変換する。
        /// </summary>
        /// <param name="source">線分です。</param>
        public void Set(Segment3 source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.Point0, source.Point1 - source.Point0);
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Line3);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Line3 other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Origin == other.Origin && this.Direction == other.Direction;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Origin.GetHashCode() ^ this.Direction.GetHashCode();
        }

        /// <summary>
        /// </summary>
        private void Normalize()
        {
            this.Direction = Vector3.Normalize(this.Direction);
        }
    }

    /// <summary>
    /// 線分のパラメトリック表現です。
    /// <code>
    /// Segment3: (1 - t) * Point0 + t * Point1 (ただし、0 ≦ t ≦ 1)
    /// </code>
    /// Point0とPoint1をつなぐ線分です。
    /// </summary>
    public class Segment3 : IEquatable<Segment3>
    {
        /// <summary>
        /// </summary>
        public Segment3()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Segment3(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="point0"></param>
        /// <param name="point1"></param>
        public Segment3(Vector3 point0, Vector3 point1)
        {
            this.Set(point0, point1);
        }

        /// <summary>
        /// </summary>
        public Vector3 Point0 { get; set; }

        /// <summary>
        /// </summary>
        public Vector3 Point1 { get; set; }

        /// <summary>
        /// </summary>
        /// <param name="point0"></param>
        /// <param name="point1"></param>
        public void Set(Vector3 point0, Vector3 point1)
        {
            this.Point0 = point0;
            this.Point1 = point1;
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 6);

            this.Set(
                new Vector3(source[0], source[1], source[2]),   // point0
                new Vector3(source[3], source[4], source[5]));  // point1
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Segment3);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Segment3 other)
        {
            if (other == null)
            {
                return true;
            }

            return this.Point0 == other.Point0 && this.Point1 == other.Point1;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Point0.GetHashCode() ^ this.Point1.GetHashCode();
        }
    }

    /// <summary>
    /// 視錐体の表現です。
    /// 主に課リングに使用します。
    /// </summary>
    public class Frustum
    {
        /// <summary>
        /// </summary>
        private System.Collections.Generic.List<Plane> ClipPlanes_ = new System.Collections.Generic.List<Plane>();

        //// ワールド座標系用のデータメンバ

        /// <summary>
        /// </summary>
        /// <remarks>
        /// フラスタムを包含するAABBで粗い判定に用いる
        /// </remarks>
        private AABB Box = new AABB();

        /// <summary>
        /// </summary>
        /// <remarks>
        /// left, right, near, far, up, downの順(Optimized View Frustum Culling Algorithmより)
        /// </remarks>
        private Plane[] Planes = new Plane[6];

        /// <summary>
        /// </summary>
        public Frustum()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="fovy"></param>
        /// <param name="aspect"></param>
        /// <param name="znear"></param>
        /// <param name="zfar"></param>
        /// <param name="camera"></param>
        public Frustum(float fovy, float aspect, float znear, float zfar, Matrix34 camera)
        {
            this.Set(fovy, aspect, znear, zfar, camera);
        }

        /// <summary>
        /// </summary>
        /// <param name="top"></param>
        /// <param name="bottom"></param>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <param name="znear"></param>
        /// <param name="zfar"></param>
        /// <param name="camera"></param>
        public Frustum(float top, float bottom, float left, float right, float znear, float zfar, Matrix34 camera)
        {
            this.Set(top, bottom, left, right, znear, zfar, camera);
        }

        /// <summary>
        /// </summary>
        public System.Collections.Generic.List<Plane> ClipPlanes
        {
            get { return this.ClipPlanes_; }
        }

        /// <summary>
        /// </summary>
        private Matrix34 Camera { get; set; }

        /// <summary>
        /// </summary>
        private Plane Top { get; set; }

        /// <summary>
        /// </summary>
        private Plane Bottom { get; set; }

        /// <summary>
        /// </summary>
        private Plane Left { get; set; }

        /// <summary>
        /// </summary>
        private Plane Right { get; set; }

        /// <summary>
        /// </summary>
        private float Near { get; set; }

        /// <summary>
        /// </summary>
        private float Far { get; set; }

        /// <summary>
        /// </summary>
        /// <param name="fovy"></param>
        /// <param name="aspect"></param>
        /// <param name="n"></param>
        /// <param name="f"></param>
        /// <param name="camera"></param>
        public void Set(float fovy, float aspect, float n, float f, Matrix34 camera)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// </summary>
        /// <param name="top"></param>
        /// <param name="bottom"></param>
        /// <param name="left"></param>
        /// <param name="right"></param>
        /// <param name="znear"></param>
        /// <param name="zfar"></param>
        /// <param name="camera"></param>
        public void Set(float top, float bottom, float left, float right, float znear, float zfar, Matrix34 camera)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// </summary>
        /// <remarks>
        /// Sphereはワールド座標系
        /// </remarks>
        /// <param name="sphere">球です。</param>
        /// <returns></returns>
        public bool Intersection(Sphere sphere)
        {
            // GPG p.406(jp)
            float dist;
            Vector3 viewPos = new Vector3();

            // 球の中心のZ座標をビュー座標系に変換する
            viewPos.Z = (this.Camera.M20 * sphere.Center.X) +
                        (this.Camera.M21 * sphere.Center.Y) +
                        (this.Camera.M22 * sphere.Center.Z) +
                        this.Camera.M23;

            // ニアプレーンの前かどうかを判定
            if (viewPos.Z - sphere.Radius > this.Near)
            {
                return false;
            }

            // ファープレーンの後ろかどうかを判定
            if (viewPos.Z + sphere.Radius < this.Far)
            {
                return false;
            }

            // 球の中心のX座標をビュー座標系に変換する
            viewPos.X = (this.Camera.M00 * sphere.Center.X) +
                        (this.Camera.M01 * sphere.Center.Y) +
                        (this.Camera.M02 * sphere.Center.Z) +
                        this.Camera.M03;

            // 左クリップ平面とテスト
            // 左クリップ平面は原点を通るのでN.d = 0で、y軸に平行なのでN.y = 0である。
            dist = (viewPos.X * this.Left.Normal.X) + (viewPos.Z * this.Left.Normal.Z);
            if (dist > sphere.Radius)
            {
                return false;
            }

            // 右クリップ平面とテスト
            // 右クリップ平面は原点を通るのでN.d = 0で、y軸に平行なのでN.y = 0である。
            dist = (viewPos.X * this.Right.Normal.X) + (viewPos.Z * this.Right.Normal.Z);
            if (dist > sphere.Radius)
            {
                return false;
            }

            // 球の中心のY座標をビュー座標系に変換する
            viewPos.Y = (this.Camera.M10 * sphere.Center.X) +
                        (this.Camera.M11 * sphere.Center.Y) +
                        (this.Camera.M12 * sphere.Center.Z) +
                        this.Camera.M13;

            // 上クリップ平面とテスト
            // 上クリップ平面は原点を通るのでN.d = 0で、x軸に平行なのでN.x = 0である。
            dist = (viewPos.Y * this.Top.Normal.Y) + (viewPos.Z * this.Top.Normal.Z);
            if (dist > sphere.Radius)
            {
                return false;
            }

            // 下クリップ平面とテスト
            // 下クリップ平面は原点を通るのでN.d = 0で、x軸に平行なのでN.x = 0である。
            dist = (viewPos.Y * this.Bottom.Normal.Y) + (viewPos.Z * this.Bottom.Normal.Z);
            if (dist > sphere.Radius)
            {
                return false;
            }

            // ClipPlanesの考慮。
            // 球の一部または全部がフラスタム内にある
            return true;
        }

        /// <summary>
        /// </summary>
        /// <param name="aabb"></param>
        /// <returns></returns>
        /// <remarks>
        /// AABBはワールド座標系
        /// </remarks>
        public bool Intersection(AABB aabb)
        {
            // http://www.lighthouse3d.com/opengl/viewfrustum/

            // フラスタムを包むAABBでフィルタリング
            if (!this.Box.Intersection(aabb))
            {
                return false;
            }

            var p = new Vector3();
            foreach (var plane in this.Planes)
            {
                // フラスタムの各面に対して
                // Test: ax + by + cz + dを最小にする点を決める。
                // これはa,b,cの符号を見ることで決められる。
                // Testが正ならAABBの全ての点はフラスタムの外ということになる。
                p.X = (plane.Normal.X >= 0) ? aabb.Minimum.X : aabb.Maximum.X;
                p.Y = (plane.Normal.Y >= 0) ? aabb.Minimum.Y : aabb.Maximum.Y;
                p.Z = (plane.Normal.Z >= 0) ? aabb.Minimum.Z : aabb.Maximum.Z;

                // フラスタムへの最近点が外ならばfalse
                if (plane.Test(p) > 0)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// </summary>
        /// <param name="box"></param>
        /// <returns></returns>
        /// <remarks>
        /// AABBはワールド座標系
        /// </remarks>
        public IntersectionResult IntersectEx(AABB box)
        {
            Ensure.Argument.NotNull(box);

            // http://www.lighthouse3d.com/opengl/viewfrustum/

            // フラスタムを包むAABBでフィルタリング
            if (!this.Box.Intersection(box))
            {
                return IntersectionResult.Outside;
            }

            IntersectionResult result = IntersectionResult.Inside;

            var p = new Vector3();
            var n = new Vector3();
            foreach (var plane in this.Planes)
            {
                if (plane.Normal.X >= 0)
                {
                    p.X = box.Minimum.X;
                    n.X = box.Maximum.X;
                }
                else
                {
                    p.X = box.Maximum.X;
                    n.X = box.Minimum.X;
                }

                if (plane.Normal.Y >= 0)
                {
                    p.Y = box.Minimum.Y;
                    n.Y = box.Maximum.Y;
                }
                else
                {
                    p.Y = box.Maximum.Y;
                    n.Y = box.Minimum.Y;
                }

                if (plane.Normal.Z >= 0)
                {
                    p.Z = box.Minimum.Z;
                    n.Z = box.Maximum.Z;
                }
                else
                {
                    p.Z = box.Maximum.Z;
                    n.Z = box.Minimum.Z;
                }

                // フラスタムへの最近点が外ならばoutside
                if (plane.Test(p) > 0)
                {
                    return IntersectionResult.Outside;
                }

                if (plane.Test(n) > 0)
                {
                    result = IntersectionResult.Intersect;
                }
            }

            return result;
        }
    }

    /// <summary>
    /// R: P + td (ただし、t >= 0)
    /// 起点からdの方向だけに向かう線である。Pが起点でdが線の方向である。
    /// 使用時にdは正規化されていることが前提となっている。
    /// </summary>
    public class Ray3 : IEquatable<Ray3>
    {
        /// <summary>
        /// </summary>
        public Ray3()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Ray3(Ray3 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public Ray3(float[] source, bool isNormalized)
        {
            this.Set(source, isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        /// <param name="isNormalized"></param>
        public Ray3(Vector3 origin, Vector3 direction, bool isNormalized)
        {
            this.Set(origin, direction, isNormalized);
        }

        /// <summary>
        /// </summary>
        public Vector3 Origin { get; private set; }

        /// <summary>
        /// </summary>
        public Vector3 Direction { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        /// <param name="isNormalized"></param>
        public void Set(Vector3 origin, Vector3 direction, bool isNormalized)
        {
            this.Origin = origin;
            this.Direction = direction;
            if (!isNormalized)
            {
                this.Normalize();
            }
        }

        /// <summary>
        /// </summary>
        /// <param name="origin"></param>
        /// <param name="direction"></param>
        public void Set(Vector3 origin, Vector3 direction)
        {
            this.Set(origin, direction, false);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        /// <param name="isNormalized"></param>
        public void Set(float[] source, bool isNormalized)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 6);

            this.Set(
                new Vector3(source[0], source[1], source[2]), // Origin
                new Vector3(source[3], source[4], source[5]), // Direction
                isNormalized);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            this.Set(source, false);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Ray3 source)
        {
            Ensure.Argument.NotNull(source);
            this.Set(source.Origin, source.Direction, true);
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Ray3);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Ray3 other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Origin == other.Origin && this.Direction == other.Direction;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Origin.GetHashCode() ^ this.Direction.GetHashCode();
        }

        /// <summary>
        /// </summary>
        private void Normalize()
        {
            this.Direction = Vector3.Normalize(this.Direction);
        }
    }

    /// <summary>
    /// 線分と線分に対する距離でカプセルが定義される。
    /// </summary>
    public class Capsule : IEquatable<Capsule>
    {
        /// <summary>
        /// </summary>
        public Capsule()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Capsule(Capsule source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Capsule(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="segment">線分です。</param>
        /// <param name="radius">半径です。</param>
        public Capsule(Segment3 segment, float radius)
        {
            this.Set(segment, radius);
        }

        /// <summary>
        /// </summary>
        /// <param name="p0"></param>
        /// <param name="p1"></param>
        /// <param name="radius">半径です。</param>
        public Capsule(Vector3 p0, Vector3 p1, float radius)
        {
            this.Set(p0, p1, radius);
        }

        /// <summary>
        /// </summary>
        public Segment3 Segment { get; private set; }

        /// <summary>
        /// </summary>
        public float Radius { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Capsule source)
        {
            Ensure.Argument.NotNull(source);

            this.Segment = source.Segment;
            this.Radius = source.Radius;
        }

        /// <summary>
        /// </summary>
        /// <param name="segment">線分です。</param>
        /// <param name="radius">半径です。</param>
        public void Set(Segment3 segment, float radius)
        {
            Ensure.Argument.NotNull(segment);
            Ensure.Argument.NotNull(radius);

            this.Segment = segment;
            this.Radius = radius;
        }

        /// <summary>
        /// </summary>
        /// <param name="p0"></param>
        /// <param name="p1"></param>
        /// <param name="radius">半径です。</param>
        public void Set(Vector3 p0, Vector3 p1, float radius)
        {
            Ensure.Argument.NotNull(p0);
            Ensure.Argument.NotNull(p1);
            this.Set(new Segment3(p0, p1), radius);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 7);

            this.Set(
                new Vector3(source[0], source[1], source[2]),
                new Vector3(source[3], source[4], source[5]),
                source[6]);
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Capsule);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Capsule other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Segment == other.Segment && this.Radius == other.Radius;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Segment.GetHashCode() ^ this.Radius.GetHashCode();
        }
    }

    /// <summary>
    /// 半径rで原点からZ軸にhだけの高さを持つ円柱と解釈される。
    /// lineやplaneとの交差判定は座標変換してから行う必要がある。
    /// </summary>
    public class Cylinder : IEquatable<Cylinder>
    {
        /// <summary>
        /// </summary>
        public Cylinder()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Cylinder(Cylinder source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Cylinder(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="radius">半径です。</param>
        /// <param name="height"></param>
        public Cylinder(float radius, float height)
        {
            this.Set(radius, height);
        }

        /// <summary>
        /// </summary>
        public float Radius { get; private set; }

        /// <summary>
        /// </summary>
        public float Height { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Cylinder source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.Radius, source.Height);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 2);
        }

        /// <summary>
        /// </summary>
        /// <param name="radius">半径です。</param>
        /// <param name="height"></param>
        public void Set(float radius, float height)
        {
            this.Radius = radius;
            this.Height = height;
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Cylinder);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Cylinder other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Radius == other.Radius && this.Height == other.Height;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Radius.GetHashCode() ^ this.Height.GetHashCode();
        }
    }

    /// <summary>
    /// 半径rで原点からZ軸にhだけの高さを持つ円錐と解釈される。
    /// lineやplaneとの交差判定は座標変換してから行う必要がある。
    /// </summary>
    public class Cone
    {
        /// <summary>
        /// </summary>
        public Cone()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Cone(Cone source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public Cone(float[] source)
        {
            this.Set(source);
        }

        /// <summary>
        /// </summary>
        /// <param name="radius">半径です。</param>
        /// <param name="height"></param>
        public Cone(float radius, float height)
        {
            this.Set(radius, height);
        }

        /// <summary>
        /// </summary>
        public float Radius { get; private set; }

        /// <summary>
        /// </summary>
        public float Height { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(Cone source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.Radius, source.Height);
        }

        /// <summary>
        /// </summary>
        /// <param name="source"></param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(source.Length == 2);

            this.Set(source[0], source[1]);
        }

        /// <summary>
        /// </summary>
        /// <param name="radius">半径です。</param>
        /// <param name="height"></param>
        public void Set(float radius, float height)
        {
            this.Radius = radius;
            this.Height = height;
        }
    }

    /// <summary>
    /// </summary>
    public class Polyline : IEquatable<Polyline>
    {
        /// <summary>
        /// </summary>
        public Polyline()
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="points"></param>
        public Polyline(Vector3[] points)
        {
            this.Set(points);
        }

        /// <summary>
        /// </summary>
        public Vector3[] Points { get; private set; }

        /// <summary>
        /// </summary>
        /// <param name="points"></param>
        public void Set(Vector3[] points)
        {
            Ensure.Argument.NotNull(points);
            Ensure.Argument.True(points.Length >= 2);

            this.Points = points;
        }

        /// <summary>
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            return this.Equals(obj as Polyline);
        }

        /// <summary>
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(Polyline other)
        {
            if (other == null)
            {
                return false;
            }

            return this.Points == other.Points;
        }

        /// <summary>
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return this.Points.GetHashCode();
        }
    }

    /// <summary>
    /// 形状間のテストを行います。
    /// </summary>
    public class GeometryTester
    {
        /// <summary>
        /// 点と線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="line">線です。</param>
        /// <returns>点から線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Line3 line)
        {
            float t;
            return DistanceSquared(point, line, out t);
        }

        /// <summary>
        /// 点と線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="line">線です。</param>
        /// <param name="t">線上に存在する point に最も近い点を、line.Origin + t * line.Direction としたときの t が格納されます。</param>
        /// <returns>点から線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Line3 line, out float t)
        {
            Ensure.Argument.NotNull(point);
            Ensure.Argument.NotNull(line);

            //// GTCG p.366

            Vector3 lp = point - line.Origin;

            t = line.Direction.Dot(lp);

            Vector3 pp = line.Origin + (line.Direction * t);
            Vector3 ppmp = point - pp;

            float distSq = ppmp.LengthSquared;

            return distSq;
        }

        /// <summary>
        /// 点と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="ray">射線です。</param>
        /// <returns>点から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Ray3 ray)
        {
            float t;
            return DistanceSquared(point, ray, out t);
        }

        /// <summary>
        /// 点と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="ray">射線です。</param>
        /// <param name="t">射線上に存在する point に最も近い点を、ray.Origin + t * ray.Direction としたときの t が格納されます。</param>
        /// <returns>点から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Ray3 ray, out float t)
        {
            Ensure.Argument.NotNull(point);
            Ensure.Argument.NotNull(ray);

            // GTCG p.368
            float distSq = DistanceSquared(point, new Line3(ray.Origin, ray.Direction, true), out t);

            if (t < 0)
            {
                t = 0;

                Vector3 vec = point - ray.Origin;
                distSq = vec.LengthSquared;
            }

            return distSq;
        }

        /// <summary>
        /// 点と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="segment">線分です。</param>
        /// <returns>点から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Segment3 segment)
        {
            float t = 0f;
            return DistanceSquared(point, segment, out t);
        }

        /// <summary>
        /// 点と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="segment">線分です。</param>
        /// <param name="t">線分上に存在する point に最も近い点を、
        /// segment.Point0 + t * (segment.Point1 - segment.Point0) / | segment.Point1 - segment.Point0 | としたときの t が格納されます。</param>
        /// <returns>点から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Segment3 segment, out float t)
        {
            Ensure.Argument.NotNull(point);
            Ensure.Argument.NotNull(segment);

            //// GTCG p.368

            var line = new Line3();
            line.Set(segment);

            float dist01 = (segment.Point0 - segment.Point1).Length;
            float distSq = DistanceSquared(point, line, out t);

            if (t < 0f)
            {
                t = 0f;

                Vector3 vec = point - segment.Point0;
                distSq = vec.LengthSquared;
            }
            else if (t > dist01)
            {
                t = 1f;

                Vector3 vec = point - segment.Point1;
                distSq = vec.LengthSquared;
            }
            else
            {
                t = t / dist01;
            }

            return distSq;
        }

        /// <summary>
        /// 点から平面への距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="plane">平面です。</param>
        /// <returns>点から平面への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Plane plane)
        {
            Vector3 q;
            return DistanceSquared(point, plane, out q);
        }

        /// <summary>
        /// 点から平面への距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="plane">平面です。</param>
        /// <param name="q">点に最も近い平面上の点が格納されます。</param>
        /// <returns>点から平面への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Plane plane, out Vector3 q)
        {
            Ensure.Argument.NotNull(point);
            Ensure.Argument.NotNull(plane);

            // GTCG p.374
            // 平面の法線ベクトルは正規化されていることが前提になっています。
            float k = plane.Test(point);

            q = point - (plane.Normal * k);

            return k * k;
        }

        /// <summary>
        /// 球と平面の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="sphere">球です。</param>
        /// <param name="plane">平面です。</param>
        /// <returns>球から平面への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Sphere sphere, Plane plane)
        {
            Ensure.Argument.NotNull(sphere);
            Ensure.Argument.NotNull(plane);

            // EMG p.537
            float distance = plane.Test(sphere.Center);
            if (distance > sphere.Radius)
            {
                return (distance - sphere.Radius) * (distance - sphere.Radius);
            }
            else if (distance < -sphere.Radius)
            {
                return (distance + sphere.Radius) * (distance + sphere.Radius);
            }
            else
            {
                return 0f;
            }
        }

        /// <summary>
        /// 点から連続する線分の距離の 2 乗を求めます。
        /// DistanceSquared(Vector3, Polyline)を繰り返し使用するより高速です。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="polyline">連続する線分です。</param>
        /// <returns>点から連続する線分の距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, Polyline polyline)
        {
            Ensure.Argument.NotNull(point);
            Ensure.Argument.NotNull(polyline);

            //// GTCG p.371

            Vector3[] points = polyline.Points;

            int numSegments = points.Length - 1;

            //// ポイントからセグメント開始点へのベクトル
            float xMinusA, yMinusB, zMinusC;
            //// ポイントからセグメント終了点へのベクトル
            float xNextMinusA, yNextMinusB, zNextMinusC;

            xMinusA = points[0].X - point.X;
            yMinusB = points[0].Y - point.Y;
            zMinusC = points[0].Z - point.Z;

            xNextMinusA = points[1].X - point.X;
            yNextMinusB = points[1].Y - point.Y;
            zNextMinusC = points[1].Z - point.Z;

            Segment3 segment = new Segment3(points[0], points[1]);
            float distSq = DistanceSquared(point, segment);

            // 続くセグメントは出来る限りリジェクトする。
            // リジェクトできなければ距離を計算
            for (int i = 1; i < numSegments; ++i)
            {
                xMinusA = xNextMinusA;
                yMinusB = yNextMinusB;
                zMinusC = zNextMinusC;

                xNextMinusA = points[i + 1].X - point.X;
                yNextMinusB = points[i + 1].Y - point.Y;
                zNextMinusC = points[i + 1].Z - point.Z;

                // Rejection Test
                if (((xMinusA * xMinusA >= distSq) &&
                     (xNextMinusA * xNextMinusA <= distSq) &&
                     (xMinusA * xNextMinusA > 0)) ||
                    ((yMinusB * yMinusB >= distSq) &&
                     (yNextMinusB * yNextMinusB <= distSq) &&
                     (yMinusB * yNextMinusB > 0)) ||
                    ((zMinusC * zMinusC >= distSq) &&
                     (zNextMinusC * zNextMinusC <= distSq) &&
                     (zMinusC * zNextMinusC > 0)))
                {
                    // リジェクト
                    continue;
                }

                segment.Set(points[i], points[i + 1]);
                float distSq_ = DistanceSquared(point, segment);

                if (distSq_ < distSq)
                {
                    distSq = distSq_;
                }
            }

            return distSq;
        }

        /// <summary>
        /// 点から AABB への距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="box">AABBです。</param>
        /// <returns>点から AABB への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, AABB box)
        {
            Vector3 q;
            return DistanceSquared(point, box, out q);
        }

        /// <summary>
        /// 点から AABB への距離の 2 乗を求めます。
        /// </summary>
        /// <param name="point">点です。</param>
        /// <param name="box">AABBです。</param>
        /// <param name="q">点に最も近い AABB 上の点が格納されます。</param>
        /// <returns>点から AABB への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Vector3 point, AABB box, out Vector3 q)
        {
            //// RCD p.130 - p.131

            float distSq = 0f;
            float v;

            float x = v = point.X;
            if (v < box.Minimum.X)
            {
                distSq += (box.Minimum.X - v) * (box.Minimum.X - v);
                x = box.Minimum.X;
            }
            else if (v > box.Maximum.X)
            {
                distSq += (box.Maximum.X - v) * (box.Maximum.X - v);
                x = box.Maximum.X;
            }

            float y = v = point.Y;
            if (v < box.Minimum.Y)
            {
                distSq += (box.Minimum.Y - v) * (box.Minimum.Y - v);
                y = box.Minimum.Y;
            }
            else if (v > box.Maximum.Y)
            {
                distSq += (box.Maximum.Y - v) * (box.Maximum.Y - v);
                y = box.Maximum.Y;
            }

            float z = v = point.Z;
            if (v < box.Minimum.Z)
            {
                distSq += (box.Minimum.Z - v) * (box.Minimum.Z - v);
                z = box.Minimum.Z;
            }
            else if (v > box.Maximum.Z)
            {
                distSq += (box.Maximum.Z - v) * (box.Maximum.Z - v);
                z = box.Maximum.Z;
            }

            q = new Vector3(x, y, z);

            return distSq;
        }

        /// <summary>
        /// 線と線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line0">線です。</param>
        /// <param name="line1">線です。</param>
        /// <returns>線から線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line0, Line3 line1)
        {
            float s, t;
            return DistanceSquared(line0, line1, out s, out t);
        }

        /// <summary>
        /// 線と線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line0">線です。</param>
        /// <param name="line1">線です。</param>
        /// <param name="s">線 line0 上に存在する line1 に最も近い点を、line0.Origin + s * line0.Direction としたときの s が格納されます。</param>
        /// <param name="t">線 line1 上に存在する line0 に最も近い点を、line1.Origin + t * line1.Direction としたときの t が格納されます。</param>
        /// <returns>線から線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line0, Line3 line1, out float s, out float t)
        {
            //// Based on article and code by Dan Sunday at http://www.geometryalgorithms.com/

            //// 線の方向ベクトルは正規化されていることが前提になっています。

            Vector3 u = line0.Origin - line1.Origin;
            //// float a = 1.0f; // a = line0.Direction.LengthSquared;
            float b = line0.Direction.Dot(line1.Direction);
            //// float c = 1.0f; // c = line1.Direction.LengthSquared;
            float d = line0.Direction.Dot(u);
            float e = line1.Direction.Dot(u);
            float f = u.LengthSquared;
            double det = 1.0 - (b * b); // det = a * c - b * b;

            // 平行かどうかをチェック
            if (det < float.Epsilon)
            {
                // 平行な場合はL0の起点から算出
                s = 0f;
                t = e;

                return f;
            }
            else
            {
                double invDet = 1f / det;

                s = (float)(((b * e) - d) * invDet);
                t = (float)((e - (b * d)) * invDet);

                Vector3 tmp0 = line0.Direction * s;
                Vector3 tmp1 = line1.Direction * t;
                Vector3 tmp2 = tmp0 - tmp1;
                tmp2 = tmp2 + u;

                return tmp2.LengthSquared;
            }
        }

        /// <summary>
        /// 線分から線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="segment0">線分です。</param>
        /// <param name="segment1">線分です。</param>
        /// <returns>線分から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Segment3 segment0, Segment3 segment1)
        {
            float s, t;
            return DistanceSquared(segment0, segment1, out s, out t);
        }

        /// <summary>
        /// 線分から線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="segment0">線分です。</param>
        /// <param name="segment1">線分です。</param>
        /// <param name="s">線分 segment0 上に存在する segment1 に最も近い点の、segment0.Point0 からの距離 s が格納されます。</param>
        /// <param name="t">線分 segment1 上に存在する segment0 に最も近い点の、segment1.Point0 からの距離 t が格納されます。</param>
        /// <returns>線分から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Segment3 segment0, Segment3 segment1, out float s, out float t)
        {
            //// Based on article and code by Dan Sunday at http://www.geometryalgorithms.com/

            Vector3 u = segment0.Point1 - segment0.Point0;
            Vector3 v = segment1.Point1 - segment1.Point0;
            Vector3 w = segment0.Point0 - segment1.Point0;
            float a = u.LengthSquared;    // always >= 0
            float b = u.Dot(v);
            float c = v.LengthSquared;    // always >= 0
            float d = u.Dot(w);
            float e = v.Dot(w);
            float det = (a * c) - (b * b);  // always >= 0
            float sc, sN, sD = det;         // sc = sN / sD, default sD = D >= 0
            float tc, tN, tD = det;         // tc = tN / tD, default tD = D >= 0

            // 線分を線とした場合の最近点を求める
            if (det < float.Epsilon)
            {                               // 2つの線分は平行
                sN = 0.0f;                  // S1の始点とS2の距離をとる
                sD = 1.0f;                  // 後のコードの除算対策
                tN = e;
                tD = c;
            }
            else
            {                               // 線分でなく線とした場合最近点を計算
                sN = (b * e) - (c * d);
                tN = (a * e) - (b * d);

                if (sN < 0.0f)
                {                           // sc < 0ならば、s=0の辺が「見える」
                    sN = 0.0f;
                    tN = e;
                    tD = c;
                }
                else if (sN > sD)
                {                           // sc > 1ならば、s=1の辺が「見える」
                    sN = sD;
                    tN = e + b;
                    tD = c;
                }
            }

            if (tN < 0.0f)
            {                               // tc < 0ならば、t=0の辺が「見える」
                tN = 0.0f;

                // tc = 0としてscの最近点を再計算
                if (-d < 0.0f)
                {
                    sN = 0.0f;
                }
                else if (-d > a)
                {
                    sN = sD;
                }
                else
                {
                    sN = -d;
                    sD = a;
                }
            }
            else if (tN > tD)
            {                               // tc > 1ならば、t=1の辺が「見える」
                tN = tD;

                // tc = 1としてscの最近点を再計算
                if ((-d + b) < 0.0f)
                {
                    sN = 0.0f;
                }
                else if ((-d + b) > a)
                {
                    sN = sD;
                }
                else
                {
                    sN = -d + b;
                    sD = a;
                }
            }

            // finally do the division to get sc and tc
            sc = Math.Abs(sN) < float.Epsilon ? 0.0f : sN / sD;
            tc = Math.Abs(tN) < float.Epsilon ? 0.0f : tN / tD;

            s = sc;
            t = tc;

            // 最近点同士の差をとる
            Vector3 dP = w + (u * sc) - (v * tc);  // = S1(sc) - S2(tc)

            return dP.LengthSquared;
        }

        /// <summary>
        /// 線と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line">線です。</param>
        /// <param name="ray">射線です。</param>
        /// <returns>線から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line, Ray3 ray)
        {
            float s, t;
            return DistanceSquared(line, ray, out s, out t);
        }

        /// <summary>
        /// 線と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line">線です。</param>
        /// <param name="ray">射線です。</param>
        /// <param name="s">線 line 上に存在する ray に最も近い点を、line.Origin + s * line.Direction としたときの s が格納されます。</param>
        /// <param name="t">射線 ray 上に存在する line に最も近い点を、ray.Origin + t * ray.Direction としたときの t が格納されます。</param>
        /// <returns>線から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line, Ray3 ray, out float s, out float t)
        {
            //// GTCG p.419

            Vector3 u = line.Origin - ray.Origin;
            //// float a = 1.0f; // a = line.Direction.LengthSquared;
            float b = line.Direction.Dot(ray.Direction);
            //// float c = 1.0f; // c = ray.Direction.LengthSquared;
            float d = line.Direction.Dot(u);
            float e = ray.Direction.Dot(u);

            double det = 1.0f - (b * b); // det = (a * c) - (b * b);

            float snum, tnum, sdenom, tdenom;
            if (det < float.Epsilon)
            {
                // 適当に決める
                snum = 0.0f;
                tnum = e;
                tdenom = 1.0f;
                sdenom = (float)det;
            }
            else
            {
                snum = (b * e) - d;
                tnum = e - (b * d);
                sdenom = tdenom = (float)det;
            }

            // t(Rayのパラメータ)が0以下の場合をチェック
            if (tnum < 0)
            {
                // Rayの起点とそれに一番近いLine上の点の組み合わせ
                tnum = 0;
                snum = -d;
                sdenom = 1.0f;
            }

            s = snum / sdenom;
            t = tnum / tdenom;

            Vector3 v = (line.Origin + (line.Direction * s)) - (ray.Origin + (ray.Direction * t));

            return v.LengthSquared;
        }

        /// <summary>
        /// 線と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line">線です。</param>
        /// <param name="segment">線分です。</param>
        /// <returns>線から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line, Segment3 segment)
        {
            float s, t;
            return DistanceSquared(line, segment, out s, out t);
        }

        /// <summary>
        /// 線と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="line">線です。</param>
        /// <param name="segment">線分です。</param>
        /// <param name="s">線 line 上に存在する segment に最も近い点を、line.Origin + s * line.Direction としたときの s が格納されます。</param>
        /// <param name="t">線分 segment 上に存在する line に最も近い点の segment.Point0 からの距離 t が格納されます。</param>
        /// <returns>線から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Line3 line, Segment3 segment, out float s, out float t)
        {
            //// GTCG p.421

            Vector3 segDir = segment.Point1 - segment.Point0;
            Vector3 u = line.Origin - segment.Point0;
            //// float a = 1.0f; // a = line.Direction.LengthSquared;
            float b = line.Direction.Dot(segDir);
            float c = segDir.LengthSquared;
            float d = line.Direction.Dot(u);
            float e = segDir.Dot(u);
            double det = c - (b * b); // det = (a * c) - (b * b);

            float snum, tnum, sdenom, tdenom;

            // 平行かどうかをチェック
            if (det < float.Epsilon)
            {
                // 適当に決める
                snum = 0.0f;
                tnum = e;
                sdenom = (float)det;
                tdenom = c;
            }
            else
            {
                snum = (b * e) - (c * d);
                tnum = e - (b * d);
                sdenom = tdenom = (float)det;
            }

            // t(Segnemtのパラメータ)が0未満、1より大きい場合のチェック
            if (tnum < 0.0f)
            {
                // Segmentの始点
                tnum = 0.0f;
                snum = -d;
                sdenom = 1.0f;
            }
            else if (tnum > tdenom)
            {
                // Segmentの終点
                tnum = tdenom;
                snum = -d + b;
                sdenom = 1.0f;
            }

            s = snum / sdenom;
            t = tnum / tdenom;

            Vector3 v = (line.Origin + (line.Direction * s)) - (segment.Point0 + (segDir * t));

            return v.LengthSquared;
        }

        /// <summary>
        /// 射線と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="ray0">射線です。</param>
        /// <param name="ray1">射線です。</param>
        /// <returns>射線から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Ray3 ray0, Ray3 ray1)
        {
            float s, t;
            return DistanceSquared(ray0, ray1, out s, out t);
        }

        /// <summary>
        /// 射線と射線の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="ray0">射線です。</param>
        /// <param name="ray1">射線です。</param>
        /// <param name="s">射線 ray0 上に存在する ray1 に最も近い点を、ray0.Origin + s * ray0.Direction としたときの s が格納されます。</param>
        /// <param name="t">射線 ray1 上に存在する ray0 に最も近い点を、ray1.Origin + t * ray1.Direction としたときの t が格納されます。</param>
        /// <returns>射線から射線への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Ray3 ray0, Ray3 ray1, out float s, out float t)
        {
            Vector3 u = ray0.Origin - ray1.Origin;
            //// float a = 1.0f; // a = ray0.Direction.LengthSquared;
            float b = ray0.Direction.Dot(ray1.Direction);
            //// float c = 1.0f; // c = ray1.Direction.LengthSquared;
            float d = ray0.Direction.Dot(u);
            float e = ray1.Direction.Dot(u);
            double det = 1.0 - (b * b); // det = (a * c) - (b * b);

            float snum, tnum, sdenom, tdenom;

            // 平行かどうかをチェック
            if (det < float.Epsilon)
            {
                // 適当に決める
                snum = 0.0f;
                tnum = e;
                tdenom = 1.0f;
                sdenom = (float)det;
            }
            else
            {
                snum = (b * e) - d;
                tnum = e - (b * d);
                sdenom = tdenom = (float)det;
            }

            if (snum < 0.0f)
            {
                // Ray0を起点にする。
                snum = 0.0f;
                tnum = e;
                tdenom = 1.0f;
            }

            if (tnum < 0.0f)
            {
                // Ray1の起点
                tnum = 0.0f;
                if (-d < 0)
                {
                    // ray0, ray1の起点同士が最も近い
                    snum = 0.0f;
                }
                else
                {
                    // ray0上のray0に垂直な線分がray1の起点に到達する場合
                    snum = -d;
                    sdenom = 1.0f;
                }
            }

            s = snum / sdenom;
            t = tnum / tdenom;

            Vector3 v = (ray0.Origin + (ray0.Direction * s)) - (ray1.Origin + (ray1.Direction * t));

            return v.LengthSquared;
        }

        /// <summary>
        /// 射線と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="ray">射線です。</param>
        /// <param name="segment">線分です。</param>
        /// <returns>射線から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Ray3 ray, Segment3 segment)
        {
            float s, t;
            return DistanceSquared(ray, segment, out s, out t);
        }

        /// <summary>
        /// 射線と線分の距離の 2 乗を求めます。
        /// </summary>
        /// <param name="ray">射線です。</param>
        /// <param name="segment">線分です。</param>
        /// <param name="s">射線 ray 上に存在する segment に最も近い点を、ray.Origin + s * ray.Direction としたときの s が格納されます。</param>
        /// <param name="t">線分 segment 上に存在する ray に最も近い点の、segment.Point0 からの距離t が格納されます。</param>
        /// <returns>射線から線分への距離の 2 乗を返します。</returns>
        public static float DistanceSquared(Ray3 ray, Segment3 segment, out float s, out float t)
        {
            //// GTCG p.424

            Vector3 segmentDir = segment.Point1 - segment.Point0;
            Vector3 u = ray.Origin - segment.Point0;
            //// float a = 1.0f; // a = ray.Direction.LengthSquared;
            float b = ray.Direction.Dot(segmentDir);
            float c = segmentDir.LengthSquared;
            float d = ray.Direction.Dot(u);
            float e = segmentDir.Dot(u);
            double det = c - (b * b); // det = (a * c) - (b * b);

            float snum, tnum, sdenom, tdenom;

            // 平行かどうかをチェック
            if (det < float.Epsilon)
            {
                // 適当に決める
                snum = 0.0f;
                tnum = e;
                tdenom = c;
                sdenom = (float)det;
            }
            else
            {
                snum = (b * e) - (c * d);
                tnum = e - (b * d);
                sdenom = tdenom = (float)det;
            }

            // Rayをチェック
            if (snum < 0.0f)
            {
                // Rayの起点を使う。
                snum = 0.0f;
                tnum = e;
                tdenom = c;
            }

            // Segmentをチェック
            if (tnum < 0.0f)
            {
                // Segmentの始点
                tnum = 0.0f;
                if (-d < 0.0f)
                {
                    snum = 0.0f;
                }
                else
                {
                    // Rayの上にSegmentの始点への最近点がある場合
                    snum = -d;
                    sdenom = 1.0f;
                }
            }
            else if (tnum > tdenom)
            {
                // Segmentの終点
                tnum = tdenom;
                if ((-d + b) < 0.0f)
                {
                    snum = 0.0f;
                }
                else
                {
                    // Rayの上にSegmentの終点への最近点がある場合
                    snum = -d + b;
                    sdenom = 1.0f;
                }
            }

            s = snum / sdenom;
            t = tnum / tdenom;

            Vector3 v = (ray.Origin + (ray.Direction * s)) - (segment.Point0 + (segmentDir * t));

            return v.LengthSquared;
        }
    }
}

#if false

// 移植の完了していないルーチン

IntersectionResult IntersectionLine3Plane(const LINE3* L, const PLANE* J, f32* t, VEC3* I);
IntersectionResult IntersectionRay3Plane(const RAY3* R, const PLANE* J, f32* t, VEC3* I);
IntersectionResult IntersectionSegment3Plane(const SEGMENT3* S, const PLANE* J, f32* t, VEC3* I);

IntersectionResult IntersectionLine3Sphere(const LINE3* L, const SPHERE* sphere, f32* t0, f32* t1);
IntersectionResult IntersectionRay3Sphere(const RAY3* R, const SPHERE* sphere, f32* t0, f32* t1);
bool               IntersectionRay3Sphere(const RAY3* R, const SPHERE* sphere);
IntersectionResult IntersectionSegment3Sphere(const SEGMENT3* S, const SPHERE* sphere, f32* t0, f32* t1);

bool               IntersectionRay3AABB(const RAY3* R, const AABB* box, f32* t);
bool               IntersectionAABB(const AABB* a, const AABB* b);
bool               IntersectionSphereAABB(const SPHERE* sphere, const AABB* aabb);
bool               IntersectionSphere(const SPHERE* s0, const SPHERE* s1);
bool               IntersectionPlaneAABB(const PLANE* J, const AABB* B);

bool               IntersectionCapsule(const CAPSULE* C0, const CAPSULE* C1);
bool               IntersectionRay3Capsule(const RAY3* R, const CAPSULE* C);
bool               IntersectionLine3Capsule(const LINE3* L, const CAPSULE* C);
bool               IntersectionPlaneCapsule(const PLANE* J, const CAPSULE* C);

//
// 3Dジオメトリのマージ
//
SPHERE* MergeSphere(SPHERE* s2, const SPHERE* s0, const SPHERE* s1);
AABB* MergeAABB(AABB* a2, const AABB* a0, const AABB* a1);

#endif
