﻿// ========================================================================
// <copyright file="RgbColorBlender.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    /// <summary>
    /// RgbaColor のアルファ成分以外をブレンドするユーティリティです。
    /// </summary>
    public static class RgbColorBlender
    {
        /// <summary>
        /// 2 つのカラーを加算します。
        /// </summary>
        /// <param name="lhs">演算の左のカラーです。</param>
        /// <param name="rhs">演算の右のカラーです。</param>
        /// <returns>lhsとrhsを加算した結果のカラーです。</returns>
        public static RgbaColor Add(RgbaColor lhs, RgbaColor rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new RgbaColor(lhs.R + rhs.R, lhs.G + rhs.G, lhs.B + rhs.B, 1.0f);
        }

        /// <summary>
        /// 2 つのカラーを減算します。
        /// </summary>
        /// <param name="lhs">演算の左のカラーです。</param>
        /// <param name="rhs">演算の右のカラーです。</param>
        /// <returns>lhsとrhsを減算した結果のカラーです。</returns>
        public static RgbaColor Subtract(RgbaColor lhs, RgbaColor rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new RgbaColor(lhs.R - rhs.R, lhs.G - rhs.G, lhs.B - rhs.B, 1.0f);
        }

        /// <summary>
        /// 2 つのカラーの乗算します。
        /// </summary>
        /// <param name="lhs">演算の左のカラーです。</param>
        /// <param name="rhs">演算の右のカラーです。</param>
        /// <returns>lhsとrhsを乗算した結果のカラーです。</returns>
        public static RgbaColor Multiply(RgbaColor lhs, RgbaColor rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new RgbaColor(lhs.R * rhs.R, lhs.G * rhs.G, lhs.B * rhs.B, 1.0f);
        }

        /// <summary>
        /// カラーにスカラーを乗算します。
        /// </summary>
        /// <param name="color">演算の左のカラーです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>カラーにスカラーを乗算した結果のカラーです。</returns>
        public static RgbaColor Multiply(RgbaColor color, float scalar)
        {
            Ensure.Argument.NotNull(color);

            return new RgbaColor(
                color.R * scalar,
                color.G * scalar,
                color.B * scalar,
                1.0f);
        }
    }
}
