﻿// ========================================================================
// <copyright file="HsvaColor.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.ComponentModel;
    using System.Diagnostics;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// HSVA 成分を保持するカラーです。
    /// </summary>
    public sealed class HsvaColor : ObservableObject, IHsvaColor, IEquatable<HsvaColor>
    {
        private float h;
        private float s;
        private float v;
        private float a;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public HsvaColor()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="h">h 成分です。</param>
        /// <param name="s">s 成分です。</param>
        /// <param name="v">v 成分です。</param>
        /// <param name="a">a 成分です。</param>
        public HsvaColor(float h, float s, float v, float a)
        {
            this.Set(h, s, v, a);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定する HsvaColor です。</param>
        public HsvaColor(IHsvaColor source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定する IColor です。</param>
        public HsvaColor(IColor source)
        {
            this.Set(source);
        }

        /// <summary>
        /// H 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float H
        {
            get
            {
                return this.h;
            }

            set
            {
                this.h = value;
                OnPropertyChanged("H");
            }
        }

        /// <summary>
        /// S 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float S
        {
            get
            {
                return this.s;
            }

            set
            {
                this.s = value;
                OnPropertyChanged("S");
            }
        }

        /// <summary>
        /// V 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float V
        {
            get
            {
                return this.v;
            }

            set
            {
                this.v = value;
                OnPropertyChanged("V");
            }
        }

        /// <summary>
        /// A 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float A
        {
            get
            {
                return this.a;
            }

            set
            {
                this.a = value;
                OnPropertyChanged("A");
            }
        }

        /// <summary>
        /// R 成分を取得します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float R
        {
            get
            {
                return this.ToRgbaColor().R;
            }
        }

        /// <summary>
        /// G 成分を取得します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float G
        {
            get
            {
                return this.ToRgbaColor().G;
            }
        }

        /// <summary>
        /// B 成分を取得します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float B
        {
            get
            {
                return this.ToRgbaColor().B;
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="h">h 成分です。</param>
        /// <param name="s">s 成分です。</param>
        /// <param name="v">v 成分です。</param>
        /// <param name="a">a 成分です。</param>
        public void Set(float h, float s, float v, float a)
        {
            this.H = h;
            this.S = s;
            this.V = v;
            this.A = a;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="source">設定する値です。</param>
        public void Set(IHsvaColor source)
        {
            Ensure.Argument.NotNull(source);

            this.H = source.H;
            this.S = source.S;
            this.V = source.V;
            this.A = source.A;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="source">設定する値です。</param>
        public void Set(IColor source)
        {
            Ensure.Argument.NotNull(source);

            IHsvaColor hsvaColor = source as IHsvaColor;
            if (hsvaColor == null)
            {
                RgbaColor rgbaColor = source as RgbaColor;
                Ensure.Operation.ObjectNotNull(rgbaColor);
                hsvaColor = rgbaColor.ToHsvaColor();
            }

            this.Set(hsvaColor);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as HsvaColor);
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new HsvaColor(this);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(HsvaColor other)
        {
            return (this.H == other.H)
                && (this.S == other.S)
                && (this.V == other.V)
                && (this.A == other.A);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as HsvaColor);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return this.H.GetHashCode() ^
                   this.S.GetHashCode() ^
                   this.V.GetHashCode() ^
                   this.A.GetHashCode();
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("[{0} {1} {2} {3}]", this.H, this.S, this.V, this.A);
        }

        /// <summary>
        /// IRgbaCololor を取得します。
        /// </summary>
        /// <returns>IRgbaColor です。</returns>
        public IRgbaColor ToRgbaColor()
        {
            float h = this.H;
            float s = this.S;
            float v = this.V;

            if (s == 0.0f)
            {
                return new RgbaColor(v, v, v, this.A);
            }

            FloatUtility.Clamp(h, 0.0f, 1.0f);

            if (h == 1.0f)
            {
                h = 0.0f;
            }

            h = h * 6.0f;
            int hi = ((int)h) % 6;

            float f = h - hi;
            float p = v * (1 - s);
            float q = v * (1 - (f * s));
            float t = v * (1 - ((1 - f) * s));

            switch (hi)
            {
                case 0:
                    return new RgbaColor(v, t, p, this.A);
                case 1:
                    return new RgbaColor(q, v, p, this.A);
                case 2:
                    return new RgbaColor(p, v, t, this.A);
                case 3:
                    return new RgbaColor(p, q, v, this.A);
                case 4:
                    return new RgbaColor(t, p, v, this.A);
                case 5:
                    return new RgbaColor(v, p, q, this.A);
                default:
                    Debug.Assert(true, "Unexpected case!");
                    return new RgbaColor(0.0f, 0.0f, 0.0f, this.A);
            }
        }
    }
}
