﻿// ========================================================================
// <copyright file="XmlComplexReader.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Xml;

    /// <summary>
    /// 複数種類のリソースが格納された、XML文章を読みとるクラスです。
    /// <para>
    /// XML文章をあらわす、文字列からリソース種類を束ねたデータを表現するクラスを構築します。
    /// </para>
    /// </summary>
    public class XmlComplexReader : IXmlComplexReader
    {
        /// <summary>
        /// ContentResolverのセットです。
        /// </summary>
        private readonly IDictionary<string, IXmlContentResolver> contentResolvers =
            new Dictionary<string, IXmlContentResolver>();

        /// <summary>
        /// リゾルバが登録されているか取得します。
        /// </summary>
        /// <param name="resolver">リゾルバです。</param>
        /// <returns>登録されているかを返します。</returns>
        public bool IsContainResolver(IXmlContentResolver resolver)
        {
            Ensure.Argument.NotNull(resolver);
            return this.contentResolvers.ContainsKey(resolver.TargetContentName);
        }

        /// <summary>
        /// リゾルバを登録します。
        /// 登録できない場合は例外
        /// (System.ArgumentException, System.NotSupportedException)を送出します。
        /// </summary>
        /// <param name="resolver">リゾルバです。</param>
        public void RegisterResolver(IXmlContentResolver resolver)
        {
            Ensure.Argument.NotNull(resolver);
            this.contentResolvers.Add(resolver.TargetContentName, resolver);
        }

        /// <summary>
        /// XML文章をよみこんで、ComplexDocumentを構築します。
        /// </summary>
        /// <param name="xmlDocumentString">XML文章です。</param>
        /// <returns>XML文章に対応するデータ構造です。</returns>
        public IXmlComplexDocument Read(string xmlDocumentString)
        {
            Ensure.Argument.StringNotEmpty(xmlDocumentString);

            using (XmlTextReader reader = new XmlTextReader(new StringReader(xmlDocumentString)))
            {
                // ルート要素の読み込み
                while (reader.Read())
                {
                    if (reader.NodeType == XmlNodeType.Element)
                    {
                        break;
                    }
                }

                // 情報を取得。
                IXmlComplexDocument document = new XmlComplexDocument(reader.Name);

                // 各コンテンツの読み込み
                reader.Read();
                while (!reader.EOF)
                {
                    // element以外は処理しません。
                    if (reader.NodeType != XmlNodeType.Element)
                    {
                        reader.Read();
                        continue;
                    }

                    string name = reader.Name;
                    string subXml = reader.ReadOuterXml();

                    // コンテンツリゾルバでコンテンツを解決します。
                    IXmlContentResolver resolver = this.FindContentResolver(name, subXml);
                    IXmlContent content;
                    if (resolver != null)
                    {
                        // リゾルバでコンテンツを構築してリストに追加します。
                        content = resolver.Resolve(name, subXml);
                    }
                    else
                    {
                        // デフォルトコンテンツを作成します。
                        content = this.CreateDeafultContent(name, subXml);
                    }

                    Debug.Assert(content != null, "Unexpected null-reference.");

                    document.Contents.Add(content);
                }

                return document;
            }
        }

        /// <summary>
        /// コンテンツ単位の、部分XML文章から、部分XML文章を解釈するContentResolver
        /// を取得します。
        /// <para>
        /// 必要に応じて、派生クラスで本関数をオーバーライドして、ContentResolverを
        /// 選択する処理を実装します。
        /// </para>
        /// </summary>
        /// <param name="name">コンテンツの名前です。</param>
        /// <param name="xmlSubDocumentString">部分XML文章です。</param>
        /// <returns>部分XML文章を解釈するContentResolverです。</returns>
        protected virtual IXmlContentResolver FindContentResolver(
            string name, string xmlSubDocumentString)
        {
            Ensure.Argument.StringNotEmpty(name);
            Ensure.Argument.StringNotEmpty(xmlSubDocumentString);

            // 登録リゾルバを検索します。
            IXmlContentResolver result = null;
            if (this.contentResolvers.ContainsKey(name))
            {
                result = this.contentResolvers[name];
            }

            return result;
        }

        /// <summary>
        /// デフォルトコンテンツを生成します。
        /// </summary>
        /// <param name="name">コンテンツの名前です。</param>
        /// <param name="xmlSubDocumentString">部分XML文章です。</param>
        /// <returns>部分XML文章に対応するIXmlContentです。</returns>
        protected virtual IXmlContent CreateDeafultContent(
            string name, string xmlSubDocumentString)
        {
            Ensure.Argument.StringNotEmpty(name);
            Ensure.Argument.StringNotEmpty(xmlSubDocumentString);

            // コンテンツを作成して返します。
            IXmlContent content = new XmlContent();
            content.Xml = xmlSubDocumentString;

            return content;
        }
    }
}
