﻿// ========================================================================
// <copyright file="XmlDomContent.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System.Xml;

    /// <summary>
    /// Domツリーでデータを表現する、データ構造です。
    /// </summary>
    public class XmlDomContent : XmlContent
    {
        /// <summary>
        /// XMLDomツリーです。
        /// </summary>
        private XmlDocument xmlDocument;

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public XmlDomContent()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のインスタンスです。</param>
        public XmlDomContent(XmlDomContent source)
        {
            Ensure.Argument.NotNull(source);
            this.Set(source);
        }

        /// <summary>
        /// XMLDomツリーを設定、取得します。
        /// </summary>
        public XmlDocument XmlDocument
        {
            get
            {
                return this.xmlDocument;
            }

            set
            {
                Ensure.Argument.NotNull(value);
                this.xmlDocument = value;
            }
        }

        /// <summary>
        /// XML文章から実体を生成します。
        /// </summary>
        /// <param name="xmlString">XML文章です。</param>
        /// <returns>実体です。</returns>
        public static IXmlContent FromXml(string xmlString)
        {
            Ensure.Argument.StringNotEmpty(xmlString);

            XmlDomContent result = new XmlDomContent();

            result.Xml = xmlString;
            result.XmlDocument = new XmlDocument();
            result.XmlDocument.LoadXml(xmlString);

            return result;
        }

        /// <summary>
        /// DOMインスタンスから実体を生成します。
        /// </summary>
        /// <param name="xmlDocument">DOMインスタンスです。</param>
        /// <returns>実体です。</returns>
        public static IXmlContent FromDom(XmlDocument xmlDocument)
        {
            Ensure.Argument.NotNull(xmlDocument);

            XmlDomContent result = new XmlDomContent();

            result.XmlDocument = xmlDocument;
            result.Xml = result.XmlDocument.InnerXml;

            return result;
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(XmlDomContent source)
        {
            Ensure.Argument.NotNull(source);

            base.Set(source as IXmlContent);
            this.XmlDocument = source.XmlDocument;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override object Clone()
        {
            return new XmlDomContent(this);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return CheckClassEqualityUtil<XmlDomContent>.Equals(
                this, other, this.Equals);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            int result = base.GetHashCode();
            if (this.XmlDocument != null)
            {
                result ^= this.XmlDocument.GetHashCode();
            }

            return result;
        }

        /// <summary>
        /// XmlWriterに内容を書き出します。
        /// </summary>
        /// <param name="writer">XmlWriterです。</param>
        public override void Export(XmlWriter writer)
        {
            Ensure.Argument.NotNull(writer);

            Ensure.Operation.ObjectNotNull(this.XmlDocument);

            this.XmlDocument.WriteTo(writer);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(XmlDomContent other)
        {
            if (!base.Equals(other))
            {
                return false;
            }

            Ensure.Argument.NotNull(other);

            // target.XmlDocumentの比較を行います。
            if (other.XmlDocument != null)
            {
                return other.XmlDocument != null;
            }

            if (!other.XmlDocument.Equals(other.XmlDocument))
            {
                return false;
            }

            return true;
        }
    }
}
