﻿// ========================================================================
// <copyright file="ConverterUtil.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Conversion
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Reflection;
    using System.Xml.Serialization;
    using NintendoWare.ToolDevelopmentKit.Xml;

    /// <summary>
    /// コンバート処理のユーティリティです。
    /// </summary>
    public class ConverterUtil
    {
        /// <summary>
        /// リフレクションの列挙対象の指定に使う値です。
        /// </summary>
        public const BindingFlags PropartyBindingFlags =
            BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance;

        //-----------------------------------------------------------------
        // 型やプロパティに関する判定。
        //-----------------------------------------------------------------

        /// <summary>
        /// プリミティブ型か判定します。
        /// </summary>
        /// <param name="type">判定する型です。</param>
        /// <returns>判定結果です。プリミティブ型ならtrueが返ります。</returns>
        public static bool IsPrimitiveType(Type type)
        {
            return type.IsPrimitive || type == typeof(string) || type.IsEnum;
        }

        /// <summary>
        /// 列挙可能型か判定します。
        /// </summary>
        /// <param name="type">判定する型です。</param>
        /// <returns>判定結果です。</returns>
        public static bool IsIEnumerable(Type type)
        {
            // 文字列は特別扱い
            if (type == typeof(string))
            {
                return false;
            }

            return type.HasInterface(typeof(IEnumerable));
        }

        /// <summary>
        /// 型のアトリビュートを探します。
        /// </summary>
        /// <typeparam name="TAttributeType">アトリビュートの型です。</typeparam>
        /// <param name="type">型情報です。</param>
        /// <returns>アトリビュートです。</returns>
        public static TAttributeType FindTypeAttribute<TAttributeType>(Type type)
            where TAttributeType : class
        {
            object[] attributes = type.GetCustomAttributes(true);
            return Array.Find(attributes, (attr) => attr is TAttributeType) as TAttributeType;
        }

        /// <summary>
        /// 型のアトリビュートを探します。
        /// </summary>
        /// <typeparam name="TAttributeType">アトリビュートの型です。</typeparam>
        /// <param name="type">型情報です。</param>
        /// <returns>アトリビュートです。</returns>
        public static TAttributeType[] FindTypeAttributes<TAttributeType>(Type type)
            where TAttributeType : class
        {
            return DoFindAttriubutes<TAttributeType>(type.GetCustomAttributes(true));
        }

        /// <summary>
        /// プロパティのアトリビュートを探します。
        /// </summary>
        /// <typeparam name="TAttributeType">アトリビュートの型です。</typeparam>
        /// <param name="propInfo">プロパティ情報です。</param>
        /// <returns>アトリビュートです。</returns>
        public static TAttributeType FindPropertyAttribute<TAttributeType>(PropertyInfo propInfo)
            where TAttributeType : class
        {
            object[] attributes = propInfo.GetCustomAttributes(true);
            return Array.Find(attributes, (attr) => attr is TAttributeType) as TAttributeType;
        }

        /// <summary>
        /// プロパティのアトリビュートセットを探します。
        /// </summary>
        /// <typeparam name="TAttributeType">アトリビュートの型です。</typeparam>
        /// <param name="propInfo">プロパティ情報です。</param>
        /// <returns>アトリビュートセットです。</returns>
        public static TAttributeType[] FindPropertyAttributes<TAttributeType>(PropertyInfo propInfo)
            where TAttributeType : class
        {
            return DoFindAttriubutes<TAttributeType>(propInfo.GetCustomAttributes(true));
        }

        /// <summary>
        /// 基底プロパティを取得します。
        /// </summary>
        /// <param name="propertyInfo">調べるプロパティです。</param>
        /// <returns>基底プロパティです。</returns>
        public static PropertyInfo GetBasePropertyInfo(PropertyInfo propertyInfo)
        {
            foreach (MethodInfo methodInfo in propertyInfo.GetAccessors(true))
            {
                MethodInfo baseMethodInfo = methodInfo.GetBaseDefinition();

                // 派生していなければ、処理しません。
                if (methodInfo == baseMethodInfo)
                {
                    continue;
                }

                // 宣言元の型から、対応するプロパティを取得します。
                Type declaringType = baseMethodInfo.DeclaringType;
                foreach (PropertyInfo baseProperty in declaringType.GetProperties())
                {
                    if (Array.Exists(
                        baseProperty.GetAccessors(true), (method) => method == baseMethodInfo))
                    {
                        return baseProperty;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// 指定プロパティに対応するプロパティを指定型から取得します。
        /// TODO:インデクサの場合には、処理
        /// TODO:アトリビュートによって明示的に指定されたプロパティが無いか確認します。
        /// </summary>
        /// <param name="searchedType">検索対象クラスの型です。</param>
        /// <param name="dstPropType">変換先プロパティの型です。</param>
        /// <param name="dstPropName">変換先プロパティの名前です。</param>
        /// <returns>変換元プロパティの情報です。</returns>
        public static PropertyInfo FindSetPropertyFromType(
            Type searchedType, Type dstPropType, string dstPropName)
        {
            Predicate<PropertyInfo> match = delegate(PropertyInfo info)
            {
                Type propType = info.PropertyType;
                return (info.Name == dstPropName) && propType.IsAssignableFrom(dstPropType);
            };

            return Array.Find(searchedType.GetProperties(PropartyBindingFlags), match);
        }

        /// <summary>
        /// 型がリストを実装しているか調査します。
        /// </summary>
        /// <param name="type">調査する型です。</param>
        /// <returns>型がリストを実装していればtrueを返します。</returns>
        public static bool CheckTypeImplementsList(Type type)
        {
            while (type != null)
            {
                if (Array.Exists(type.GetInterfaces(), ConverterUtil.CheckInterfaceImplemntsList))
                {
                    return true;
                }

                type = type.BaseType;
            }

            return false;
        }

        /// <summary>
        /// 型がリストを実装しているか調査します。
        /// </summary>
        /// <remarks>
        /// 非ジェネリックListには対応していないので注意してください。
        /// </remarks>
        /// <param name="interfaceType">調査する型です。</param>
        /// <returns>型がリストを実装していればtrueを返します。</returns>
        public static bool CheckInterfaceImplemntsList(Type interfaceType)
        {
            if (interfaceType.IsGenericType &&
                interfaceType.GetGenericTypeDefinition() == typeof(IList<>))
            {
                return true;
            }

            if (interfaceType.IsArray)
            {
                return true;
            }

            if (interfaceType.GetInterface(typeof(IList<>).Name) != null)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 型がリストを実装していれば要素の型を取得します。
        /// <remarks>
        /// 非ジェネリックListには対応していないので注意してください。
        /// </remarks>
        /// </summary>
        /// <param name="type">調査する型です。</param>
        /// <returns>型がリストを実装していれば要素の型を返します。</returns>
        public static Type GetIListElementType(Type type)
        {
            if (type.IsGenericType && type.GetGenericTypeDefinition() == typeof(IList<>))
            {
                return type.GetGenericArguments()[0];
            }

            if (type.IsArray)
            {
                return type.GetElementType();
            }

            Type lsitType = type.GetInterface(typeof(IList<>).Name);
            if (lsitType != null)
            {
                return lsitType.GetGenericArguments()[0];
            }

            return null;
        }

        /// <summary>
        /// 読み取り可能プロパティを取得します。
        /// </summary>
        /// <param name="property">プロパティです。</param>
        /// <returns>読み取り可能プロパティです。</returns>
        public static PropertyInfo GetReadableProperty(PropertyInfo property)
        {
            while (property != null && !property.CanRead)
            {
                property = ConverterUtil.GetBasePropertyInfo(property);
            }

            return property;
        }

        /// <summary>
        /// 書き込み可能プロパティを取得します。
        /// </summary>
        /// <param name="property">プロパティです。</param>
        /// <returns>書き込み可能プロパティです。</returns>
        public static PropertyInfo GetWritableProperty(PropertyInfo property)
        {
            while (property != null && !property.CanWrite)
            {
                property = ConverterUtil.GetBasePropertyInfo(property);
            }

            return property;
        }

        //-----------------------------------------------------------------
        // 列挙。
        //-----------------------------------------------------------------

        /// <summary>
        /// プロパティを列挙します。
        /// </summary>
        /// <param name="targetObject">対象オブジェクトです。</param>
        /// <returns>反復子を返します。</returns>
        public static IEnumerable<PropertyInfo> EnumerateProperties(object targetObject)
        {
            return EnumerateProperties(targetObject.GetType());
        }

        /// <summary>
        /// プロパティを列挙します。
        /// </summary>
        /// <param name="type">対象オブジェクトの型です。</param>
        /// <returns>反復子を返します。</returns>
        public static IEnumerable<PropertyInfo> EnumerateProperties(Type type)
        {
            foreach (PropertyInfo dstPropInfo in GetProperties(type))
            {
                yield return dstPropInfo;
            }
        }

        /// <summary>
        /// プロパティ配列を取得します。
        /// </summary>
        /// <param name="type">対象オブジェクトの型です。</param>
        /// <returns>プロパティ配列です。</returns>
        public static PropertyInfo[] GetProperties(Type type)
        {
            return type.GetProperties(ConverterUtil.PropartyBindingFlags);
        }

        /// <summary>
        /// 名前つきプロパティの名前を取得します。
        /// </summary>
        /// <param name="propertyInfo">調査するプロパティです。</param>
        /// <returns>名前つきプロパティの名前です。</returns>
        public static string GetNamedPropaertyName(PropertyInfo propertyInfo)
        {
            NamedPropertyAttribute namedPropertyAttribute =
                ConverterUtil.FindPropertyAttribute<NamedPropertyAttribute>(propertyInfo);

            return (namedPropertyAttribute != null) ? namedPropertyAttribute.Name : null;
        }

        // ----------------------------------------------------------------
        // コンバータの取得。
        // ----------------------------------------------------------------

        /// <summary>
        /// XMLシリアライズ処理共通のコンバータを生成します。
        /// </summary>
        /// <returns>生成したコンバータです。</returns>
        public static Converter BuildXmlConverter()
        {
            Converter converter = new Converter();

            foreach (Type xmlType in XmlUtility.EnumerateXmlConcreteTypes())
            {
                ConverterUtil.RegisterTypeConverter(converter, xmlType);
            }

            return converter;
        }

        //-----------------------------------------------------------------
        // コンバートハンドラの登録処理。
        //-----------------------------------------------------------------

        /// <summary>
        /// 型に付加されたCoupledTypeAttributeから、コンバータを登録します。
        /// 型内部の型も再帰的に処理します。
        /// </summary>
        /// <param name="converter">コンバータです。</param>
        /// <param name="type">処理の起点となる型です。</param>
        public static void RegisterTypeConverterRecursively(Converter converter, Type type)
        {
            List<Type> checkedTypeSet = new List<Type>();
            DoRegisterTypeConverterRecursively(converter, type, checkedTypeSet);
        }

        /// <summary>
        /// 型に付加されたCoupledTypeAttributeから、コンバータを登録します。
        /// </summary>
        /// <param name="converter">コンバータです。</param>
        /// <param name="type">処理の起点となる型です。</param>
        public static void RegisterTypeConverter(Converter converter, Type type)
        {
            List<Type> checkedTypeSet = new List<Type>();
            DoRegisterTypeConverter(converter, type, checkedTypeSet);
        }

        /// <summary>
        /// 型ペアに対して、型コンバート処理を登録します。
        /// <para>
        /// 一つの型に複数のコンバータを登録することは不正です。
        /// </para>
        /// <remarks>
        /// (RegisterTypeConverterRecursivelyが実装されたため、廃止になる予定です。)
        /// </remarks>
        /// </summary>
        /// <param name="couple">型ペアです。</param>
        /// <param name="handler">型コンバート処理です。</param>
        /// <param name="converter">コンバータです。</param>
        public static void RegisterTypeConverter(
            TypeCouple couple, ConvertHandler handler, Converter converter)
        {
            Ensure.Argument.NotNull(couple);
            Ensure.Argument.NotNull(couple.SrcType);
            Ensure.Argument.NotNull(couple.DstType);
            Ensure.Argument.True(!converter.HasConverterFor(couple.SrcType));

            converter.TypeConverters.Add(couple, handler);

            // 逆変換用のコンバータも同時の登録します。
            // TODO:この処理はオプションで指定できるようにしたほうがよいかもしれません。
            TypeCouple reverseCouple = couple.Reverse();
            converter.TypeConverters.Add(reverseCouple, handler);
        }

        //----------------------------------------------------------
        // アトリビュート検索
        //----------------------------------------------------------

        /// <summary>
        /// 指定型のアトリビュートを探します。
        /// </summary>
        /// <typeparam name="TAttributeType">アトリビュートの型です。</typeparam>
        /// <param name="attributes">アトリビュート配列です。</param>
        /// <returns>指定型のアトリビュートセットです。</returns>
        private static TAttributeType[] DoFindAttriubutes<TAttributeType>(
            object[] attributes) where TAttributeType : class
        {
            object[] resultObjects = Array.FindAll(attributes, (attr) => attr is TAttributeType);

            if (resultObjects.Length == 0)
            {
                return new TAttributeType[0];
            }

            TAttributeType[] result = new TAttributeType[resultObjects.Length];
            Array.Copy(resultObjects, result, resultObjects.Length);
            return result;
        }

        //----------------------------------------------------------
        // RegisterTypeConverter
        //----------------------------------------------------------

        /// <summary>
        /// 型に付加されたCoupledTypeAttributeから、コンバータを登録します。
        /// 型内部の型も再帰的に処理します。
        /// </summary>
        /// <param name="converter">コンバータです。</param>
        /// <param name="type">処理の起点となる型です。</param>
        /// <param name="checkedTypeSet">確認済みの型セットです。</param>
        private static void DoRegisterTypeConverterRecursively(
            Converter converter, Type type, List<Type> checkedTypeSet)
        {
            // プリミティブ型や確認済みの型なら処理しません。
            if (IsPrimitiveType(type) || checkedTypeSet.Contains(type))
            {
                return;
            }

            // 型を調査します。
            Type listElementType = ConverterUtil.GetIListElementType(type);
            if (listElementType == null)
            {
                // 型について調査を試みます。
                DoRegisterTypeConverter(converter, type, checkedTypeSet);
            }
            else
            {
                // 行列の要素型について調査を試みます。
                DoRegisterTypeConverterRecursively(converter, listElementType, checkedTypeSet);
            }

            // メンバプロパティの型について...
            foreach (PropertyInfo propInfo in EnumerateProperties(type))
            {
                // 再帰的に実行します。
                DoRegisterTypeConverterRecursively(converter, propInfo.PropertyType, checkedTypeSet);

                // XmlElementアトリビュートで指定されている型も調査します。
                foreach (XmlElementAttribute attr in
                    ConverterUtil.FindPropertyAttributes<XmlElementAttribute>(propInfo))
                {
                    if (attr.Type != null)
                    {
                        DoRegisterTypeConverterRecursively(converter, attr.Type, checkedTypeSet);
                    }
                }

                // XmlArrayItemアトリビュートで指定されている型も調査します。
                foreach (XmlArrayItemAttribute attr in
                    ConverterUtil.FindPropertyAttributes<XmlArrayItemAttribute>(propInfo))
                {
                    if (attr.Type != null)
                    {
                        DoRegisterTypeConverterRecursively(converter, attr.Type, checkedTypeSet);
                    }
                }
            }
        }

        /// <summary>
        /// 型に付加されたCoupledTypeAttributeから、コンバータを登録します。
        /// </summary>
        /// <param name="converter">コンバータです。</param>
        /// <param name="selfType">処理の起点となる型です。</param>
        /// <param name="checkedTypeSet">確認済みの型セットです。</param>
        private static void DoRegisterTypeConverter(
            Converter converter, Type selfType, List<Type> checkedTypeSet)
        {
            // 確認済みなら処理しません。
            Ensure.Operation.True(!checkedTypeSet.Contains(selfType));

            // プリミティブ型なら処理しません。
            if (IsPrimitiveType(selfType))
            {
                return;
            }

            // 確認済みとして登録します。
            checkedTypeSet.Add(selfType);

            // 関連型を記述するアトリビュートを探します。
            CoupledTypeAttribute[] coupledTypeAttributes
                = ConverterUtil.FindTypeAttributes<CoupledTypeAttribute>(selfType);

            // アトリビュートが見つかったら...
            if (ArrayUtility.IsNullOrEmpty(coupledTypeAttributes))
            {
                return;
            }

            foreach (CoupledTypeAttribute coupledTypeAttribute in coupledTypeAttributes)
            {
                // 関連型へのコンバータを登録します。
                foreach (Type coupledType in coupledTypeAttribute.CoupledTypes)
                {
                    DoRegisterOneCoupledType(
                        converter, selfType, coupledType, coupledTypeAttribute);

                    // 指定があれば、派生クラスを列挙して自動的に登録します。
                    if (coupledTypeAttribute.CoupledTypeAutoMode == CoupledTypeAutoMode.DerivedType)
                    {
                        foreach (Type derivedType in TypeUtility.EnumrateTypes(
                            (type) => !type.IsInterface && type.IsSupported(coupledType)))
                        {
                            DoRegisterOneCoupledType(
                                converter, selfType, derivedType, coupledTypeAttribute);
                        }
                    }
                }
            }

            // 既定クラスも調査します。
            if (selfType.BaseType != null && !checkedTypeSet.Contains(selfType.BaseType))
            {
                DoRegisterTypeConverter(converter, selfType.BaseType, checkedTypeSet);
            }
        }

        /// <summary>
        /// 関連型情報をコンバータを登録します。
        /// </summary>
        /// <param name="converter">コンバータです。</param>
        /// <param name="selfType">処理の起点となる型です。</param>
        /// <param name="coupledType">関連付けられた型です。</param>
        /// <param name="coupledTypeAttribute">関連型を記述するアトリビュートです。</param>
        private static void DoRegisterOneCoupledType(
            Converter converter,
            Type selfType,
            Type coupledType,
            CoupledTypeAttribute coupledTypeAttribute)
        {
            TypeCouple typeCouple = new TypeCouple(selfType, coupledType);
            if (!converter.TypeConverters.ContainsKey(typeCouple))
            {
                // coupledType => type
                converter.TypeConverters.Add(
                    typeCouple.Reverse(), coupledTypeAttribute.ForwardConvertHandler);
                if (selfType != coupledType)
                {
                    // type => coupledType
                    converter.TypeConverters.Add(
                        typeCouple, coupledTypeAttribute.BackwardConvertHandler);
                }
            }
        }
    }
}
