﻿// ========================================================================
// <copyright file="Converter.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Conversion
{
    using System;
    using System.Collections.Generic;
    using System.Reflection;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using NintendoWare.ToolDevelopmentKit.Logs;

    /// <summary>
    /// 型を変換します。
    /// </summary>
    /// <param name="src">入力です。</param>
    /// <param name="dstType">出力型です。</param>
    /// <param name="converter">変換の状態です。Converterから渡されます。</param>
    /// <returns>出力です。</returns>
    public delegate object ConvertHandler(object src, Type dstType, Converter converter);

    /// <summary>
    /// 型変換を行うクラスです。
    /// </summary>
    public sealed class Converter
    {
        private IList<NamedProperty> namedProperties;
        private IDictionary<TypeCouple, ConvertHandler> typeConverters;
        private IDictionary<object, object> convertedCache;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Converter()
        {
            this.NamedProperties = new List<NamedProperty>();
            this.TypeConverters = new Dictionary<TypeCouple, ConvertHandler>(
                new TypeCouple.TypeCoupleComparer());
            this.convertedCache = new Dictionary<object, object>(new ReferenceEqualityComparer<object>());
            this.UseConvertedCache = false;
            this.Logger = new NullLogger();
        }

        /// <summary>
        /// 型コンバータの辞書です。
        /// </summary>
        public IDictionary<TypeCouple, ConvertHandler> TypeConverters
        {
            get
            {
                return this.typeConverters;
            }

            set
            {
                Ensure.Argument.NotNull(value);
                this.typeConverters = value;
            }
        }

        /// <summary>
        /// 名前付きプロパティです。
        /// </summary>
        public IList<NamedProperty> NamedProperties
        {
            get
            {
                return this.namedProperties;
            }

            set
            {
                Ensure.Argument.NotNull(value);
                this.namedProperties = value;
            }
        }

        /// <summary>
        /// コンバート済みインスタンスの辞書です。
        /// </summary>
        public IDictionary<object, object> ConvertedCache
        {
            get
            {
                return this.convertedCache;
            }

            set
            {
                Ensure.Argument.NotNull(value);
                this.convertedCache = value;
            }
        }

        /// <summary>
        /// コンバート結果のキャッシュを利用するかどうかを設定します。
        /// </summary>
        public bool UseConvertedCache { get; set; }

        /// <summary>
        /// ロガーです。
        /// </summary>
        public ILogger Logger { get; set; }

        //-----------------------------------------------------------------
        // 変換
        //-----------------------------------------------------------------

        /// <summary>
        /// キャッシュ使用の設定が有効な場合に、コンバート結果をキャッシュに登録します。
        /// </summary>
        /// <param name="src">コンバート前のインスタンスです。</param>
        /// <param name="result">コンバート結果のインスタンスです。</param>
        public void TouchToConvertedCache(object src, object result)
        {
            if (this.UseConvertedCache && src != null && result != null)
            {
                if (!this.ConvertedCache.ContainsKey(src))
                {
                    this.ConvertedCache.Add(src, result);
                }
            }
        }

        /// <summary>
        /// 変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <returns>出力です。</returns>
        public object Convert(object src, Type dstType)
        {
            // コンバート済みデータに含まれている場合は、データベース中のインスタンスを返します。
            if (this.UseConvertedCache && this.ConvertedCache.ContainsKey(src))
            {
                return this.ConvertedCache[src];
            }

            // 関連付けられた実体があるならそれを利用します。
            ICoupledInstanceType srcCoupledInstanceType = src as ICoupledInstanceType;
            if (srcCoupledInstanceType != null && srcCoupledInstanceType.CoupledInstance != null)
            {
                return srcCoupledInstanceType.CoupledInstance;
            }

            Type srcType = src.GetType();
            ConvertHandler convertHandler = null;

            // 登録コンバータから該当するコンバータを取得します。
            Type resultType = null;
            Type[] resultTypes = this.FindConvertionResultTypes(srcType);

            // 変換先の型が指定されていれば絞り込みます...
            if (dstType != null)
            {
                // 設定可能なものを選出します。
                resultType = Array.Find(resultTypes, (type) => dstType.IsAssignableFrom(type));
                if (resultType == null)
                {
                    // 登録がなければ規定のコンバータを探します。
                    resultType = dstType;
                    convertHandler = this.FindDefaultTypeConverter(src.GetType(), dstType);
                }
                else
                {
                    this.TypeConverters.TryGetValue(
                        new TypeCouple(srcType, resultType), out convertHandler);
                }
            }
            else
            {
                if (resultTypes.Length > 0)
                {
                    // 最初に見つかった型に変換する。
                    resultType = resultTypes[0];
                    this.TypeConverters.TryGetValue(
                        new TypeCouple(srcType, resultType), out convertHandler);
                }
            }

            // ハンドラが発見されれば、変換を行います。
            if (convertHandler != null)
            {
                this.TryRegisterObjectNamedProperty(src);
                object result = convertHandler(src, resultType, this);

                // 関連づいたインスタンスなら、変換結果を記録しておきます。
                if (result is ICoupledInstanceType)
                {
                    (result as ICoupledInstanceType).CoupledInstance = src;
                }
                else if (src is ICoupledInstanceType)
                {
                    (src as ICoupledInstanceType).CoupledInstance = result;
                }

                this.TouchToConvertedCache(src, result);

                return result;
            }

            return null;
        }

        /// <summary>
        /// 実体を生成します。
        /// <para>
        /// 名前つきプロパティの登録処理を行っています。
        /// 必要に応じて、拡張ハンドラ内で、インスタンスを生成する場合にも利用してください。
        /// </para>
        /// </summary>
        /// <param name="type">生成する実体の型です。</param>
        /// <returns>実体です。</returns>
        public object CreateInstance(Type type)
        {
            try
            {
                object result = Activator.CreateInstance(type);

                this.TryRegisterObjectNamedProperty(result);

                return result;
            }
            catch (Exception ex)
            {
                // インスタンスの生成に失敗しました。
                string msg = string.Format(
                    "Converter : Can't constract a instance of [{0}].", type.Name);

                Exception newException = new InvalidOperationException(msg, ex);
                throw newException;
            }
        }

        /// <summary>
        /// 実体を生成します。
        /// </summary>
        /// <typeparam name="TInstanceType">生成する実体の型です。</typeparam>
        /// <returns>実体です。</returns>
        public TInstanceType CreateInstance<TInstanceType>()
            where TInstanceType : class
        {
            return this.CreateInstance(typeof(TInstanceType)) as TInstanceType;
        }

        //-----------------------------------------------------------------
        // コンバートハンドラの検索
        //-----------------------------------------------------------------

        /// <summary>
        /// コンバートが登録されているか調査します。
        /// </summary>
        /// <param name="srcType">変換前の型です。</param>
        /// <returns>コンバート可能ならtrueを返します。</returns>
        public bool HasConverterFor(Type srcType)
        {
            return this.FindConvertionResultTypes(srcType).Length != 0;
        }

        /// <summary>
        /// 型コンバータを取得します。
        /// </summary>
        /// <param name="srcType">変換前の型です。</param>
        /// <param name="dstType">変換後の型です。</param>
        /// <returns>型コンバータです。</returns>
        public ConvertHandler FindDefaultTypeConverter(Type srcType, Type dstType)
        {
            Ensure.Argument.NotNull(srcType);
            Ensure.Argument.NotNull(dstType);

            // プリミティブならプリミティブコンバータを適用します。
            if (ConverterUtil.IsPrimitiveType(srcType) && (dstType == srcType))
            {
                return DefaultConverters.ConvertPrimitive;
            }

            // リストならリストコンバータを適用します。
            if (ConverterUtil.CheckTypeImplementsList(srcType))
            {
                return DefaultConverters.ConvertList;
            }

            // 最後にはクラスコンバータを適用します。
            return DefaultConverters.ConvertClass;
        }

        /// <summary>
        /// 変換結果の型をすべて取得します。
        /// </summary>
        /// <param name="srcType">変換元の型です。</param>
        /// <returns>変換後の型です。</returns>
        public Type[] FindConvertionResultTypes(Type srcType)
        {
            List<Type> result = new List<Type>();
            foreach (TypeCouple couple in this.TypeConverters.Keys)
            {
                if (couple.SrcType == srcType)
                {
                    result.Add(couple.DstType);
                }
            }

            return result.ToArray();
        }

        //-----------------------------------------------------------------
        // 名前つきプロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したプロパティに対応する名前つきプロパティを取得します。
        /// </summary>
        /// <param name="name">プロパティ名です。</param>
        /// <returns>名前つきプロパティです。</returns>
        public NamedProperty FindNamedPropaerty(string name)
        {
            NamedProperty namedProperty = null;
            if (name != null)
            {
                namedProperty = ListUtility.Find(this.NamedProperties, (prop) => prop.Name == name);
            }

            return namedProperty;
        }

        /// <summary>
        /// 名前つきプロパティの登録を試みます。
        /// </summary>
        /// <param name="propertyInfo">プロパティ情報です。</param>
        /// <param name="targetObject">対象オブジェクトです。</param>
        private void TryRegisterNamedProperty(PropertyInfo propertyInfo, object targetObject)
        {
            string propertyName = ConverterUtil.GetNamedPropaertyName(propertyInfo);

            if (propertyName != null)
            {
                NamedProperty namedProperty = this.FindNamedPropaerty(propertyName);
                ListUtility.Find(this.NamedProperties, (prop) => prop.Name == propertyName);

                if (namedProperty == null)
                {
                    namedProperty = new NamedProperty(propertyName, propertyInfo);
                    namedProperty.TargetObject = targetObject;

                    this.NamedProperties.Add(namedProperty);
                }
                else
                {
                    namedProperty.TargetObject = targetObject;
                }
            }
        }

        /// <summary>
        /// 名前つきプロパティの登録を試みます。
        /// </summary>
        /// <param name="targetObject">対象オブジェクトです。</param>
        private void TryRegisterObjectNamedProperty(object targetObject)
        {
            Ensure.Argument.NotNull(targetObject);
            foreach (PropertyInfo dstPropInfo in ConverterUtil.EnumerateProperties(targetObject))
            {
                this.TryRegisterNamedProperty(dstPropInfo, targetObject);
            }
        }
    }
}
