﻿// ========================================================================
// <copyright file="CacheManagerUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.ComponentModel
{
    using System;
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// キャッシュマネージャのユーティリティクラスです。
    /// </summary>
    public static class CacheManagerUtility
    {
        /// <summary>
        /// キャッシュ値を取得します。
        /// </summary>
        /// <typeparam name="TValue">キャッシュ値の型を指定します。</typeparam>
        /// <param name="cacheManager">キャッシュマネージャを指定します。</param>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <returns>
        /// キャッシュ値を返します。
        /// キャッシュが見つからない場合は、新しい TValue のインスタンスをキャッシュし、
        /// その値を返します。
        /// </returns>
        public static TValue GetCacheValue<TValue>(
            this CacheManager cacheManager,
            object keyObject)
            where TValue : new()
        {
            return GetCacheValue<TValue>(cacheManager, keyObject, keyObj => new TValue());
        }

        /// <summary>
        /// キャッシュ値を取得します。
        /// </summary>
        /// <typeparam name="TValue">キャッシュ値の型を指定します。</typeparam>
        /// <param name="cacheManager">キャッシュマネージャを指定します。</param>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <param name="createValue">キャッシュ値を作成するメソッドを指定します。</param>
        /// <returns>
        /// キャッシュ値を返します。
        /// キャッシュが見つからない場合は、新しい TValue のインスタンスをキャッシュし、
        /// その値を返します。
        /// </returns>
        public static TValue GetCacheValue<TValue>(
            this CacheManager cacheManager,
            object keyObject,
            Func<object, TValue> createValue)
        {
            Ensure.Argument.NotNull(cacheManager);
            Ensure.Argument.NotNull(keyObject);
            Ensure.Argument.NotNull(createValue);

            CacheItem result = cacheManager.TryGetCache(keyObject);

            if (result == null ||
                !result.IsValid ||
                !(result.Value is TValue))
            {
                TValue newValue = createValue(keyObject);
                Ensure.Operation.ObjectNotNull(newValue);

                cacheManager.AddCache(keyObject, newValue);

                return newValue;
            }

            return (TValue)result.Value;
        }
    }
}
