﻿// ========================================================================
// <copyright file="ObservableList.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Collections.Specialized;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// コレクション変更通知をサポートするリストです。
    /// </summary>
    /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
    public class ObservableList<TItem> : ObservableCollection<TItem>
    {
        private readonly HashSet<IDisposable> collectionChangedLockObjects =
            new HashSet<IDisposable>();

        private bool isDirty = false;
        private NotifyCollectionChangedEventArgs collectionChangedEventArgs;

        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public ObservableList()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の列挙子です。</param>
        public ObservableList(IEnumerable<TItem> source)
        {
            this.AddRange(source);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 複数のアイテムを追加します。
        /// </summary>
        /// <param name="items">アイテムの列挙子を指定します。</param>
        public void AddRange(IEnumerable<TItem> items)
        {
            using (this.SuspendNotifyCollectionChanged())
            {
                foreach (TItem item in items)
                {
                    this.Add(item);
                }
            }
        }

        /// <summary>
        /// 複数のアイテムを削除します。
        /// </summary>
        /// <param name="items">アイテムの列挙子を指定します。</param>
        public void RemoveRange(IEnumerable<TItem> items)
        {
            using (this.SuspendNotifyCollectionChanged())
            {
                foreach (TItem item in items)
                {
                    this.Remove(item);
                }
            }
        }

        /// <summary>
        /// コレクション変更通知を一時的に停止します。
        /// </summary>
        /// <returns>コレクション変更通知のロックオブジェクトを返します。</returns>
        /// <remarks>
        /// 戻り値で返されるオブジェクトが破棄されるまでの間、コレクション変更通知は発行されません。
        /// </remarks>
        public IDisposable SuspendNotifyCollectionChanged()
        {
            IDisposable result = new DisposeActionObject(this.ResumeNotifyCollectionChanged);
            this.collectionChangedLockObjects.Add(result);

            return result;
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            if (other == this)
            {
                return true;
            }

            if ((other == null) || (other.GetType() != GetType()))
            {
                return false;
            }

            return this.Equals(other as ObservableList<TItem>);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            int hashCode = GetType().GetHashCode();
            foreach (TItem item in this)
            {
                if (item != null)
                {
                    hashCode ^= item.GetHashCode();
                }
            }

            return hashCode;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(ObservableList<TItem> other)
        {
            Ensure.Argument.NotNull(other);

            int count = this.Count;
            if (count != other.Count)
            {
                return false;
            }

            for (int i = 0; i < count; i++)
            {
                if (!object.Equals(this[i], other[i]))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// コレクションが変更されると発生します。
        /// </summary>
        /// <param name="e">これんクション変更イベントデータを指定します。</param>
        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            if (this.collectionChangedLockObjects.Count > 0)
            {
                this.collectionChangedEventArgs = this.isDirty ? null : e;
                this.isDirty = true;
                return;
            }

            this.isDirty = false;
            this.collectionChangedEventArgs = null;

            base.OnCollectionChanged(e);
        }

        /// <summary>
        /// コレクション変更通知を再開します。
        /// </summary>
        /// <param name="lockObject">コレクション変更通知のロックオブジェクトを指定します。</param>
        private void ResumeNotifyCollectionChanged(DisposeActionObject lockObject)
        {
            Assertion.Argument.NotNull(lockObject);

            this.collectionChangedLockObjects.Remove(lockObject);

            if (this.isDirty && this.collectionChangedLockObjects.Count == 0)
            {
                this.OnCollectionChanged(
                    this.collectionChangedEventArgs ??
                    new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Reset));
            }
        }
    }
}
