﻿// ========================================================================
// <copyright file="ObservableChildList.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System.Collections.Generic;

    /// <summary>
    /// コレクション変更通知をサポートする子オブジェクトのリストです。
    /// </summary>
    /// <typeparam name="TParent">親オブジェクトの型を指定します。</typeparam>
    /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
    public class ObservableChildList<TParent, TItem> : ObservableInstanceList<TItem>
        where TParent : class
        where TItem : class, IChildObject<TParent>
    {
        private readonly TParent parent = null;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親オブジェクトを指定します。</param>
        public ObservableChildList(TParent parent)
        {
            Ensure.Argument.NotNull(parent);
            this.parent = parent;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の列挙子を指定します。</param>
        /// <param name="parent">親オブジェクトを指定します。</param>
        public ObservableChildList(IEnumerable<TItem> source, TParent parent)
            : this(parent)
        {
            this.AddRange(source);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 親オブジェクトを取得します。
        /// </summary>
        public TParent Parent
        {
            get
            {
                return this.parent;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コレクション内の指定したインデックスの位置にアイテムを挿入します。
        /// </summary>
        /// <param name="index">アイテムの挿入先インデックスを指定します。</param>
        /// <param name="item">新しいアイテムを指定します。</param>
        protected override void InsertItem(int index, TItem item)
        {
            Ensure.Argument.NotNull(item);

            Ensure.Argument.True(item.Parent == null || item.Parent == this.Parent);
            item.Parent = this.Parent;

            base.InsertItem(index, item);
        }

        /// <summary>
        /// 指定したインデックスが示す位置にあるアイテムをコレクションから削除します。
        /// </summary>
        /// <param name="index">削除するアイテムのインデックスを指定します。</param>
        protected override void RemoveItem(int index)
        {
            this.Items[index].Parent = null;
            base.RemoveItem(index);
        }

        /// <summary>
        /// 指定したインデックス位置にあるアイテムを置き換えます。
        /// </summary>
        /// <param name="index">置き換えるアイテムのインデックスを指定します。</param>
        /// <param name="item">新しいアイテムを指定します。</param>
        protected override void SetItem(int index, TItem item)
        {
            Ensure.Argument.NotNull(item);

            Ensure.Argument.True(item.Parent == null || item.Parent == this.Parent);
            item.Parent = this.Parent;

            base.SetItem(index, item);
        }

        /// <summary>
        /// アイテムをクリアします。
        /// </summary>
        protected override void ClearItems()
        {
            try
            {
                foreach (TItem item in this.Items)
                {
                    Assertion.Operation.True(item.Parent == null || item.Parent == this.Parent);
                    item.Parent = null;
                }
            }
            finally
            {
                base.ClearItems();
            }
        }
    }
}
