﻿// ========================================================================
// <copyright file="ChildDictionaryDecorator.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System.Collections.Generic;

    /// <summary>
    /// 親を持つディクショナリーのデコレータです。
    /// </summary>
    /// <typeparam name="TParent">親のテンプレート型です。</typeparam>
    /// <typeparam name="TKey">ディクショナリー内のキーのテンプレート型です。</typeparam>
    /// <typeparam name="TValue">ディクショナリー内の値のテンプレート型です。</typeparam>
    public class ChildDictionaryDecorator<TParent, TKey, TValue> :
        DictionaryDecorator<TKey, TValue>
        where TParent : class
        where TValue : class, IChildObject<TParent>
    {
        //-----------------------------------------------------------------
        // フィールド
        //-----------------------------------------------------------------
        private readonly TParent parent = null;

        //-----------------------------------------------------------------
        // コンストラクタ
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        /// <param name="dictionary">デコレート対象のIDictionaryを持つオブジェクトです。</param>
        /// <param name="parent">要素に設定する親です。</param>
        public ChildDictionaryDecorator(IDictionary<TKey, TValue> dictionary, TParent parent)
            : base(dictionary)
        {
            Ensure.Argument.NotNull(parent);

            this.parent = parent;
        }

        //-----------------------------------------------------------------
        // public プロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// 要素に設定する親を取得します。
        /// </summary>
        protected virtual TParent Parent
        {
            get { return this.parent; }
        }

        /// <summary>
        /// 指定したキーを持つ要素を取得または設定します。
        /// </summary>
        /// <param name="key">取得または設定する要素のキーです。</param>
        /// <returns>指定したキーを持つ要素です。</returns>
        public override TValue this[TKey key]
        {
            set
            {
                Ensure.Argument.True(
                    (value == null) || (value.Parent == null) || (value.Parent == this.Parent));

                TValue oldValue = base[key];

                Ensure.Operation.True((oldValue == null) || (oldValue.Parent == this.Parent));

                base[key] = value;

                if (value != null)
                {
                    value.Parent = this.Parent;
                }

                if (oldValue != null)
                {
                    oldValue.Parent = null;
                }
            }
        }

        //-----------------------------------------------------------------
        // public メソッド
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーおよび値を持つ要素を追加します。
        /// </summary>
        /// <param name="key">追加する要素のキーです。</param>
        /// <param name="value">追加する要素の値です。</param>
        public override void Add(TKey key, TValue value)
        {
            if (value != null)
            {
                Ensure.Argument.True((value.Parent == null) || (value.Parent == this.Parent));

                base.Add(key, value);
                value.Parent = this.Parent;
            }
            else
            {
                base.Add(key, value);
            }
        }

        /// <summary>
        /// 指定したキーを持つ値を削除します。
        /// </summary>
        /// <param name="key">削除する要素のキーです。</param>
        /// <returns>
        /// 要素が見つかり正常に削除された場合は true です。それ以外の場合は false です。
        /// </returns>
        public override bool Remove(TKey key)
        {
            bool result;

            TValue value = this[key];
            Ensure.Operation.True((value == null) || (value.Parent == this.Parent));

            result = base.Remove(key);

            if (value != null)
            {
                value.Parent = null;
            }

            return result;
        }

        /// <summary>
        /// すべての要素を削除します。
        /// </summary>
        public override void Clear()
        {
            foreach (TValue value in this.Values)
            {
                Ensure.Operation.True((value == null) || (value.Parent == this.Parent));

                if (value != null)
                {
                    value.Parent = null;
                }
            }

            base.Clear();
        }
    }
}
