﻿// ========================================================================
// <copyright file="IConnecter.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Net;
using System.IO;
using System.Text;
using NWCore.Protocols;

namespace NWCore.Viewer
{
    #region Class for connection context

    /// <summary>
    /// Class for the information of the connection.
    /// </summary>
    public class ConnectionContext
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public ConnectionContext()
        {
            Reset();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source context to copy from.</param>
        public ConnectionContext( ConnectionContext src )
        {
            Copy( src );
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">The display name for the connection.</param>
        /// <param name="identifier">The identifier for the connection.</param>
        /// <param name="identifier">The identifier for the connector the connection uses.</param>
        /// <param name="ipAddr">The IP address to connect to.</param>
        /// <param name="port">The port number to connect with.</param>
        /// <param name="fileExt">The binary file extension for the connection.</param>
        /// <param name="platform">The platform type the connection connects to.</param>
        /// <param name="encoding">The encoding the connection uses.</param>
        public ConnectionContext( string name,
                                  string identifier,
                                  string connecterIdentifier,
                                  IPAddress ipAddr,
                                  int port,
                                  string fileExt,
                                  PlatformType platform,
                                  Encoding encoding )
        {
            this.Name                = string.Copy( name );
            this.Identifier          = string.Copy( identifier );
            this.ConnecterIdentifier = string.Copy( connecterIdentifier );
            this.IPAddress           = ipAddr;
            this.Port                = port;
            this.BinaryFileExt       = string.Copy( fileExt );
            this.Platform            = platform;
            this.Encoding            = encoding;

            this.ShouldAcquireChannelInfo = false;
            this.ChannelInfoPort          = -1;
            this.ChannelHeaderCode        = string.Empty;

            this.UseWinMessage       = false;
            this.ViewerProcessName   = string.Empty;

            this.IsReady = true;
        }


        /// <summary>
        /// Constructor.
        /// This connection requires first connecting to the viewer and acquire actual
        /// port number from the received channel information.
        /// </summary>
        /// <param name="name">The display name for the connection.</param>
        /// <param name="identifier">The identifier for the connection.</param>
        /// <param name="identifier">The identifier for the connector the connection uses.</param>
        /// <param name="ipAddr">The IP address to connect to.</param>
        /// <param name="channelInfoPort">The port number for acquiring the channel information.</param>
        /// <param name="channelHeaderCode">The header code for the channel to connect to.</param>
        /// <param name="fileExt">The binary file extension for the connection.</param>
        /// <param name="platform">The platform type the connection connects to.</param>
        /// <param name="encoding">The encoding the connection uses.</param>
        public ConnectionContext( string name,
                                  string identifier,
                                  string connecterIdentifier,
                                  IPAddress ipAddr,
                                  int channelInfoPort,
                                  string channelHeaderCode,
                                  string fileExt,
                                  PlatformType platform,
                                  Encoding encoding ) :
            this( name,
                  identifier,
                  connecterIdentifier,
                  ipAddr,
                  -1,
                  fileExt,
                  platform,
                  encoding )
        {
            this.ShouldAcquireChannelInfo = true;
            this.ChannelInfoPort          = channelInfoPort;
            this.ChannelHeaderCode        = string.Copy( channelHeaderCode );

            this.IsReady = true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the display name for the connection.</summary>
        public string Name { get; set; }

        /// <summary>Get or set the identifier string for the connection.</summary>
        public string Identifier { get; set; }

        /// <summary>Get or set the identifier string of the connector the connection uses.</summary>
        public string ConnecterIdentifier { get; set; }

        /// <summary>Get or set the IP address of the connection.</summary>
        public IPAddress IPAddress { get; set; }

        /// <summary>Get or set the port number of the connection.</summary>
        public int Port { get; set; }

        /// <summary>Get or set the port number for acquiring channel information.</summary>
        public int ChannelInfoPort { get; set; }

        /// <summary>
        /// Get or set the flag indicating if should ask the server
        /// for the actual port number for connection.
        /// </summary>
        public bool ShouldAcquireChannelInfo { get; set; }

        /// <summary>Get or set the header code for the channel to connect to.</summary>
        public string ChannelHeaderCode { get; set; }

        /// <summary>Get or set the binary file extension for the connection.</summary>
        public string BinaryFileExt { get; set; }

        /// <summary>Get or set the platform type the connection connects to.</summary>
        public PlatformType Platform { get; set; }

        /// <summary>Get or set the encoding the connection uses.</summary>
        public Encoding Encoding { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the information is ready for use.
        /// </summary>
        public bool IsReady { get; private set; }

        /// <summary>Get or set the flag indicating whether to use windows message.</summary>
        public bool UseWinMessage { get; set; }

        /// <summary>Get or set the viewer process name.</summary>
        public string ViewerProcessName { get; set; }

        #endregion

        #region Utility methods

        /// <summary>
        /// Copy from the given connection context.
        /// </summary>
        /// <param name="src">The source context to copy from.</param>
        /// <returns>True on success.</returns>
        public bool Copy( ConnectionContext src )
        {
            if ( src==null )
                return false;

            this.Name                = string.Copy( src.Name );
            this.Identifier          = string.Copy( src.Identifier );
            this.ConnecterIdentifier = string.Copy( src.Identifier );
            this.IPAddress           = new IPAddress( src.IPAddress.GetAddressBytes() ) ;
            this.Port                = src.Port;
            this.BinaryFileExt       = string.Copy( src.BinaryFileExt );
            this.Platform            = src.Platform;
            this.Encoding            = src.Encoding;

            this.ShouldAcquireChannelInfo = src.ShouldAcquireChannelInfo;
            this.ChannelInfoPort          = src.ChannelInfoPort;
            this.ChannelHeaderCode        = string.Copy( src.ChannelHeaderCode );

            this.UseWinMessage       = src.UseWinMessage;
            this.ViewerProcessName   = string.Copy( src.ViewerProcessName );

            this.IsReady = src.IsReady;

            return true;
        }


        /// <summary>
        /// Clean out the connection information.
        /// </summary>
        public void Reset()
        {
            this.Name                = string.Empty;
            this.Identifier          = string.Empty;
            this.ConnecterIdentifier = string.Empty;
            this.IPAddress           = null;
            this.Port                = -1;
            this.BinaryFileExt       = string.Empty;
            this.Platform            = PlatformType.Windows;
            this.Encoding            = Encoding.UTF8;

            this.ShouldAcquireChannelInfo = false;
            this.ChannelInfoPort          = -1;
            this.ChannelHeaderCode        = string.Empty;

            this.UseWinMessage       = false;
            this.ViewerProcessName   = string.Empty;

            this.IsReady = false;
        }


        /// <summary>
        /// Check if the target of this and the given connection context are the same.
        /// </summary>
        /// <param name="context">The connection context to compare with.</param>
        /// <returns>True if the targets are the same.</returns>
        public bool IsSameTarget( ConnectionContext context )
        {
            if ( context==null )
                return false;

            // Connector
            if ( this.ConnecterIdentifier!=context.ConnecterIdentifier )
                return false;

            // Use windows message. Only for backward compatiblility.
            if ( this.UseWinMessage!=context.UseWinMessage )
            {
                return false;
            }
            else if ( this.UseWinMessage==true &&
                      this.ViewerProcessName!=context.ViewerProcessName )
            {
                return false;
            }
            else if ( this.UseWinMessage==true &&
                      this.ViewerProcessName==context.ViewerProcessName )
            {
                return true;
            }

            // IP address
            if ( this.IPAddress.Equals( context.IPAddress )==false )
                return false;

            // Acquire channel information?
            if ( this.ShouldAcquireChannelInfo!=context.ShouldAcquireChannelInfo )
                return false;

            // Port
            if ( this.ShouldAcquireChannelInfo==true &&
                 (this.ChannelInfoPort!=context.ChannelInfoPort ||
                  this.ChannelHeaderCode!=context.ChannelHeaderCode) )
            {
                return false;
            }
            else if ( this.ShouldAcquireChannelInfo==false &&
                      this.Port!=context.Port )
            {
                return false;
            }

            return true;
        }

        #endregion
    }

    #endregion

    /// <summary>
    /// コネクター用インターフェイス
    /// </summary>
    public interface IConnecter : IDisposable
    {
        /// <summary>
        /// Check if the connector is currently connected.
        /// </summary>
        bool IsConnected { get; }

        /// <summary>
        /// Get the identifier of the connector.
        /// </summary>
        string Identifier { get; }

        /// <summary>
        /// Get the size of the buffered incoming data.
        /// </summary>
        bool IsIncomingBufferFull { get; }

        /// <summary>
        /// Initialize the connector.
        /// </summary>
        bool Initialize();

        /// <summary>
        /// Check the existence of the target of the given connection information.
        /// </summary>
        /// <param name="context">The connection context.</param>
        /// <returns>True if the target exists.</returns>
        bool CheckTargetExistence( ConnectionContext context );

        /// <summary>
        /// Connect with the specified connection context( connect settings ).
        /// </summary>
        bool Connect( ConnectionContext context );

        /// <summary>
        /// Close the connection.
        /// </summary>
        void Close();

        /// <summary>
        /// Get the information of the current connection.
        /// Null is returned if there is no connection.
        /// </summary>
        /// <returns>The information of the current connection or null if no connection.</returns>
        ConnectionContext GetConnectionContext();

        /// <summary>
        /// Send data through the connection.
        /// </summary>
        /// <param name="buffer">The data buffer to send.</param>
        /// <param name="iBufferSize">Size of the data buffer.</param>
        /// <returns>True on success.</returns>
        bool Send( byte[] buffer, int iBufferSize );

        /// <summary>
        /// Receive data from the connection.
        /// </summary>
        /// <param name="buffer">The buffer to store the read data.</param>
        /// <returns>Size of the buffer read from the connection.</returns>
        int Receive( out byte[] buffer );
    }
}
