﻿// ========================================================================
// <copyright file="HostIODriver.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using NW4F.Protocols;

namespace NW4F.Viewer.Driver
{
    /// <summary>
    /// HostIO 制御用
    /// </summary>
    public class HostIODriver : IDisposable
    {
        private const string STR_HEADCODE = "NW_MCS";
        private const int PORT_NO = 6003;
        private static readonly byte[] IP_ADR_BYTE = new byte[] { 127, 0, 0, 1 };

        private volatile object m_lock = new object();
        private int m_channel;
        private int m_connectionPort;
//        private TcpClient m_client;
        private TcpClient m_commSocket;
//        private TcpListener m_listener;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HostIODriver()
        {
            m_channel = -1;
            m_connectionPort = -1;
        }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="sendChannel">利用する送信チャンネル</param>
        /// <param name="recvChannel">利用する受信チャンネル</param>
        public void Initialize(int sendChannel, int recvChannel)
        {
            this.m_channel = sendChannel;
        }

        /// <summary>
        /// 破棄
        /// </summary>
        public void Dispose()
        {
            if (HasOpened)
            {
                Close();
            }
        }

        /// <summary>
        /// 既にオープンしているか？
        /// </summary>
        public bool HasOpened
        {
            get { return (this.m_commSocket != null); }
        }

        /// <summary>
        /// オープン
        /// </summary>
        /// <returns>=true..成功 /=false..失敗</returns>
        public bool Open()
        {
            lock (m_lock)
            {
                if (m_commSocket == null)
                {
                    try
                    {
                        // 接続処理
                        var ipAddress = new IPAddress(IP_ADR_BYTE);
                        using (var client = new TcpClient())
                        {
                            client.Connect(ipAddress, PORT_NO);
                            Thread.Sleep(100);
                            if (client != null && client.Connected)
                            {
                                // コネクションポートを取得
                                m_connectionPort = GetTargetPort(client);
                                if (m_connectionPort >= 0)
                                {
                                    var point = new IPEndPoint(ipAddress, m_connectionPort);
                                    //                                 m_listener = new TcpListener(point);
                                    //                                 m_listener.Start();
                                    m_commSocket = new TcpClient();
                                    m_commSocket.Connect(point);
                                    Thread.Sleep(100);
                                }
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        // 何かエラーが発生
                        DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                        m_connectionPort = -1;
                    }
                }
            }
            if (m_connectionPort < 0)
            {
                // 接続失敗
                Close();
                DebugConsole.DebugConsole.Instance.WriteLine("- Couldn't connect with Cafe");
                return false;
            }

            return true;
        }

        /// <summary>
        /// クローズ
        /// </summary>
        /// <returns></returns>
        public bool Close()
        {
            if ( HasOpened == true )
            {
                lock (m_lock)
                {
                    if (m_commSocket != null)
                    {
                        m_commSocket.Close();
                        m_commSocket = null;
                    }

//                     if (this.m_listener != null)
//                     {
//                         this.m_listener.Stop();
//                         this.m_listener = null;
//                     }
//
//                     if (this.m_client != null)
//                     {
//                         this.m_client.Close();
//                         this.m_client = null;
//                     }
                }
            }

            this.m_channel = -1;
            return true;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        /// <param name="stream"></param>
        /// <returns>実際に読み込んだデータ</returns>
        public int Read(ref byte[] readData)
        {
            if (m_commSocket == null || HasOpened == false)
            {
                return 0;
            }

            try
            {
                if (m_commSocket.Available > 0)
                {
                    int size = 0;
                    lock (m_lock)
                    {
                        NetworkStream stream = m_commSocket.GetStream();
                        readData = new byte[m_commSocket.Available];
                        size = stream.Read(readData, 0, m_commSocket.Available);
                    }

                    return size;
                }
            }
            catch
            {
                // 何かエラーが発生
            }
            return 0;
        }

        /// <summary>
        /// 書き込み
        /// </summary>
        /// <param name="sendData"></param>
        /// <returns></returns>
        public bool Write(byte[] sendData)
        {
            if (m_commSocket == null || HasOpened == false)
            {
                return false;
            }

            try
            {
                lock (m_lock)
                {
                    var stream = m_commSocket.GetStream();
                    stream.Write(sendData, 0, sendData.Length);
                    stream.Flush();
                }
            }
            catch (Exception ex)
            {
                // 何かエラーが発生
                DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                return false;
            }
            return true;
        }


        /// <summary>
        /// 文字列をチャンネル情報の配列に分割します。
        /// </summary>
        /// <param name="message">全チャンネル情報のはいった文字列です。</param>
        /// <returns>チャンネル情報単位に分割して配列を返します。</returns>
        IList<string> SplitChannelInfos(string message)
        {
            List<string> stringList = new List<string>();

            for (int index = 0; index < message.Length; )
            {
                if (message[index] == '+' || message[index] == '-')
                {
                    int i = index + 1;
                    for (; i < message.Length; ++i)
                    {
                        if (message[i] == '+' || message[i] == '-')
                        {
                            break;
                        }
                    }

                    stringList.Add(message.Substring(index, i - index));
                    index = i;
                }
                else
                {
                    ++index;
                }
            }

            return stringList;
        }

        /// <summary>
        /// ネットワーク上のストリームからポート情報を読み出します。
        /// </summary>
        /// <param name="stream">ポート情報を取得する為のストリームです。</param>
        /// <returns>NW_MCS のポート番号を返します。</returns>
        int GetTargetPort(TcpClient client)
        {
            MemoryStream memory = new MemoryStream();
            NetworkStream stream = client.GetStream();

            try
            {
                int mcsPortNo = -1;

                int readBytes = 0;
                byte[] buffer = new byte[256];

                do
                {
                    readBytes = stream.Read(buffer, 0, buffer.Length);

                    if (readBytes > 0)
                    {
                        memory.Write(buffer, 0, readBytes);
                    }

                } while (client.Available > 0);

                string infoString = Encoding.ASCII.GetString(memory.ToArray());

                IList<string> channels = this.SplitChannelInfos(infoString);

                foreach (string channel in channels)
                {
                    string[] tokens = channel.Split(new char[] { ':' }, 2);

                    if (tokens.Length != 2)
                    {
                        continue;
                    }

                    string channelName = tokens[0];
                    int port = int.Parse(tokens[1]);

                    DebugConsole.DebugConsole.Instance.WriteLine(
                        "- Channel name : " + channelName.Substring(1) + " / port : " + port);

                    if (0 != channelName.Substring(1).CompareTo(STR_HEADCODE))
                    {
                        continue;
                    }

                    if (channelName[0] == '+' || channelName[0] == '-')
                    {
                        return port;
                    }

                    mcsPortNo = port;
                }

                return mcsPortNo;
            }
            catch (IOException)
            {
                return -1;
            }
        }
    }
}
