﻿// ========================================================================
// <copyright file="AnimTableListXml.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Globalization;

using NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0;

namespace NWCore.Serializer.Major_1.Minor_5.Build_0.Revision_0
{
    /// <summary>
    /// Class for serializing a list of the animation tables.
    /// </summary>
    public abstract class AnimTableListXml : IXmlSerializable
    {
        #region Properties

        /// <summary>
        /// The animation table list.
        /// </summary>
        public AnimTableList AnimationList { get; set; }

        #endregion

        #region Implemented for IXmlSerializable

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml( XmlReader reader )
        {
            if ( reader.MoveToContent()==XmlNodeType.Element &&
                 reader.LocalName=="animations" )
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml( reader.ReadOuterXml() );

                foreach ( XmlNode child in doc.DocumentElement.ChildNodes )
                {
                    if ( child!=null && child.Name=="key_frame_animation" )
                    {
                        // Parse the animation table.
                        AnimTableData data = ParseAnimationTable( child );

                        this.AnimationList.Animations.Add( data );
                    }
                }
            }
        }


        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml( XmlWriter writer )
        {
            foreach ( AnimTableData data in this.AnimationList.Animations )
            {
                WriteAnimationTableNode( writer, data );
            }
        }


        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        #endregion

        #region Parse XML

        /// <summary>
        /// Parse the animation table.
        /// </summary>
        /// <param name="node">The XML node containing the animation table data.</param>
        /// <returns>The created animation table data.</returns>
        private AnimTableData ParseAnimationTable( XmlNode node )
        {
            AnimTableData data = new AnimTableData();

            #region Parse the attributes of the XML node

            // "name" attribute
            {
                AnimTableTargetTypes value;
                if ( Enum.TryParse<AnimTableTargetTypes>(node.Attributes["name"].InnerText,
                                                         out value)==false )
                {
                    throw new System.Xml.XmlException( "\"name\" attribute is missing or of a wrong type in <key_frame_animation>." );
                }
                data.Name = value;
            }

            // "enabled" attribute
            {
                bool value;
                if ( bool.TryParse(node.Attributes["enabled"].InnerText,
                                   out value)==false )
                {
                    throw new System.Xml.XmlException( "\"enabled\" attribute is missing or of a wrong type in <key_frame_animation>." );
                }
                data.IsEnabled = value;
            }

            // "loop" attribute
            {
                bool value;
                if ( bool.TryParse(node.Attributes["loop"].InnerText,
                                   out value)==false )
                {
                    throw new System.Xml.XmlException( "\"loop\" attribute is missing or of a wrong type in <key_frame_animation>." );
                }
                data.IsLoop = value;
            }

            // "time_based" attribute
            {
                AnimTimeBaseTypes value;
                if ( Enum.TryParse<AnimTimeBaseTypes>(node.Attributes["time_based"].InnerText,
                                                      out value)==false )
                {
                    throw new System.Xml.XmlException( "\"time_based\" attribute is missing or of a wrong type in <key_frame_animation>." );
                }
                data.TimeBaseType = value;
            }

            // "interpolation" attribute
            {
                AnimInterpolationTypes value;
                if ( Enum.TryParse<AnimInterpolationTypes>(node.Attributes["interpolation"].InnerText,
                                                           out value)==false )
                {
                    throw new System.Xml.XmlException( "\"interpolation\" attribute is missing or of a wrong type in <key_frame_animation>." );
                }
                data.InterpolationType = value;
            }

            #endregion

            // Parse the key frames
            foreach ( XmlNode child in node.ChildNodes )
            {
                if ( child!=null && child.Name=="key" )
                {
                    // Create the key frame data of the correct data type.
                    AnimTableKeyFrameData keyData =
                        this.AnimationList.CreateAnimationTableKeyFrame( data.Name );

                    // Parse the key frame data from XML node.
                    if ( ParseKeyFrame(child,
                                       ref keyData)==true )
                    {
                        data.KeyFrames.Add( keyData );
                    }
                }
            }

            return data;
        }


        /// <summary>
        /// Parse the key frame data from the given XML node.
        /// </summary>
        /// <param name="node">The XML node.</param>
        /// <param name="data">The key frame data.</param>
        /// <returns>True on success.</returns>
        private bool ParseKeyFrame( XmlNode node,
                                    ref AnimTableKeyFrameData data )
        {
            // "x" attribute ( frame number of the key frame )
            {
                int iKeyFrame;
                if ( int.TryParse( node.Attributes["x"].InnerText,
                                   NumberStyles.Number,
                                   CultureInfo.InvariantCulture,
                                   out iKeyFrame )==false )
                {
                    throw new System.Xml.XmlException( "\"x\" attribute is missing or of a wrong type in <key>." );
                }
                data.KeyFrame = iKeyFrame;
            }

            // "y" attribute ( animation value of the key frame )
            {
                data.StrValue = node.Attributes["y"].InnerText;
            }

            return true;
        }

        #endregion

        #region Write XML

        /// <summary>
        /// Write the animation table data to XML.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        /// <param name="data">The animation table data.</param>
        private void WriteAnimationTableNode( XmlWriter writer,
                                              AnimTableData data )
        {
            if ( writer==null )
                return;

            if ( data==null )
                return;

            writer.WriteStartElement( "key_frame_animation" );

            #region Write attributes

            // "time_based" attribute
            writer.WriteAttributeString( "time_based",
                                         data.TimeBaseType.ToString() );

            // "name" attribute
            writer.WriteAttributeString( "name",
                                         data.Name.ToString() );

            // "enabled" attribute
            writer.WriteAttributeString( "enabled",
                                         data.IsEnabled.ToString() );

            // "num_keys" attribute
            writer.WriteAttributeString( "num_keys",
                                         data.KeyFrames.Count.ToString() );

            // "interpolation" attribute
            writer.WriteAttributeString( "interpolation",
                                         data.InterpolationType.ToString() );

            // "loop" attribute
            writer.WriteAttributeString( "loop",
                                         data.IsLoop.ToString() );

            #endregion

            foreach ( AnimTableKeyFrameData keyData in data.KeyFrames )
            {
                WriteKeyFrameNode( writer, keyData );
            }

            writer.WriteEndElement();
        }


        /// <summary>
        /// Write the animation key frame data to XML.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        /// <param name="data">The animation key frame data.</param>
        private void WriteKeyFrameNode( XmlWriter writer,
                                        AnimTableKeyFrameData data )
        {
            if ( writer==null )
                return;

            if ( data==null )
                return;

            writer.WriteStartElement( "key" );

            #region Write attributes

            // "x" attribute ( the frame number of the key frame )
            writer.WriteAttributeString( "x",
                                         data.KeyFrame.ToString( CultureInfo.InvariantCulture ) );

            // "y" attribute ( the animation value of the key frame )
            writer.WriteAttributeString( "y",
                                         data.StrValue );

            #endregion

            writer.WriteEndElement();
        }

        #endregion
    }

    #region The animation table list serializer for emitters

    public class EmitterAnimTableListXml : AnimTableListXml
    {
        #region Constructors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public EmitterAnimTableListXml()
        {
            this.AnimationList = new EmitterAnimTableList();
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="list">The animation table list.</param>
        public EmitterAnimTableListXml( EmitterAnimTableList list )
        {
            this.AnimationList = list;
        }

        #endregion
    }

    #endregion
}
