﻿// ========================================================================
// <copyright file="EnvConfigData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System.Collections.Generic;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// 環境設定データ
    /// </summary>
    public sealed class EnvConfigData : BaseDataModel<EnvConfigData>
    {
        #region Constructors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public EnvConfigData()
        {
            this.PreviewConfigData   = new PreviewConfigData();
            this.ClearColor          = new RgbaColor();
            this.BackgroundImagePath = string.Empty;
            this.BGImageMultColor    = new RgbaColor();
            this.CameraPos           = new Vector3( 0.0f, 50.0f, 150.0f );
            this.CameraLookAt        = new Vector3( 0.0f, 0.0f, 0.0f );
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public EnvConfigData(EnvConfigData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.PreviewConfigData.Initialize();

            #if BUILD_FOR_CTR
            this.ClearColor.Set( 0.2f, 0.2f, 0.2f, 1.0f );
            #else
            this.ClearColor.Set( 0.03f, 0.03f, 0.03f, 1.0f );
            #endif

            this.BackgroundImagePath = string.Empty;
            this.BGImageMultColor.Set( 1.0f, 1.0f, 1.0f, 1.0f );

            #if BUILD_FOR_CTR
            this.CameraPerspectiveAspectRatio  = 1.66f;
            this.CameraOrthogonalAspectRatio   = 1.66f;
            #else
            this.CameraPerspectiveAspectRatio  = 1.77f;
            this.CameraOrthogonalAspectRatio   = 1.77f;
            #endif

            this.CameraRotation                = 0.0f;
            this.CameraProjType                = CameraProjTypes.Perspective;
            this.CameraPerspectiveFOV          = 1.0472f; // 60 degrees.
            this.CameraOrthogonalFrustumHeight = 72.0f;
            this.CameraFrustumNear             = 0.1f;
            this.CameraFrustumFar              = 1000.0f;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 設定
        /// </summary>
        public override void Set(EnvConfigData source)
        {
            Ensure.Argument.NotNull(source);

            this.PreviewConfigData.Set(source.PreviewConfigData);
            this.ClearColor.Set( source.ClearColor );
            this.BackgroundImagePath = source.BackgroundImagePath;
            this.BGImageMultColor.Set( source.BGImageMultColor );

            this.CameraPos.Set( source.CameraPos );
            this.CameraLookAt.Set( source.CameraLookAt );
            this.CameraRotation                = source.CameraRotation;
            this.CameraProjType                = source.CameraProjType;
            this.CameraPerspectiveAspectRatio  = source.CameraPerspectiveAspectRatio;
            this.CameraPerspectiveFOV          = source.CameraPerspectiveFOV;
            this.CameraOrthogonalAspectRatio   = source.CameraOrthogonalAspectRatio;
            this.CameraOrthogonalFrustumHeight = source.CameraOrthogonalFrustumHeight;
            this.CameraFrustumNear             = source.CameraFrustumNear;
            this.CameraFrustumFar              = source.CameraFrustumFar;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override EnvConfigData Clone()
        {
            return new EnvConfigData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// プレビュー設定データモデル
        /// </summary>
        public PreviewConfigData PreviewConfigData { get; set; }

        /// <summary>
        /// クリアカラー
        /// </summary>
        public RgbaColor ClearColor { get; set; }

        /// <summary>
        /// 背景画像のファイルパス
        /// </summary>
        public string BackgroundImagePath { get; set; }

        /// <summary>
        /// クリアカラー
        /// </summary>
        public RgbaColor BGImageMultColor { get; set; }

        /// <summary>
        /// カメラの位置。
        /// </summary>
        public Vector3 CameraPos { get; set; }

        /// <summary>
        /// カメラの注視点。
        /// </summary>
        public Vector3 CameraLookAt { get; set; }

        /// <summary>
        /// カメラの回転。
        /// </summary>
        public float CameraRotation { get; set; }

        /// <summary>
        /// カメラタイプ。
        /// </summary>
        public CameraProjTypes CameraProjType { get; set; }

        /// <summary>
        /// アスペクト比。(Perspective projection)
        /// </summary>
        public float CameraPerspectiveAspectRatio { get; set; }

        /// <summary>
        /// 画角。(Perspective projection)
        /// </summary>
        public float CameraPerspectiveFOV { get; set; }

        /// <summary>
        /// オルソアスペクト比。(Orthogonal projection)
        /// </summary>
        public float CameraOrthogonalAspectRatio { get; set; }

        /// <summary>
        /// Frustum height.(Orthogonal projection)
        /// </summary>
        public float CameraOrthogonalFrustumHeight { get; set; }

        /// <summary>
        /// Frustum near plane.
        /// </summary>
        public float CameraFrustumNear { get; set; }

        /// <summary>
        /// Frustum far plane.
        /// </summary>
        public float CameraFrustumFar { get; set; }

        #endregion
    }
}
