﻿// ========================================================================
// <copyright file="PreviewConfigData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0;
using NWCore.Serializer.Major_1.Minor_5.Build_0.Revision_0;

namespace NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0
{
    /// <summary>
    /// プレビュー設定データ
    /// </summary>
    [Serializable]
    public sealed class PreviewConfigData : BaseDataModel<PreviewConfigData>
    {
        #region Constructors

        /// <summary>
        /// プレビューモデルの最少数
        /// </summary>
        public const int MinPreviewModelCount = 5;

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public PreviewConfigData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public PreviewConfigData(PreviewConfigData source)
            : this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.InfinitePreview = false;
            this.MaxPreviewFrame = 180;
            this.AutoTranslateType = NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0.AutoTranslateType.None;
            this.TranslateRadiusXZ = 8.0f;
            this.TranslateVelXZ = 1.0f;
            this.VibrationAmplitudeY = 0.0f;
            this.VibrationVelY = 5.0f;
            this.ZRollVel = 0.0f;
            this.CameraHorizontalDegree = 336.0f;
            this.CameraVerticalDegree = -20.0f;
            this.CameraHorizontalVel = 0.0f;
            this.CameraRotateRadius = 22.0f;
            this.CameraVerticalViewAngle = 90.0f;

            this.BackgroundImageFileName = "";
            this.BackgroundImageVertexColor = new RgbaColor(1.0f, 1.0f, 1.0f, 1.0f);
//            this.BackgroundColorColor = new RgbaColor(0.2f, 0.2f, 0.2f, 0.0f);

            this.LightDir = new Vector3(-0.4f, -0.8f, -0.4f);
            this.LightDiffuse = new Vector3(1.0f, 1.0f, 1.0f);
            this.LightAmbient = new Vector3(0.2f, 0.2f, 0.2f);
            this.LightSpecular0 = new Vector3(1.0f, 1.0f, 1.0f);
            this.LightSpecular1 = new Vector3(1.0f, 1.0f, 1.0f);
            this.HLightDir = new Vector3(0.0f, 1.0f, 0.0f);
            this.HLightSkyColor = new Vector3(0.9f, 1.0f, 1.0f);
            this.HLightGroundColor = new Vector3(0.1f, 0.0f, 0.0f);

            this.SkyColorWeightXZ = 0.5f;
            this.FogType = NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0.FogType.None;
            this.FogColor = new Vector3(1.0f, 1.0f, 1.0f);
            this.FogBeginDistance = 0.0f;
            this.FogEndDistance = 200.0f;
            this.FogDensity = 2.0f;

            this.PreviewModels = new List<PreviewModelData>( MinPreviewModelCount );
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source"></param>
        public override void Set(PreviewConfigData source)
        {
            Ensure.Argument.NotNull(source);

            this.InfinitePreview = source.InfinitePreview;
            this.MaxPreviewFrame = source.MaxPreviewFrame;
            this.AutoTranslateType = source.AutoTranslateType;
            this.TranslateRadiusXZ = source.TranslateRadiusXZ;
            this.TranslateVelXZ = source.TranslateVelXZ;
            this.VibrationAmplitudeY = source.VibrationAmplitudeY;
            this.VibrationVelY = source.VibrationVelY;
            this.ZRollVel = source.ZRollVel;
            this.CameraHorizontalDegree = source.CameraHorizontalDegree;
            this.CameraVerticalDegree = source.CameraVerticalDegree;
            this.CameraHorizontalVel = source.CameraHorizontalVel;
            this.CameraRotateRadius = source.CameraRotateRadius;
            this.CameraVerticalViewAngle = source.CameraVerticalViewAngle;

            this.BackgroundImageFileName = source.BackgroundImageFileName;
            this.BackgroundImageVertexColor.Set(source.BackgroundImageVertexColor);
//            this.BackgroundColorColor.Set(source.BackgroundColorColor);

            this.LightDir.Set(source.LightDir);
            this.LightDiffuse.Set(source.LightDiffuse);
            this.LightAmbient.Set(source.LightAmbient);
            this.LightSpecular0.Set(source.LightSpecular0);
            this.LightSpecular1.Set(source.LightSpecular1);
            this.HLightDir.Set(source.HLightDir);
            this.HLightSkyColor.Set(source.HLightSkyColor);
            this.HLightGroundColor.Set(HLightGroundColor);

            this.SkyColorWeightXZ = source.SkyColorWeightXZ;
            this.FogType = source.FogType;
            this.FogColor.Set(source.FogColor);
            this.FogBeginDistance = source.FogBeginDistance;
            this.FogEndDistance = source.FogEndDistance;
            this.FogDensity = source.FogDensity;

            // Compute the array capacity we need for copying the model data.
            int iNumOrigModels = this.PreviewModels.Count;
            int iNumTargModels = source.PreviewModels.Count;
            int iArrayCapacity = System.Math.Min( iNumTargModels, MinPreviewModelCount );

            // Allocate the array first, this can make the copy faster.
            if ( iNumOrigModels!=iArrayCapacity )
                this.PreviewModels.Capacity = iNumTargModels;

            // Copy the model data
            int i;
            for ( i=0;i<iNumTargModels;++i )
            {
                if ( i<this.PreviewModels.Count )
                    this.PreviewModels[i].Set( source.PreviewModels[i] );
                else
                    this.PreviewModels.Add( new PreviewModelData(source.PreviewModels[i]) );
            }

            // The number of the source model data is smaller than the minimum number,
            // reset the rest of the model data in the array.
            if ( iArrayCapacity>iNumTargModels )
            {
                for ( i=iNumTargModels;i<iArrayCapacity;++i )
                {
                    if ( i<this.PreviewModels.Count )
                    {
                        PreviewModelData data = this.PreviewModels[i];
                        if ( data==null )
                            this.PreviewModels[i] = new PreviewModelData();
                        else
                            data.Initialize();
                    }
                    else
                    {
                        this.PreviewModels.Add( new PreviewModelData() );
                    }
                }
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override PreviewConfigData Clone()
        {
            return new PreviewConfigData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 無限プレビュー
        /// </summary>
        public bool InfinitePreview { get; set; }

        /// <summary>
        /// プレビュー最大フレーム
        /// </summary>
        public int MaxPreviewFrame { get; set; }

        /// <summary>
        /// 児童移動タイプ
        /// </summary>
        public AutoTranslateType AutoTranslateType { get; set; }

        /// <summary>
        /// XZ移動半径
        /// </summary>
        public float TranslateRadiusXZ { get; set; }

        /// <summary>
        /// XZ移動速度
        /// </summary>
        public float TranslateVelXZ { get; set; }

        /// <summary>
        /// Y揺れ振幅
        /// </summary>
        public float VibrationAmplitudeY { get; set; }

        /// <summary>
        /// Y揺れ速度
        /// </summary>
        public float VibrationVelY { get; set; }

        /// <summary>
        /// Zロール速度
        /// </summary>
        public float ZRollVel { get; set; }

        /// <summary>
        /// 横回転角度
        /// </summary>
        public float CameraHorizontalDegree { get; set; }

        /// <summary>
        /// 縦回転角度
        /// </summary>
        public float CameraVerticalDegree { get; set; }

        /// <summary>
        /// 横回転角速度
        /// </summary>
        public float CameraHorizontalVel { get; set; }

        /// <summary>
        /// 回転半径
        /// </summary>
        public float CameraRotateRadius { get; set; }

        /// <summary>
        /// 垂直画角
        /// </summary>
        public float CameraVerticalViewAngle { get; set; }

        /// <summary>
        /// 背景画像
        /// </summary>
        public string BackgroundImageFileName { get; set; }

        /// <summary>
        /// 背景画像頂点カラー
        /// </summary>
        public RgbaColor BackgroundImageVertexColor { get; set; }

        /// <summary>
        /// 背景クリアカラー
        /// </summary>
//        public RgbaColor BackgroundColorColor { get; set; }

        /// <summary>
        /// ライト方向
        /// </summary>
        public Vector3 LightDir { get; set; }

        /// <summary>
        /// ライトディフューズカラー
        /// </summary>
        public Vector3 LightDiffuse { get; set; }

        /// <summary>
        /// アンビエントカラー
        /// </summary>
        public Vector3 LightAmbient { get; set; }

        /// <summary>
        /// スペキュラカラー0
        /// </summary>
        public Vector3 LightSpecular0 { get; set; }

        /// <summary>
        /// スペキュラカラー1
        /// </summary>
        public Vector3 LightSpecular1 { get; set; }

        /// <summary>
        /// 半球ライト方向
        /// </summary>
        public Vector3 HLightDir { get; set; }

        /// <summary>
        /// 半球ライトスカイカラー
        /// </summary>
        public Vector3 HLightSkyColor { get; set; }

        /// <summary>
        /// 半球ライトグラウンドカラー
        /// </summary>
        public Vector3 HLightGroundColor { get; set; }

        /// <summary>
        /// XZスカイカラーウェイト
        /// </summary>
        public float SkyColorWeightXZ { get; set; }

        /// <summary>
        /// フォグタイプ
        /// </summary>
        public FogType FogType { get; set; }

        /// <summary>
        /// フォグカラー
        /// </summary>
        public Vector3 FogColor { get; set; }

        /// <summary>
        /// フォグ開始距離
        /// </summary>
        public float FogBeginDistance { get; set; }

        /// <summary>
        /// フォグ終了距離
        /// </summary>
        public float FogEndDistance { get; set; }

        /// <summary>
        /// フォグ濃度
        /// </summary>
        public float FogDensity { get; set; }

        /// <summary>
        /// プレビューモデル
        /// </summary>
        public List<PreviewModelData> PreviewModels { get; set; }

        #endregion

        #region Methods for accessing the model data

        /// <summary>
        /// Get the model data at the given index.
        /// </summary>
        /// <param name="iIndex">The index to the model data.</param>
        /// <param name="bCreateModelData">Create model data if the index is out of range.</param>
        /// <returns>The model data or null if the index is out of range.</returns>
        public PreviewModelData GetModelData( int iIndex,
                                              bool bCreateModelData = false )
        {
            int iModelIndex = iIndex;
            if ( iIndex<0 || iIndex>=this.PreviewModels.Count )
            {
                if ( bCreateModelData==false )
                    return null;

                if ( iIndex<0 )
                {
                    iModelIndex = this.PreviewModels.Count;
                    this.PreviewModels.Add( new PreviewModelData() );
                }
                else
                {
                    int iOrigArraySize = this.PreviewModels.Count;
                    int iArrayCapacity = iIndex + 1;
                    this.PreviewModels.Capacity = iArrayCapacity;
                    for ( int i=iOrigArraySize;i<iArrayCapacity;++i )
                    {
                        this.PreviewModels.Add( new PreviewModelData() );
                    }
                }
            }

            return this.PreviewModels[iModelIndex];
        }

        #endregion
    }
}
