﻿// ========================================================================
// <copyright file="InitConfigData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0;

namespace NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0
{
    /// <summary>
    /// 初期設定データ
    /// </summary>
    [Serializable]
    [XmlRoot]
    public sealed class InitSettingsData : BaseDataModel<InitSettingsData>
    {
        #region Constructors

        private static XmlSerializer s_serializer = new XmlSerializer(typeof(InitSettingsData));

        private const int USER_DATA_BIT_LENGTH = 16;
        private const int USER_NUMERICAL_VALUE_COUNT = 2;
        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public InitSettingsData()
        {
            this.WinProcessNames = new List<string>();
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public InitSettingsData(InitSettingsData source)
            : this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 最新のバージョンかチェックします。
        /// </summary>
        /// <returns>=true..最新版</returns>
        public bool IsLatestVersion()
        {
            return this.Version.Equals(STR_VERSION);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Version = STR_VERSION;

            this.EditEnvFilename = String.Empty;
            this.FrameRate = InitSettingFrameRate.FPS60;
            this.Resolution = InitSettingResolution.RES_960_540;
            this.ClearColor = new RgbaColor(0.2f, 0.2f, 0.2f, 0.0f);
            this.WorldScaleRange = 100.0f;
            this.WorldScaleTime = 1000;
            this.GridScale = 1.0f;
            this.BitLabel = new string[USER_DATA_BIT_LENGTH];
            for (int i=0; i<USER_DATA_BIT_LENGTH; ++i)
            {
                this.BitLabel[i] = "[" + i + "]";
            }
            this.NumberLabel = new string[USER_NUMERICAL_VALUE_COUNT];
            for (int i = 0; i < USER_NUMERICAL_VALUE_COUNT; ++i)
            {
                this.NumberLabel[i] =
                    res.Strings.INIT_CONFIG_DATA_PROPERTY_USER_DATA_NUMERICAL_VALUE_LABEL + i;
            }
            this.WinProcessNames.Clear();
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        public override void Set(InitSettingsData source)
        {
            this.Version = source.Version;

            this.EditEnvFilename = source.EditEnvFilename;
            this.FrameRate = source.FrameRate;
            this.Resolution = source.Resolution;
            this.ClearColor.Set(source.ClearColor);
            this.WorldScaleRange = source.WorldScaleRange;
            this.WorldScaleTime = source.WorldScaleTime;
            this.GridScale = source.GridScale;
            for (int i = 0 ; i < USER_DATA_BIT_LENGTH; ++i)
            {
                this.BitLabel[i] = source.BitLabel[i];
            }
            for (int i=0; i<USER_NUMERICAL_VALUE_COUNT; ++i)
            {
                this.NumberLabel[i] = source.NumberLabel[i];
            }
            this.WinProcessNames.Clear();
            foreach (string item in source.WinProcessNames)
            {
                this.WinProcessNames.Add(item);
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override InitSettingsData Clone()
        {
            return new InitSettingsData(this);
        }

        #endregion

        #region シリアライズ

        /// <summary>
        /// 指定ファイルに保存
        /// </summary>
        public static bool SaveTo(string filePath, InitSettingsData data)
        {
            using (FileStream file =
                new FileStream(filePath, FileMode.Create, FileAccess.Write, FileShare.Write))
            {
                try
                {
                    s_serializer.Serialize(file, data);
                    file.Flush();
                }
                catch (Exception ex)
                {
                    DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// 指定ファイルから読み込み
        /// </summary>
        public static InitSettingsData LoadFrom(string filePath)
        {
            InitSettingsData data = null;
            using (FileStream file =
                new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                try
                {
                    data = s_serializer.Deserialize(file) as InitSettingsData;
                }
                catch (Exception ex)
                {
                    DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                    data = null;
                }
            }
            return data;
        }

        #endregion

        // >>プロパティ追加削除を変更したら番号を上げてください<<
        private const string STR_VERSION = "1.0.0";

        /// <summary>
        /// Iniファイルのバージョン
        /// </summary>
        public string Version { get; set; }

        /// <summary>
        /// Win版用のビューアプロセス名
        /// </summary>
        public List<string> WinProcessNames { get; set; }

        /// <summary>
        /// 自動で読み込むエディット環境ファイル
        /// </summary>
        public string EditEnvFilename { get; set; }

        /// <summary>
        /// フレームレート
        /// </summary>
        public InitSettingFrameRate FrameRate { get; set; }

        /// <summary>
        /// 解像度
        /// </summary>
        public InitSettingResolution Resolution { get; set; }

        /// <summary>
        /// クリアカラー
        /// </summary>
        public RgbaColor ClearColor { get; set; }

        /// <summary>
        /// ワールドスケール 位置・速度・サイズレンジ
        /// </summary>
        public float WorldScaleRange
        {
            get { return s_WorldScaleRange; }
            set { s_WorldScaleRange = value; }
        }

        /// <summary>
        /// ワールドスケール 時間
        /// </summary>
        public int WorldScaleTime { get { return s_WorldScaleTime; } set { s_WorldScaleTime = value;} }

        /// <summary>
        /// ワールドスケール グリッドと床の表示スケール
        /// </summary>
        public float GridScale { get; set; }

        /// <summary>
        /// ユーザーデータビットのラベル
        /// </summary>
        public string[] BitLabel { get; set; }

        /// <summary>
        /// ユーザーデータ数値のラベル
        /// </summary>
        public string[] NumberLabel { get; set; }
    }
}
