﻿// ========================================================================
// <copyright file="ChildData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_0.Minor_9.Build_0.Revision_0
{
    /// <summary>
    /// チャイルド
    /// </summary>
    public sealed class ChildData : BaseDataModel<ChildData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ChildData()
        {
            this.Color = new RgbaColor();        // 色
            this.TextureRes = new TextureRes();  // チャイルド用テクスチャ
            this.Scale = new Vector2(1, 1); ;    // スケール値
            this.RandVel = new Vector3();        // ランダム速度
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ChildData(ChildData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.AirRegist = 1.0f;
            this.Alpha = 1.0f;
            this.EmitRate = 1;
            this.EmitStep = 10;
            this.EmitTiming = 60;
            this.ScaleInheritRate = 1.0f;
            this.VelInheritRate = 0.0f;
            this.VelInheritRate = 1.0f;
            this.UseSortParticle = ShaderType.Particle;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;
            this.RandomScale = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * worldScale;
                this.Scale.Set(1*worldScale, 1*worldScale);
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ChildData source)
        {
            Ensure.Argument.NotNull(source);

            this.Color.Set(source.Color);
            this.TextureRes.Set(source.TextureRes);
            this.Scale.Set(source.Scale);
            this.RandVel.Set(source.RandVel);

            this.AirRegist = source.AirRegist;
            this.Alpha = source.Alpha;
            this.AlphaInit = source.AlphaInit;
            this.AlphaTarget = source.AlphaTarget;
            this.BillboardType = source.BillboardType;
            this.CombinerType = source.CombinerType;
            this.EmitRate = source.EmitRate;
            this.EmitStep = source.EmitStep;
            this.EmitTiming = source.EmitTiming;
            this.FigureVelocity = source.FigureVelocity;
            this.InitPosRand = source.InitPosRand;
            this.RandomScale = source.RandomScale;
            this.ScaleInheritRate = source.ScaleInheritRate;
            this.VelInheritRate = source.VelInheritRate;
            this.ZBufATestType = source.ZBufATestType;
            this.UseSortParticle = source.UseSortParticle;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ChildData Clone()
        {
            return new ChildData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 放出量
        /// </summary>
        public int EmitRate { get; set; }

        /// <summary>
        /// タイミング
        /// </summary>
        public int EmitTiming { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public int EmitStep { get; set; }

        /// <summary>
        /// 速度継承率
        /// </summary>
        public float VelInheritRate { get; set; }

        /// <summary>
        /// 形状速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// ランダム速度
        /// </summary>
        public Vector3 RandVel { get; set; }

        /// <summary>
        /// 初期位置のランダム量
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// ビルボードタイプ
        /// </summary>
        public BillboardType BillboardType { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// チャイルド用テクスチャ
        /// </summary>
        public TextureRes TextureRes { get; set; }

        /// <summary>
        /// 色
        /// </summary>
        public RgbaColor Color { get; set; }

        /// <summary>
        /// アルファ
        /// </summary>
        public float Alpha { get; set; }

        /// <summary>
        /// アルファアニメのターゲット
        /// </summary>
        public float AlphaTarget { get; set; }

        /// <summary>
        /// アルファアニメの初期値
        /// </summary>
        public float AlphaInit { get; set; }

        /// <summary>
        /// スケール継承率
        /// </summary>
        public float ScaleInheritRate { get; set; }

        /// <summary>
        /// スケール値
        /// </summary>
        public Vector2 Scale { get; set; }

        /// <summary>
        /// ランダム値
        /// </summary>
        public float RandomScale { get; set; }

        /// <summary>
        /// コンバイナタイプ
        /// </summary>
        public CombinerType CombinerType { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public ShaderType UseSortParticle { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        #endregion
    }
}
