﻿// ========================================================================
// <copyright file="DocumentCreator.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
//using App.Command;
using App.Data;
using NWCore.DataModel;

namespace App.IO
{
    /// <summary>
    /// ドキュメントクリエータ。
    /// </summary>
    public static class DocumentCreator
    {
        #region Utility methods

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        private static int FindUnusedSerialNumber( List<int> usedSerialNumList )
        {
            int  iSerial = 1;
            bool bFound  = true;
            while ( bFound==true )
            {
                bFound = false;
                foreach ( int iNum in usedSerialNumList )
                {
                    if ( iNum==iSerial )
                    {
                        ++iSerial;
                        bFound = true;
                        break;
                    }
                }
            }

            return iSerial;
        }

        #endregion

        #region Create default document name

        /// <summary>
        /// Create default project name.
        /// </summary>
        /// <param name="owner">Owner of the document to create.</param>
        /// <returns>Name for the document.</returns>
        public static string CreateDefaultProjectName()
        {
            string docName = res.Strings.DEFAULT_ENV_NAME;

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( IProjectDocument existingDoc in ProjectManager.Projects )
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }


        /// <summary>
        /// Create default emitter set name.
        /// </summary>
        /// <param name="owner">Owner of the document to create.</param>
        /// <returns>Name for the document.</returns>
        public static string CreateDefaultEmitterSetName( IProjectDocument owner )
        {
            string docName = res.Strings.DEFAULT_EMITTERSET_NAME;

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( EmitterSetDocument existingDoc in owner.EmitterSetDocuments )
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }


        /// <summary>
        /// Create default emitter name.
        /// </summary>
        /// <param name="owner">Owner of the document to create.</param>
        /// <returns>Name for the document.</returns>
        public static string CreateDefaultEmitterName( IEmitterSetDocument owner )
        {
            string docName = Postfix.EmitterPostfix;

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( EmitterDocument existingDoc in owner.EmitterDocuments )
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }


        /// <summary>
        /// Create default game settings document name.
        /// </summary>
        /// <param name="owner">Owner of the document to create.</param>
        /// <returns>Name for the document.</returns>
        public static string CreateDefaultGameSettingsDocName( IEmitterSetDocument owner )
        {
            string docName = owner.Name;

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "_([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( GameSettingsDocument existingDoc in owner.GameSettingsDocuments )
            {
                if ( existingDoc.Name==docName )
                {
                    existingSerialList.Add( 1 );
                }
                else
                {
                    System.Text.RegularExpressions.Match match =
                        regexCaptureSerial.Match( existingDoc.Name );
                    if ( match.Groups.Count<2 )
                        continue;

                    int iCurrSerial;
                    if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                        existingSerialList.Add( iCurrSerial );
                }
            }

            // Add the serial number to the document name
            int iSerial = FindUnusedSerialNumber( existingSerialList );
            if ( iSerial>1 )
                docName += "_" + iSerial.ToString();

            return docName;
        }

        #endregion

        #region Create copied document name

        /// <summary>
        /// Compose copied emitter set name.
        /// </summary>
        /// <param name="owner">Owner of the destination document.</param>
        /// <param name="srcDocumentName">Source document name.</param>
        /// <returns>Name for the destination document.</returns>
        public static string ComposeCopiedEmitterSetName( IProjectDocument owner,
                                                          string srcDocumentName )
        {
            string docName = srcDocumentName + "_Copy";

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( EmitterSetDocument existingDoc in owner.EmitterSetDocuments )
            {
                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }


        /// <summary>
        /// Compose copied emitter name.
        /// </summary>
        /// <param name="owner">Owner of the destination document.</param>
        /// <param name="srcDocumentName">Source document name.</param>
        /// <param name="skipDoc">The document to skip when looking for the document name.</param>
        /// <returns>Name for the destination document.</returns>
        public static string ComposeCopiedEmitterName( IEmitterSetDocument owner,
                                                       string srcDocumentName,
                                                       IEmitterDocument skipDoc = null )
        {
            IEmitterDocument foundDoc =
                (owner as EmitterSetDocument).FindEmitterDocument(srcDocumentName);
            if ( foundDoc==null || foundDoc==skipDoc )
                return srcDocumentName;

            string docName = srcDocumentName + "_Copy";

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( EmitterDocument existingDoc in owner.EmitterDocuments )
            {
                if ( skipDoc==existingDoc )
                    continue;

                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }

        /// <summary>
        /// Compose copied game settings document name.
        /// </summary>
        /// <param name="owner">Owner of the destination document.</param>
        /// <param name="srcDocumentName">Source document name.</param>
        /// <param name="skipDoc">The document to skip when looking for the document name.</param>
        /// <returns>Name for the destination document.</returns>
        public static string ComposeCopiedGameSettingsDocName( IEmitterSetDocument owner,
                                                               string srcDocumentName,
                                                               IGameSettingsDocument skipDoc = null )
        {
            GameSettingsDocument foundDoc =
                (owner as EmitterSetDocument).FindGameSettingsDocument(srcDocumentName);
            if ( foundDoc==null || foundDoc==skipDoc )
                return srcDocumentName;

            string docName = srcDocumentName + "_Copy";

            // Regular expression for extracting serial number in the document name.
            System.Text.RegularExpressions.Regex regexCaptureSerial =
                new System.Text.RegularExpressions.Regex( "^" + docName + "([0-9]+)$" );

            List<int> existingSerialList = new List<int>();

            // Collect all the existing serial numbers
            foreach ( GameSettingsDocument existingDoc in owner.GameSettingsDocuments )
            {
                if ( skipDoc==existingDoc )
                    continue;

                System.Text.RegularExpressions.Match match =
                    regexCaptureSerial.Match( existingDoc.Name );
                if ( match.Groups.Count<2 )
                    continue;

                int iCurrSerial;
                if ( int.TryParse( match.Groups[1].Value, out iCurrSerial )==true )
                    existingSerialList.Add( iCurrSerial );
            }

            // Add the serial number to the document name
            docName += FindUnusedSerialNumber( existingSerialList ).ToString();

            return docName;
        }

        #endregion

        #region IProjectDocumentの生成

        //---------------------------------------------------------------------
        /// <summary>
        /// プロジェクトを作成する。（名前自動生成）
        /// </summary>
        public static IProjectDocument CreateProject()
        {
            // 名前を決める
            string name = CreateDefaultProjectName();
            // プロジェクト生成
            return CreateProject(name);
        }

        /// <summary>
        /// プロジェクトを作成する。
        /// </summary>
        public static IProjectDocument CreateProject(string name)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            // プロジェクト生成
            IProjectDocument project = CreateProject(name, changed);
            // 変更通知
            Document.NotifyComponentChanged(null, changed);
            return project;
        }

        /// <summary>
        /// プロジェクトを作成する。
        /// </summary>
        public static IProjectDocument CreateProject(
            string name, DocumentComponentChangedEventArgs changed)
        {
            // プロジェクトドキュメントの生成
            IProjectDocument project = new EffectProjectDocument(name);
            changed.AddOpenedDocument(project);
            return project;
        }

        #endregion
    }
}
