﻿// ========================================================================
// <copyright file="DocumentCloser.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
//using App.Controls;
using App.Data;
using NWCore.DataModel;
using NWCore.Viewer;

namespace App.IO
{
    /// <summary>
    /// ドキュメントクローザ。
    /// </summary>
    public class DocumentCloser
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// 閉じる。
        /// </summary>
        public bool Close(IDocument document)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            bool result = CloseWithChildren(document, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// 閉じる（TemplateMethod）。
        /// </summary>
        public virtual bool Close(IDocument document, DocumentComponentChangedEventArgs changed)
        {
            IProjectDocument project = document as IProjectDocument;
            if (project != null)
            {
                List<IDocument> documents = new List<IDocument>();
                foreach (IDocument child in project.SubDocuments)
                {
                    if (System.String.IsNullOrEmpty(child.FileExt) == false)
                    {
                        documents.Add(child);
                    }
                }
                bool bResult = Close(documents);

                if ( bResult==true )
                    project.Reset();

                return bResult;
            }
            else
            {
                // 他のドキュメントから参照されていないか
                if (!CheckReference(document))
                {
                    ErrorFileIsReferenced(document.FileName);
                    return false;
                }
                // 編集状態でないか
                if (!CheckModified(document))
                {
                    return false;
                }
                return CloseDocument(document, changed);
            }
        }

        /// <summary>
        /// 参照チェック。
        /// </summary>
        protected bool CheckReference(IDocument document)
        {
            return true;
        }

        /// <summary>
        /// 編集チェック。
        /// </summary>
        private bool CheckModified(IDocument document)
        {
            return false;

            /*
            if ( document.Modified==false )
                return true;

            DialogResult result =
                ThreadSafeMsgBox.Show( document.FileName + res.Strings.DIALOG_MSG_CONFIRM_SAVE_CHANGES,
                                       res.Strings.CONFIRM_CAPTION,
                                       System.Windows.Forms.MessageBoxButtons.YesNoCancel,
                                       System.Windows.Forms.MessageBoxIcon.Question );
            if ( result==DialogResult.Cancel )
                return false;
            else if ( result==DialogResult.No )
                return true;

            DocumentSaver saver = new DocumentSaver();
            if ( saver.Save( document,
                             document.FilePath )==false )
            {
                return false;
            }

            return true;
            */
        }

        /// <summary>
        /// 編集チェック。
        /// </summary>
        protected bool CheckModified(List<IDocument> documents)
        {
            return false;

            /*
            // 未保存ファイルの洗い出し
            List<IDocument> overwriteDocList = new List<IDocument>();
            List<IDocument> pathEmptyDocList = new List<IDocument>();
            foreach ( IDocument doc in documents )
            {
                if ( doc.Modified==true )
                {
                    if ( System.IO.Path.IsPathRooted( doc.FilePath )==true )
                        overwriteDocList.Add( doc );
                    else
                        pathEmptyDocList.Add( doc );
                }
            }

            // 未保存確認
            bool result = true;

            // Save the files those had already been saved ( save location is set ).
            if ( overwriteDocList.Count>0 )
            {
                foreach ( IDocument doc in overwriteDocList )
                {
                    DialogResult dlgResult =
                        ThreadSafeMsgBox.Show( doc.FileName + res.Strings.DIALOG_MSG_CONFIRM_SAVE_CHANGES,
                                               res.Strings.CONFIRM_CAPTION,
                                               System.Windows.Forms.MessageBoxButtons.YesNoCancel,
                                               System.Windows.Forms.MessageBoxIcon.Question );
                    if ( dlgResult==DialogResult.Cancel )
                        return false;
                    else if ( dlgResult==DialogResult.No )
                        continue;

                    DocumentSaver saver = new DocumentSaver();
                    if ( saver.Save( doc,
                                     doc.FilePath )==false )
                    {
                        return false;
                    }
                }
            }

            // Save the files those haven't been saved yet ( save location is unknown ).
            if ( pathEmptyDocList.Count>0 )
            {
                foreach ( IDocument doc in pathEmptyDocList )
                {
                    DialogResult dlgResult =
                        ThreadSafeMsgBox.Show( doc.FileName + res.Strings.DIALOG_MSG_CONFIRM_SAVE_CHANGES,
                                               res.Strings.CONFIRM_CAPTION,
                                               System.Windows.Forms.MessageBoxButtons.YesNoCancel,
                                               System.Windows.Forms.MessageBoxIcon.Question );
                    if ( dlgResult==DialogResult.Cancel )
                        return false;
                    else if ( dlgResult==DialogResult.No )
                        continue;

                    DocumentSaver saver = new DocumentSaver();
                    if ( saver.Save( doc,
                                     doc.FilePath )==false )
                    {
                        return false;
                    }
                }
            }

            return result;
            */
        }

        /// <summary>
        /// ドキュメントを閉じる。
        /// </summary>
        protected bool CloseDocument(IDocument document, DocumentComponentChangedEventArgs changed)
        {
            // 通信していたらクローズパケット送信します。
            if (MCSManager.IsConnected && (document is IEmitterSetDocument))
            {
                // MCSを切断してから閉じる
                //Viewer.Message.Close.Send(
                //    NWCore.Protocols.BinaryDataType.Ptcl, document.Name);
            }

            // ドキュメントを閉じます。
            //TheApp.MainFrame.StatusMessage = StringResource.Get("IO_CLOSING_MSG", document.FileName);
            bool result = true;
            using (MCSFreezeBlock freezeBlock = new MCSFreezeBlock())
            {
                switch (document.ObjectID)
                {
                    case GuiObjectID.EffectProject:
                        result = CloseProjectDocument(document);
                        break;
                    case GuiObjectID.EmitterSet:
                        result = CloseCptlDocument(document);
                        break;
                    default:
//						Debug.Assert(false);
                        result = false;
                        break;
                }
                if (result)
                {
                    changed.AddClosedDocument(document);
                }
            }
            return result;
        }

        /// <summary>
        /// パーティクル・プロジェクトを閉じる
        /// </summary>
        private bool CloseProjectDocument(IDocument document)
        {
            DocumentPropertyChangedEventArgs changed =
                new DocumentPropertyChangedEventArgs(
                    new DocumentPropertyChangedArgs((document)));
            IProjectDocument project = (IProjectDocument)document;

            // マネージャから外します。
            ProjectManager.Remove(project);

            // Notify the document that it's been removed
            document.NotifyRemovedFromParent();

            // ガーベージコレクションを起動します。
            System.GC.Collect(0);

            // 通知
            Document.NotifyPropertyChanged(this, changed);
            return true;
        }

        /// <summary>
        /// パーティクル・ドキュメントを閉じる
        /// </summary>
        private bool CloseCptlDocument(IDocument document)
        {
            var emitterSetDoc = document as EmitterSetDocument;

            if (emitterSetDoc != null)
            {
                DisableActiveDocument(emitterSetDoc);

                var project = emitterSetDoc.Project;
                if (project != null)
                {
                    project.RemoveEmitterSetDocument(emitterSetDoc);
                    project.BuildTreeNodes();
                }
                // 名前が変更されたら元に戻せないのでアンドゥをクリアしておく
                TheApp.CommandManager.ClearActiveCommandStacks();
            }

            return true;
        }

        /// <summary>
        /// もしもクローズしようしている Document が ActiveDocumentならば=nullにする
        /// </summary>
        private void DisableActiveDocument(IDocument document)
        {
            if (document == null || ProjectManager.ActiveDocument == null)
            {
                return;
            }
            if (ProjectManager.ActiveDocument == document)
            {
                ProjectManager.ActiveDocument = null;
            }
            else
            {
                if (document.SubDocuments != null)
                foreach (var childDoc in document.SubDocuments)
                {
                    if (childDoc != null)
                    {
                        if (ProjectManager.ActiveDocument == childDoc)
                        {
                            ProjectManager.ActiveDocument = null;
                            return;
                        }
                        DisableActiveDocument(childDoc);
                        if (ProjectManager.ActiveDocument == null)
                        {
                            return;
                        }
                    }
                }
            }

        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 閉じる。
        /// </summary>
        public bool Close(List<IDocument> documents)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            bool result = Close(documents, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// 閉じる（TemplateMethod）。
        /// </summary>
        public virtual bool Close(List<IDocument> documents, DocumentComponentChangedEventArgs changed)
        {
            if (!CheckModified(documents))
            {
                return false;
            }
            return CloseDocument(documents, changed);
        }

        /// <summary>
        /// ドキュメントを閉じる。
        /// </summary>
        protected bool CloseDocument(
            List<IDocument> documents, DocumentComponentChangedEventArgs changed)
        {
            // 後ろから閉じる
            bool result = true;
            if (documents.Count > 0)
            {
                using (MCSFreezeBlock freezeBlock = new MCSFreezeBlock())
                {
                    foreach (IDocument document in documents)
                    {
                        // プロジェクト・ドキュメント閉じるとき
                        IProjectDocument project = document as IProjectDocument;
                        if (project != null)
                        {
                            // 通信で閉じたことを通知します。
                            if (MCSManager.IsConnected)
                            {
                                // 纏めてドキュメントを閉じるパケットを送信します。
                                //Viewer.Message.CloseAll.Send();
                                MCSManager.WaitTransmission();

                                // MCSを切断してから切断する
                                MCSManager.Close();
                            }
                        }
                    }

                    for (int i = documents.Count - 1; i >= 0; i--)
                    {
                        if (!CloseDocument(documents[i], changed))
                        {
                            result = false;
                        }
                        if (!CheckReference(documents[i]))
                        {
                            result = false;
                            continue;
                        }
                    }
                }
            }
            return result;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 子を含めて閉じる。
        /// </summary>
        public bool CloseWithChildren(IDocument document)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            bool result = CloseWithChildren(document, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// 子を含めて閉じる。
        /// </summary>
        public bool CloseWithChildren(IDocument document, DocumentComponentChangedEventArgs changed)
        {
            // 子の削除
            List<IDocument> closeList = new List<IDocument>();

            // プロジェクトの場合
            IProjectDocument project = document as IProjectDocument;
            if (project != null)
            {
                closeList.Add(project);
                // 中身も
                foreach (IDocument child in project.SubDocuments)
                {
                    if (System.String.IsNullOrEmpty(child.FileExt) == false)
                    {
                        closeList.Add(child);
                    }
                }
            }

            // 閉じるファイルがあれば複数ドキュメントを閉じる
            if (closeList.Count > 0)
            {
                return Close(closeList, changed);
            }
            else
            {
                return Close(document, changed);
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 全て閉じる。
        /// </summary>
        public bool CloseAll()
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            bool result = CloseAll(changed);
            Document.NotifyComponentChanged(this, changed);

            return result;
        }

        /// <summary>
        /// 全て閉じる。
        /// </summary>
        public bool CloseAll(DocumentComponentChangedEventArgs changed)
        {
            bool result = true;
            List<IProjectDocument> list =
                new List<IProjectDocument>(ProjectManager.Projects.ToArray());
            foreach (IProjectDocument project in list)
            {
                if (Close(project, changed) == false)
                {
                    result = false;
                }
            }

            return result;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 選択した複数のファイルを閉じる。
        /// </summary>
        public bool CloseSelection(List<IDocument> documents)
        {
            // 順に閉じていく
            bool result = true;
            foreach (IDocument document in documents)
            {
                // モデルを閉じる時は、アニメーション、テクスチャ、パレットも同時に閉じる
                // 既に閉じられたアニメーション、テクスチャ、パレットを閉じようとする
                // 可能性があるため、DocumentManager に存在確認をしてから閉じる
                if (DocumentManager.Documents.Contains(document))
                {
                    if (!CloseWithChildren(document))
                    {
                        result = false;
                    }
                }
            }
            return result;
        }

        //---------------------------------------------------------------------
        // ファイルが参照されているエラー
        private void ErrorFileIsReferenced(string fileName)
        {
            string message = string.Format(res.Strings.IO_FILE_IS_REFERENCED_MSG, fileName);
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning );
            */
        }
    }

    //=========================================================================
    /// <summary>
    /// スタータ処理用ドキュメントクローザ。
    /// </summary>
    public sealed class StarterDocumentCloser : DocumentCloser
    {
        /// <summary>
        /// 閉じる（TemplateMethod）。
        /// </summary>
        public override bool Close(IDocument document, DocumentComponentChangedEventArgs changed)
        {
            // 編集チェックは行わずに強制的に閉じる
            return CloseDocument(document, changed);
        }
    }

    //=========================================================================
    /// <summary>
    /// 入れ替え処理用ドキュメントクローザ。
    /// </summary>
    public sealed class SwapDocumentCloser : DocumentCloser
    {
        // やっていることは StarterDocumentCloser と全く同じだが、
        // コンテキストが異なるため、別クラスとして定義する

        /// <summary>
        /// 閉じる（TemplateMethod）。
        /// </summary>
        public override bool Close(IDocument document, DocumentComponentChangedEventArgs changed)
        {
            // 編集チェックは行わずに強制的に閉じる
            return CloseDocument(document, changed);
        }
    }

    //=========================================================================
    /// <summary>
    /// ファイナライズ処理用ドキュメントクローザ。
    /// </summary>
    public sealed class FinalizeDocumentCloser : DocumentCloser
    {
        /// <summary>
        /// 閉じる（TemplateMethod）。
        /// </summary>
        public override bool Close( List<IDocument> documents,
                                    DocumentComponentChangedEventArgs changed)
        {
            if (!CheckModified(documents))
            {
                return false;
            }

            CloseDocument(documents, changed);

            //App.Viewer.Message.PreviewCommunicator.CloseAllModelInfo();
            MCSManager.WaitTransmission();

            // MCSを切断してから閉じる
            //App.Viewer.Message.CloseConnection.Send();
            MCSManager.WaitTransmission();
            MCSManager.Close();

            return true;
        }
    }


    /// <summary>
    /// Close document without prompt for saving modified documents
    /// </summary>
    public sealed class ForceDocumentCloser : DocumentCloser
    {
        /// <summary>
        /// Override the saving part
        /// </summary>
        public override bool Close(List<IDocument> documents,
                                   DocumentComponentChangedEventArgs changed)
        {
            bool result = CloseDocument(documents, changed);
            return result;
        }
    }
}
