﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using App.Data;
using System.Collections.ObjectModel;
using NWCore.DataModel;

namespace App
{
    public delegate void RecentDocumentManagerHandler(RecentDocumentManager recentDocumentManager);

    /// <summary>
    /// Manages recently opened and saved documents.
    /// </summary>
    public class RecentDocumentManager : IEnumerable<string>
    {
        public const int MaximumRecentDocumentCount = 20;
        public event RecentDocumentManagerHandler RecentDocumentsTrayChanged;

        private readonly object lockCountSyncRoot = new object();
        private int lockCount;

        /// <summary>
        /// Gets whether the recent document manager is locked or not.
        /// When locked, the NotifyDocumentLoaded method produces no side effects.
        /// </summary>
        public bool IsLocked
        {
            get
            {
                lock (lockCountSyncRoot)
                    return lockCount > 0;
            }
        }

        protected readonly object syncRoot = new object();

        private readonly List<string> documents = new List<string>();

        /// <summary>
        /// Initializes the RecentDocumentManager.
        /// </summary>
        public void Initialize()
        {
            if (Config.Data.DocumentIO == null)
                return;

            EnsureDataModelValidity();

            UdpateDocumentManager();

            bool raiseEvent;
            lock (syncRoot)
            {
                raiseEvent = documents.Count > 0;
            }

            if (raiseEvent)
                OnRecentDocumentsTrayChanged();
        }

        /// <summary>
        /// Clears the recent files list.
        /// </summary>
        public void Clear()
        {
            documents.Clear();
            UpdateDataModel();
            OnRecentDocumentsTrayChanged();
        }

        /// <summary>
        /// Update the lower-level data.
        /// Must be called before saving data to configuration file.
        /// </summary>
        public void Flush()
        {
            UpdateDataModel();
        }

        /// <summary>
        /// Locks the RecentDocumentManager.
        /// When locked, the NotifyDocumentLoaded method produces no side effects.
        /// Note: This method is reentrant.
        /// </summary>
        public void Lock()
        {
            lock (lockCountSyncRoot)
            {
                lockCount++;
            }
        }

        /// <summary>
        /// Unlocks the RecentDocumentManager.
        /// When locked, the NotifyDocumentLoaded method produces no side effects.
        /// Note: This method is reentrant.
        /// </summary>
        public void Unlock()
        {
            lock (lockCountSyncRoot)
            {
                lockCount = Math.Max(0, lockCount - 1);
            }
        }

        /// <summary>
        /// Gets the current recent documents, in proper open/save order.
        /// </summary>
        public string[] Documents
        {
            get
            {
                lock (syncRoot)
                {
                    // generate a thread-safe cold list
                    return documents.ToArray();
                }
            }
        }

        /// <summary>
        /// Used to notify the RecentDocumentManager that a file has been loaded or saved.
        /// If the RecentDocumentManager is locked, this method has no effect.
        /// </summary>
        /// <param name="docs">Documents to notify.</param>
        public void NotifyDocumentLoaded(params IDocument[] docs)
        {
            if (docs == null || docs.Length == 0 || IsLocked)
                return;

            lock (syncRoot)
            {
                foreach (var doc in docs)
                {
                    if (documents.Any(file => string.Equals(file, doc.FilePath,
                        StringComparison.OrdinalIgnoreCase)))
                    {
                        // ensure the document is not in the list anymore
                        documents.RemoveAll(file => string.Equals(file, doc.FilePath,
                            StringComparison.OrdinalIgnoreCase));

                        // then insert the document at the begining of the tray
                        documents.Insert(0, doc.FilePath);
                    }
                    else
                    {
                        // insert the document at the begining of the tray
                        documents.Insert(0, doc.FilePath);
                    }
                }

                // then ensure the tray respect the maximum size
                if (documents.Count > MaximumRecentDocumentCount)
                {
                    documents.RemoveRange(MaximumRecentDocumentCount,
                        documents.Count - MaximumRecentDocumentCount);
                }
            }

            OnRecentDocumentsTrayChanged();
        }

        /// <summary>
        /// Synchronize the content of the Config.Data.* data model
        /// with the RecentDocumentManager elements.
        /// </summary>
        private void UpdateDataModel()
        {
            lock (syncRoot)
            {
                Config.Data.DocumentIO.LastLoadedFileList.LastLoadedFiles = documents
                    .Take(MaximumRecentDocumentCount)
                    .ToArray();
            }
        }

        /// <summary>
        /// Synchronize the content of the RecentDocumentManager
        /// with the Config.Data.* data model information
        /// </summary>
        private void UdpateDocumentManager()
        {
            var fileList = Config.Data.DocumentIO.LastLoadedFileList.LastLoadedFiles;

            lock (syncRoot)
            {
                documents.Clear();
                documents.AddRange(fileList.Take(MaximumRecentDocumentCount));
            }
        }

        /// <summary>
        /// This method ensures that Config.Data structures are properly set.
        /// </summary>
        private static void EnsureDataModelValidity()
        {
            if (Config.Data.DocumentIO.LastLoadedFileList == null)
            {
                Config.Data.DocumentIO.LastLoadedFileList = new CfLastLoadedFileList
                {
                    LastLoadedFiles = new string[0],
                };
            }
            else if (Config.Data.DocumentIO.LastLoadedFileList.LastLoadedFiles == null)
                Config.Data.DocumentIO.LastLoadedFileList.LastLoadedFiles = new string[0];
        }

        /// <summary>
        /// When overridden, called when the recent documents tray changes.
        /// </summary>
        protected virtual void OnRecentDocumentsTrayChanged()
        {
            var handler = RecentDocumentsTrayChanged;

            if (handler != null)
                handler(this);
        }

        /// <summary>
        /// Allows the RecentDocumentManager to be seen as a collection of recent files.
        /// </summary>
        /// <returns>Returns an enumerator of strings.</returns>
        public IEnumerator<string> GetEnumerator()
        {
            return ((IEnumerable<string>)Documents).GetEnumerator();
        }

        /// <summary>
        /// Allows the RecentDocumentManager to be seen as a collection of recent files.
        /// </summary>
        /// <returns>Returns an enumerator of strings.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return Documents.GetEnumerator();
        }
    }

    /// <summary>
    /// Represent a scoped RecentDocumentManager locker/unlocker.
    /// </summary>
    public class RecentDocumentManagerLocker : IDisposable
    {
        private RecentDocumentManager[] managersToLock;

        /// <summary>
        /// Locks the given RecentDocumentManagers.
        /// </summary>
        /// <param name="managersToLock">Instances of RecentDocumentManager to lock.</param>
        public RecentDocumentManagerLocker(params RecentDocumentManager[] managersToLock)
        {
            if (managersToLock == null)
                throw new ArgumentNullException("managersToLock");

            if (managersToLock.Length == 0)
                throw new ArgumentException("managersToLock must not be empty");

            this.managersToLock = managersToLock;
            foreach (var manager in managersToLock)
                manager.Lock();
        }

        /// <summary>
        /// Unlocks the RecentDocumentManager.
        /// </summary>
        public void Dispose()
        {
            if (managersToLock != null)
            {
                var managers = managersToLock;
                managersToLock = null;

                foreach (var manager in managers)
                    manager.Unlock();
            }
        }
    }
}
