﻿// ========================================================================
// <copyright file="DataModelUpdaterManager.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.IO;
using System.Xml;
using System.Collections.Generic;

using NWCore.DataModel;
using App.IO;

namespace App.Data
{
    #region Enumerator for error types

    public enum UpdaterErrorTypes
    {
        Succeed,

        UpdaterTypeNotFound,
        UnknownVersion,
        InvalidVersionFormat,
        InvalidXMLFile,
        FailedDeserialize,
        FailedUpdate
    }

    #endregion

    /// <summary>
    /// Class for the data model updater manager.
    /// </summary>
    public class DataModelUpdaterManager
    {
        #region Class for updater list

        /// <summary>
        /// Class for data model list for a specific updater type.
        /// </summary>
        private class UpdaterList
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="updaterType">The type of the updaters.</param>
            public UpdaterList( string updaterType ) :
                base()
            {
                this.UpdaterType    = string.Copy( updaterType );
                this.UpdaterTypeCRC = TheApp.CRC32Helper.ComputeCRC32Str( this.UpdaterType );
            }

            #endregion

            #region Properties

            /// <summary>The type of the updaters.</summary>
            public string UpdaterType { get; private set; }


            /// <summary>The CRC32 hash code of the type of the updaters.</summary>
            public uint UpdaterTypeCRC { get; private set; }

            #endregion

            #region Updaters

            /// <summary>
            /// Add a new updater to the list.
            /// </summary>
            /// <param name="updater">The updater to add.</param>
            public void Add( IDataModelUpdater updater )
            {
                int iIndex = FindUpdaterIndexForInsert( updater );

                if ( iIndex>=m_updaters.Count )
                    m_updaters.Add( updater );
                else
                    m_updaters.Insert( iIndex, updater );
            }


            /// <summary>
            /// Find the index for inserting the updater.
            /// </summary>
            /// <param name="updater">The updater to insert.</param>
            /// <returns>The updater index to insert.</returns>
            private int FindUpdaterIndexForInsert( IDataModelUpdater updater )
            {
                int iCompare = 0;

                // Do the trivial check
                int iSize = m_updaters.Count;
                if ( iSize==0 ||
                     m_updaters[0].CurrentVersion.CompareTo(updater.CurrentVersion)>0 )
                {
                    return 0;
                }
                else if ( m_updaters[iSize-1].CurrentVersion.CompareTo(updater.CurrentVersion)<=0 )
                {
                    return iSize;
                }

                int iLeft  = 0;
                int iRight = iSize - 1;
                int iMid   = (iLeft + iRight) / 2;

                // Search the updater index with binary search
                while ( iLeft<iRight )
                {
                    // Compare updater versions
                    iCompare = m_updaters[iMid].CurrentVersion.CompareTo( updater.CurrentVersion );
                    if ( iCompare >0 )
                        iRight = iMid;
                    else if ( iCompare<0 )
                        iLeft  = iMid + 1;
                    else
                        return iMid + 1;

                    // Compute middle index
                    iMid = (iLeft + iRight) / 2;
                }

                return iLeft;
            }

            #endregion

            #region Load data model

            /// <summary>
            /// Load the data model from the given memory stream.
            ///
            /// The method is virtual so that you can derive this method
            /// and load it with your way whenever it's necessary.
            /// </summary>
            /// <param name="stream">The memory stream.</param>
            /// <param name="err">Error type.</param>
            /// <returns>The loaded data model of the latest version.</returns>
            public virtual object LoadDataModel( Stream stream,
                                                 out UpdaterErrorTypes err )
            {
                // Parse binary version.
                Version binaryVersion = ParseBinaryVersion( stream, out err );
                if ( err!=UpdaterErrorTypes.Succeed )
                    return null;

                // Rewind the stream pointer.
                stream.Position = 0;

                // Find the updaters to load and update the data.
                IDataModelUpdater updater   = null;
                object            dataModel = null;
                foreach ( IDataModelUpdater item in m_updaters )
                {
                    if ( updater==null &&
                         item.CanDeserialize( binaryVersion )==true )
                    {
                        try
                        {
                            Utility.ConvertTimer.Start(Utility.TimerType.TIMER_DATAMODEL_DESERIALIZE);

                            // Deserialize the memory stream to data model.
                            dataModel = item.Deserialize( binaryVersion, stream );
                            updater   = item;

                            Utility.ConvertTimer.Stop(Utility.TimerType.TIMER_DATAMODEL_DESERIALIZE);
                        }
                        catch ( Exception ex )
                        {
                            if ( ex!=null )
                            {
                                Utility.ConvertTimer.Stop(Utility.TimerType.TIMER_DATAMODEL_DESERIALIZE);

                                DebugConsole.WriteLine(ex.Message);
                                System.Diagnostics.Trace.WriteLine(ex.Message);
                                TheApp.Logger.Fatal.AddMessage( res.Strings.DATAMODELUPDATERMGR_FILE_LOAD_EXCEPTION + ex.Message );

                                err = UpdaterErrorTypes.FailedDeserialize;
                                return dataModel;
                            }
                        }
                    }
                    else if ( updater!=null &&
                              item.CanUpdate( dataModel )==true )
                    {
                        try
                        {
                            Utility.ConvertTimer.Start(Utility.TimerType.TIMER_DATAMODEL_UPDATE);

                            // Update the loaded data model to the latest version.
                            dataModel = item.Update( dataModel );
                            updater   = item;

                            Utility.ConvertTimer.Stop(Utility.TimerType.TIMER_DATAMODEL_UPDATE);
                        }
                        catch (System.Exception ex)
                        {
                            if ( ex!=null )
                            {
                                Utility.ConvertTimer.Stop(Utility.TimerType.TIMER_DATAMODEL_UPDATE);

                                err = UpdaterErrorTypes.FailedUpdate;
                                return dataModel;
                            }
                        }
                    }
                }

                if ( updater==null )
                {
                    err = UpdaterErrorTypes.UnknownVersion;
                    return null;
                }

                return dataModel;
            }


            /// <summary>
            /// Parse the XML document in the given memory stream and
            /// find the binary version.
            ///
            /// The method is virtual so that you can derive this method
            /// and parse it with your way whenever it's necessary.
            /// </summary>
            /// <param name="stream">The memory stream.</param>
            /// <param name="err">Error type.</param>
            /// <returns>The parsed binary version.</returns>
            public virtual Version ParseBinaryVersion( Stream stream,
                                                       out UpdaterErrorTypes err )
            {
                err = UpdaterErrorTypes.Succeed;

                // Rewind the stream pointer.
                stream.Position = 0;

                // Load XML document for the binary version
                try
                {
                    XmlDocument doc = new XmlDocument();
                    doc.Load( stream );

                    XmlElement root        = doc.DocumentElement;
                    XmlNode    versionNode = root.SelectSingleNode( "Version" );
                    if ( versionNode==null )
                        return new Version( 0, 0, 0, 0 );

                    Version version;
                    if ( Version.TryParse( versionNode.InnerText, out version )==false )
                    {
                        err = UpdaterErrorTypes.InvalidVersionFormat;
                        return null;
                    }

                    return version;
                }
                catch ( XmlException e )
                {
                    if ( e!=null )
                        err = UpdaterErrorTypes.InvalidXMLFile;
                }

                return null;
            }

            #endregion

            #region Member variables

            private List<IDataModelUpdater> m_updaters = new List<IDataModelUpdater>();

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelUpdaterManager()
        {
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_0_9_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_1_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_2_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_3_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_4_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_5_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_6_0_0() );
            RegisterUpdater( DocumentConstants.DotEset, new EsetUpdater_1_7_0_0() );
        }

        #endregion

        #region Updaters / updater lists

        public bool RegisterUpdater( string updaterType,
                                     IDataModelUpdater updater )
        {
            // Find the updater list by its type
            UpdaterList list = FindUpdaterList( updaterType );
            if ( list==null )
            {
                // The updater list does not exist, find the sorted index
                // for adding a new updater list.
                int iIndex = FindUpdaterListIndexForInsert( updaterType );

                // Create the updater list and add it.
                list = new UpdaterList( updaterType );
                if ( iIndex==m_updaters.Count )
                    m_updaters.Add( list );
                else
                    m_updaters.Insert( iIndex, list );
            }

            // Add the updater to the correct updater list.
            list.Add( updater );

            return true;
        }


        /// <summary>
        /// Find the updater list for the specified type.
        /// </summary>
        /// <param name="updaterType">The updater type.</param>
        /// <returns>The updater list or null if the given type has no updaters.</returns>
        private UpdaterList FindUpdaterList( string updaterType )
        {
            uint dwCRC = TheApp.CRC32Helper.ComputeCRC32Str( updaterType );

            // Do the trivial check
            int iSize = m_updaters.Count;
            if ( iSize==0 )
            {
                return null;
            }
            else if ( iSize==1 )
            {
                if ( (m_updaters[0]).UpdaterTypeCRC==dwCRC )
                    return m_updaters[0];
                else
                    return null;
            }

            int iLeft  = 0;
            int iRight = iSize - 1;
            int iMid;

            // Search the updater list with binary search
            while ( iLeft<(iRight-1) )
            {
                // Compute middle index
                iMid = (iLeft + iRight) / 2;

                // Compare updater type
                UpdaterList list = m_updaters[iMid];
                if ( list.UpdaterTypeCRC>dwCRC )
                    iRight = iMid;
                else if ( list.UpdaterTypeCRC<dwCRC )
                    iLeft = iMid;
                else
                    return list;
            }

            if ( (m_updaters[iLeft]).UpdaterTypeCRC==dwCRC )
                return m_updaters[iLeft];
            else if ( (m_updaters[iRight]).UpdaterTypeCRC==dwCRC )
                return m_updaters[iRight];

            return null;
        }


        /// <summary>
        /// Find the updater list index for the specified type.
        /// If the update list was not found, the method returns
        /// the previous index for inserting.
        /// </summary>
        /// <param name="updaterType">The updater type.</param>
        /// <returns>The updater list index or previous index for inserting.</returns>
        private int FindUpdaterListIndexForInsert( string updaterType )
        {
            uint dwCRC = TheApp.CRC32Helper.ComputeCRC32Str( updaterType );

            // Do the trivial check
            int iSize = m_updaters.Count;
            if ( iSize==0 || (m_updaters[0]).UpdaterTypeCRC>dwCRC )
                return 0;
            else if ( (m_updaters[iSize-1]).UpdaterTypeCRC<=dwCRC )
                return iSize;

            int iLeft  = 0;
            int iRight = iSize - 1;
            int iMid   = (iLeft + iRight) / 2;

            // Search the updater list with binary search
            while ( iLeft<iRight )
            {
                // Compare updater list types
                UpdaterList list = m_updaters[iMid];
                if ( list.UpdaterTypeCRC>dwCRC )
                    iRight = iMid;
                else if ( list.UpdaterTypeCRC<dwCRC )
                    iLeft  = iMid + 1;
                else
                    return iMid + 1;

                // Compute middle index
                iMid = (iLeft + iRight) / 2;
            }

            return iLeft;
        }

        #endregion

        #region Load and update data model

        /// <summary>
        /// Load and update data model from the given memory stream.
        /// </summary>
        /// <param name="updaterType">The updater type.</param>
        /// <param name="stream">The memory stream that contains the data model file.</param>
        /// <param name="err">Error type.</param>
        /// <returns>The loaded data model.</returns>
        public object LoadDataModel( string updaterType,
                                     MemoryStream stream,
                                     out UpdaterErrorTypes err )
        {
            UpdaterList list = FindUpdaterList( updaterType );
            if ( list==null )
            {
                err = UpdaterErrorTypes.UpdaterTypeNotFound;
                return null;
            }

            return list.LoadDataModel( stream, out err );
        }

        #endregion

        #region Member variables

        private List<UpdaterList> m_updaters = new List<UpdaterList>();

        #endregion
    }
}
