﻿// ========================================================================
// <copyright file="PreviewDocument.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
//using App.Command;
using App.IO;
//using App.PropertyEdit;
using NWCore.DataModel;
using NintendoWare.ToolDevelopmentKit;

namespace App.Data
{
    /// <summary>
    /// パーティクル・セット用ドキュメント
    /// </summary>
    public sealed class PreviewDocument : BaseProjectPanelDocument, IPreviewDocument
    {
        #region Memeber Variables

        private readonly IProjectDocument m_projectDoc = null;    // parent/owner
        private readonly PreviewConfigData m_previewData = null;

        private PreviewModelDocument[] m_modelDocuments = null;
        //private bool m_bUpdatingBackground;

        #endregion

        #region Properties

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                return res.Strings.PREVIEW_NODE_NAME;
            }
            set
            {
            }
        }

        /// <summary>
        /// オーナードキュメント。
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.m_projectDoc; }
        }

        /// <summary>
        /// Emitter that this child belongs to
        /// </summary>
        public IProjectDocument Project
        {
            get { return this.m_projectDoc; }
        }

        /// <summary>
        /// データモデル：プレビュー設定
        /// </summary>
        public PreviewConfigData PreviewConfigData
        {
            get { return this.m_previewData; }
        }

        /// <summary>
        /// Shortcut for accessing clear color property of the work space data.
        /// </summary>
        public RgbaColor InitSettingsClearColor
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    #if BUILD_FOR_CTR
                    return new RgbaColor( 0.2f, 0.2f, 0.2f, 1.0f );
                    #else
                    return new RgbaColor( 0.03f, 0.03f, 0.03f, 1.0f );
                    #endif
                }

                return project.EnvConfigData.ClearColor;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null )
                {
                    if ( project.EnvConfigData!=null )
                        project.EnvConfigData.ClearColor.Set( value );
                }
            }
        }

        /// <summary>
        /// Shortcut for accessing background image path property of the work space data.
        /// </summary>
        public string InitSettingsBackgroundImagePath
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return string.Empty;
                }

                return project.EnvConfigData.BackgroundImagePath;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.BackgroundImagePath = value;
                }
            }
        }

        /// <summary>
        /// Shortcut for accessing background image multiply color property of the work space data.
        /// </summary>
        public RgbaColor InitSettingsBGImageMultiplyColor
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );
                }

                return project.EnvConfigData.BGImageMultColor;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.BGImageMultColor.Set( value );
                }
            }
        }

        /// <summary>
        /// カメラの位置。
        /// </summary>
        public Vector3 ViewerCameraPos
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return new Vector3( 0.0f, 50.0f, 150.0f );
                }

                return project.EnvConfigData.CameraPos;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraPos.Set( value );
                }
            }
        }

        /// <summary>
        /// カメラの注視点。
        /// </summary>
        public Vector3 ViewerCameraLookAt
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return new Vector3( 0.0f, 0.0f, 0.0f );
                }

                return project.EnvConfigData.CameraLookAt;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraLookAt.Set( value );
                }
            }
        }

        /// <summary>
        /// カメラの回転。
        /// </summary>
        public float ViewerCameraRotation
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return 0.0f;
                }

                return project.EnvConfigData.CameraRotation;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraRotation = value;
                }
            }
        }

        /// <summary>
        /// カメラタイプ。
        /// </summary>
        public CameraProjTypes ViewerCameraProjType
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return CameraProjTypes.Perspective;
                }

                return project.EnvConfigData.CameraProjType;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraProjType = value;
                }
            }
        }

        /// <summary>
        /// アスペクト比。(Perspective projection)
        /// </summary>
        public float ViewerCameraPerspectiveAspectRatio
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    #if BUILD_FOR_CTR
                    return 1.66f;
                    #else
                    return 1.77f;
                    #endif
                }

                return project.EnvConfigData.CameraPerspectiveAspectRatio;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraPerspectiveAspectRatio = value;
                }
            }
        }

        /// <summary>
        /// 画角。(Perspective projection)
        /// </summary>
        public float ViewerCameraPerspectiveFOV
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return 1.0472f; // 60 degrees in radian.
                }

                return project.EnvConfigData.CameraPerspectiveFOV;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraPerspectiveFOV = value;
                }
            }
        }

        /// <summary>
        /// オルソアスペクト比。(Orthogonal projection)
        /// </summary>
        public float ViewerCameraOrthogonalAspectRatio
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    #if BUILD_FOR_CTR
                    return 1.66f;
                    #else
                    return 1.77f;
                    #endif
                }

                return project.EnvConfigData.CameraOrthogonalAspectRatio;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraOrthogonalAspectRatio = value;
                }
            }
        }

        /// <summary>
        /// Frustum height.(Orthogonal projection)
        /// </summary>
        public float ViewerCameraOrthogonalFrustumHeight
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return 72.0f;
                }

                return project.EnvConfigData.CameraOrthogonalFrustumHeight;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraOrthogonalFrustumHeight = value;
                }
            }
        }

        /// <summary>
        /// Frustum near plane.
        /// </summary>
        public float ViewerCameraFrustumNear
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return 0.1f;
                }

                return project.EnvConfigData.CameraFrustumNear;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraFrustumNear = value;
                }
            }
        }

        /// <summary>
        /// Frustum far plane.
        /// </summary>
        public float ViewerCameraFrustumFar
        {
            get
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project==null ||
                     project.EnvConfigData==null )
                {
                    return 1000.0f;
                }

                return project.EnvConfigData.CameraFrustumFar;
            }

            set
            {
                EffectProjectDocument project = this.Project as EffectProjectDocument;
                if ( project!=null &&
                     project.EnvConfigData!=null )
                {
                    project.EnvConfigData.CameraFrustumFar = value;
                }
            }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "PreviewConfigData";
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return this.Project.DataScrPath
                       + ".PreviewDocument";
            }
        }

        /// <summary>
        /// Get the preview model documents.
        /// </summary>
        public IPreviewModelDocument[] PreviewModelDocuments
        {
            get
            {
                return m_modelDocuments;
            }
        }

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PreviewDocument(IProjectDocument projectDoc, PreviewConfigData datamodel) :
            base(res.Strings.PREVIEW_NODE_NAME)
        {
            this.Name = res.Strings.PREVIEW_NODE_NAME;
            this.m_projectDoc = projectDoc;
            this.m_previewData = datamodel;

            // Create preview model documents
            if ( datamodel!=null )
            {
                // How many model documents should we create?
                int iNumChildren = PreviewConfigData.MinPreviewModelCount;

                // Create the array for the model documents.
                m_modelDocuments =
                    new PreviewModelDocument[iNumChildren];

                // Create the model documents.
                int i;
                for ( i=0;i<iNumChildren;++i )
                {
                    m_modelDocuments[i] = new PreviewModelDocument( this,
                                                                    datamodel.GetModelData( i, true ),
                                                                    i );
                }
            }

            DocumentManager.NotifyDocumentCreated(this);
        }


        #endregion

        #region オブジェクト

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.Preview; }
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Check if the given document can be a child of this document.
        /// E.q. Emitter can be emitter set's child, but cannot be a child
        /// of an effect project document.
        /// </summary>
        /// <param name="doc">The child document.</param>
        /// <returns>True if the given document is a valid child document type.</returns>
        public override bool CanAddAsChildDoc( IDocument doc )
        {
            if ( doc is PreviewModelDocument )
                return true;

            return false;
        }

        /// <summary>
        /// Reload texture.
        /// </summary>
        public bool ReloadTexture(string path)
        {
            if ( path == null ||
                 path == InitSettingsBackgroundImagePath )
            {
                string fullPath = String.Empty;
                TextureRes textureRes = new TextureRes();
                bool result = DocumentIO.ReadTextureFile(this.FileLocation,
                                                        InitSettingsBackgroundImagePath,
                                                        textureRes,
                                                        ref fullPath);

                // Get the data source path of the texture resource.
                string docPath = this.DataScrPath + "." +
                                    "InitSettingsBackgroundImagePath";

                // Create the command argument with the
                // texture resource(bitmaps) as data source.
                /*
                PropertyEditCommandArgs args =
                    new PropertyEditCommandArgs( new NWCore.Utility.Path(""),
                                                 new NWCore.Utility.Path(docPath),
                                                 this,
                                                 InitSettingsBackgroundImagePath,
                                                 InitSettingsBackgroundImagePath);

                // Create a command to notify that the bitmaps has changed.
                ICommand newCommand = PropertyEditCommand.CreateCommand(args);

                ((CommandManager)TheApp.CommandManager).TriggerCommandExecutedEvent(newCommand, CommandExecutionTypes.Execute, this);
                */

                return true;
            }

            return false;
        }

        /// <summary>
        /// Handle background image path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnBGImagePathChanged( ICommand command,
                                          CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if (propCommand == null)
                return;

            ReloadTexture(null);

            if (m_bUpdatingBackground == true)
                return;

            m_bUpdatingBackground = true;

            // Get the data source path of the background image
            var docPath = this.DataScrPath + "." +
                "InitSettingsBackgroundImagePath";

            TextureRes textureRes = new TextureRes();
            string texPatFileName = InitSettingsBackgroundImagePath;
            TextureResourceHelper.ReadTextureAndIssueCommand(docPath, propCommand.ControlPath, this, textureRes, ref texPatFileName);
            InitSettingsBackgroundImagePath = texPatFileName;

            m_bUpdatingBackground = false;

            //CheckAssetsReachability(true);
            */
        }


        #endregion

        #region Event handlers

        /// <summary>
        /// Gamma correction mode changed event
        /// </summary>
        private void OnGammaCorrectionModeChanged()
        {
            ReloadTexture(null);
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            TheApp.OnGammaCorrectionModeChanged += this.OnGammaCorrectionModeChanged;
            DocumentManager.RegisterDocumentPath(this, this.DataScrPath);

            foreach (PreviewModelDocument childDoc in m_modelDocuments)
            {
                // Notify the document that it has been added
                if (childDoc != null)
                    childDoc.NotifyAddedToParent();
            }

            /*
             *  BG Image変更のイベントリスナー
             */
            // Get the data source path of the background image
            var docPath = this.DataScrPath + "." +
                "InitSettingsBackgroundImagePath";

            // Add command listener to monitor the texture file path.
            TheApp.CommandManager.AddCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                      docPath,
                                                      OnBGImagePathChanged );

        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            /*
             *  Primitive変更のイベントリスナー
             */
            // Get the data source path of the emitter document
            var docPath = this.DataScrPath + "." +
                "InitSettingsBackgroundImagePath";

            // Remove the command listener.
            TheApp.CommandManager.RemoveCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                         docPath,
                                                         OnBGImagePathChanged );

            foreach (PreviewModelDocument childDoc in m_modelDocuments)
            {
                // Notify the document that it's going to been removed
                if (childDoc != null)
                    childDoc.NotifyRemovedFromParent();
            }

            TheApp.OnGammaCorrectionModeChanged -= this.OnGammaCorrectionModeChanged;
            DocumentManager.UnregisterDocumentPath(this, this.DataScrPath);

            base.NotifyRemovedFromParent();
        }

        #endregion
    }
}
