﻿// ========================================================================
// <copyright file="EmitterDocument.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
//using App.Command;
using App.IO;
//using App.PropertyEdit;
//using App.Controls;
using App.Utility;
using NWCore.DataModel;
using NintendoWare.ToolDevelopmentKit;

namespace App.Data
{
    /// <summary>
    /// パーティクル・セット用ドキュメント
    /// </summary>
    public sealed class EmitterDocument : BaseProjectPanelDocument,
                                          IEmitterDocument,
                                          IDataModelProxyOwner
    {
        #region Inner Struct - HostIOColorInheritTypeConvTbl

        /// <summary>
        /// カラー継承の定義
        /// </summary>
        private struct HostIOColorInheritTypeConvTbl
        {
            public HostIOColorInheritTypeConvTbl(CombinerType type, bool bEditable)
            {
                this.CombinerType = type;
                this.IsColorEditable = bEditable;
            }

            public CombinerType CombinerType;
            public bool IsColorEditable;
        };

        #endregion

        #region  Inner Struct - HostIOColorTypeConvTbl

        /// <summary>
        /// internal host color type
        /// </summary>
        struct HostIOColorTypeConvTbl
        {
            public HostIOColorTypeConvTbl(ColorCalcType colorCalcType,
                                          CombinerType combinerType,
                                          uint emitterFlg,
                                          bool isConstColorEditable,
                                          bool isColor1Editable,
                                          bool isColor2Editable,
                                          bool isColor3Editable,
                                          bool is3ColorAnimEditable,
                                          string colorLabel0 = "",
                                          string colorLabel1 = "",
                                          string colorLabel2 = "",
                                          string colorLabel3 = "")
            {
                this.ColorCalcType = colorCalcType;
                this.CombinerType = combinerType;
                this.EmitterFlg = emitterFlg;
                this.IsConstColorEditable = isConstColorEditable;
                this.IsColor1Editable = isColor1Editable;
                this.IsColor2Editable = isColor2Editable;
                this.IsColor3Editable = isColor3Editable;
                this.Is3ColorAnimEditable = is3ColorAnimEditable;
                this.ColorLabel0 = colorLabel0;
                this.ColorLabel1 = colorLabel1;
                this.ColorLabel2 = colorLabel2;
                this.ColorLabel3 = colorLabel3;
            }

            /// <summary>
            ///
            /// </summary>
            public ColorCalcType ColorCalcType;

            /// <summary>
            ///
            /// </summary>
            public CombinerType CombinerType;

            /// <summary>
            ///
            /// </summary>
            public uint EmitterFlg;

            /// <summary>
            ///
            /// </summary>
            public bool IsConstColorEditable;

            /// <summary>
            ///
            /// </summary>
            public bool IsColor1Editable;

            /// <summary>
            ///
            /// </summary>
            public bool IsColor2Editable;

            /// <summary>
            ///
            /// </summary>
            public bool IsColor3Editable;

            /// <summary>
            ///
            /// </summary>
            public bool Is3ColorAnimEditable;

            /// <summary>
            ///
            /// </summary>
            public string ColorLabel0;

            /// <summary>
            ///
            /// </summary>
            public string ColorLabel1;

            /// <summary>
            ///
            /// </summary>
            public string ColorLabel2;

            /// <summary>
            ///
            /// </summary>
            public string ColorLabel3;
        };

        #endregion

        #region Member Variables

        private IEmitterSetDocument m_emitterSetDoc = null;        // parent/owner
        private EmitterData         m_editData = null;             // data model
        private ChildDocument       m_childDocument = null;        // data model for complex emitter
        private FieldDocument       m_filedDocument = null;        // data model for complex emitter
        private FluctuationDocument m_fluctunationDocument = null; // data model for complex emitter

        private DataModelProxy      m_dataModelProxy = null;       // Data model proxy for dev tab

        private bool                m_bUpdatingTexture0 = false;
        private bool                m_bUpdatingTexture1 = false;
        private bool                m_bUpdatingTexture2 = false;

        //private CustomFileSystemWatcher m_primitiveFileWatcher = new CustomFileSystemWatcher();

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ（エフェクト用）
        /// </summary>
        public EmitterDocument(EmitterSetDocument emitterSetDoc,
                               EmitterData editData) : base(editData.Name)
        {
            this.Name = editData.Name;
            this.m_emitterSetDoc = emitterSetDoc;
            this.m_editData = editData;

            if ( editData.Type==EmitterType.Complex )
            {
                CreateChildDocument();
                CreateFieldDocument();
                CreateFluctuationDocument();
            }

            InitDataModelProxy();

            if ( m_editData!=null &&
                 m_editData.AnimTableList!=null )
            {
                // Create the animation tables those were registered
                // but not having been  loaded or created yet.
                m_editData.AnimTableList.AddRegisteredAnimationTables();
            }
        }

        /// <summary>
        /// コンストラクタ (for xml deserialization purpose)
        /// </summary>
        public EmitterDocument()
            : base("")
        {
            this.m_editData = new EmitterData();

            InitDataModelProxy();
        }

        #endregion

        #region Properties

        /// <summary>
        /// デバッグ表示するか？
        /// </summary>
        public bool IsShow
        {
            get
            {
                return this.EmitterData.EmitterGameData.DispEmitter;
            }
            set
            {
                this.EmitterData.EmitterGameData.DispEmitter = value;
            }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.Emitter; }
        }

        /// <summary>
        /// アイコンのイメージキー
        /// </summary>
        public override string ImageKey
        {
            get
            {
                string baseName = this.ObjectID.ToString();
                string complexSuffix = (this.EmitterType != EmitterType.Simple ? Document.ComplexEmitterKey : string.Empty);
                string unreachableAssetSuffix = (this.HasUnreachableAsset ? Document.UnreachableAssetKey : string.Empty);
                string typeMismatchTextureSuffix = string.Empty;

                DialogResult result;
                if (!this.HasUnreachableAsset
                  && this.HasTypeMismatchTexture
                  && Config.ShouldShowMessageBox("LinearEditTextureWarning", out result))
                {
                    typeMismatchTextureSuffix = Document.TextureTypeMismatchKey;
                }

                return baseName + complexSuffix + unreachableAssetSuffix + typeMismatchTextureSuffix;
            }
        }

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                if (this.EmitterData != null)
                {
                    return this.EmitterData.Name;
                }
                return String.Empty;
            }
            set
            {
                if (this.EmitterData != null)
                {
                    this.EmitterData.Name = value;
                }
            }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileExt
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileExt
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileFilter
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileFilter
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// オーナードキュメント
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.EmitterSetDocument; }
        }

        /// <summary>
        /// emitter set this emitter belongs to
        /// </summary>
        public IEmitterSetDocument EmitterSetDocument
        {
            get
            {
                return this.m_emitterSetDoc;
            }
            set
            {
                this.m_emitterSetDoc = value;
            }
        }

        /// <summary>
        /// 関連するテクスチャのパスを取得します
        /// </summary>
        public IEnumerable<string> TexturePaths
        {
            get
            {
                // テクスチャ０パス
                yield return this.EmitterData.TexPatData0.UI_texPatFileName;
                // テクスチャ１パス
                yield return this.EmitterData.TexPatData1.UI_texPatFileName;
                // テクスチャ２パス
                yield return this.EmitterData.TexPatData2.UI_texPatFileName;
                // チャイルド・テクスチャパス
                if ( this.EmitterData.ChildFlag.Enable==true )
                {
                    yield return this.EmitterData.ChildTexPatData.UI_texPatFileName;
                }
            }
        }

        /// <summary>
        /// 関連するテクスチャを取得します
        /// </summary>
        public IEnumerable<EmitterAssetInfo> Textures
        {
            get
            {
                // テクスチャ０
                yield return new EmitterAssetInfo( this,
                                                   EmitterAssetTypes.EmitterTexture0,
                                                   this.EmitterData.TexPatData0.UI_texPatFileName );
                // テクスチャ１
                yield return new EmitterAssetInfo( this,
                                                   EmitterAssetTypes.EmitterTexture1,
                                                   this.EmitterData.TexPatData1.UI_texPatFileName );
                // テクスチャ２
                yield return new EmitterAssetInfo( this,
                                                   EmitterAssetTypes.EmitterTexture2,
                                                   this.EmitterData.TexPatData2.UI_texPatFileName );
                // チャイルド・テクスチャー
                if ( this.EmitterData.ChildFlag.Enable==true )
                {
                    yield return new EmitterAssetInfo( this,
                                                       EmitterAssetTypes.ChildTexture0,
                                                       this.EmitterData.ChildTexPatData.UI_texPatFileName );
                }
            }
        }

        /// <summary>
        /// 関連するプリミティブのパスを取得します
        /// </summary>
        public IEnumerable<string> PrimitivePaths
        {
            get
            {
                if ( this.EmitterData.MeshType==MeshType.Primitive )
                    yield return this.EmitterData.BillboardPrimitiveFileSource;

                if ( this.EmitterData.ChildFlag.Enable==true &&
                     this.EmitterData.ChildData.MeshType==MeshType.Primitive )
                {
                    yield return this.EmitterData.ChildData.BillboardPrimitiveFileSource;
                }
            }
        }

        /// <summary>
        /// 関連するプリミティブを取得します
        /// </summary>
        public IEnumerable<EmitterAssetInfo> Primitives
        {
            get
            {
                if ( this.EmitterData.MeshType==MeshType.Primitive )
                {
                    yield return new EmitterAssetInfo( this,
                                                       EmitterAssetTypes.EmitterPrimitive,
                                                       this.EmitterData.BillboardPrimitiveFileSource );
                }

                if ( this.EmitterData.ChildFlag.Enable==true &&
                     this.EmitterData.ChildData.MeshType==MeshType.Primitive )
                {
                    yield return new EmitterAssetInfo( this,
                                                       EmitterAssetTypes.ChildPrimitive,
                                                       this.EmitterData.ChildData.BillboardPrimitiveFileSource );
                }
            }
        }

        /// <summary>
        /// accessor for emitter data model encapsulated in this document
        /// </summary>
        public EmitterData EmitterData
        {
            get { return this.m_editData; }
        }

        /// <summary>
        /// Emitter type
        /// </summary>
        public EmitterType EmitterType
        {
            get
            {
                return this.EmitterData.Type;
            }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "EmitterData";
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return this.EmitterSetDocument.DataScrPath +
                       ".Emitter_uid" +
                       this.DocumentID.ToString();
            }
        }

        /// <summary>
        /// Sub Documents
        /// </summary>
        public override IEnumerable<IDocument> SubDocuments
        {
            get
            {
                yield return ChildDocument;
                yield return FieldDocument;
                yield return FluctuationDocument;
            }
        }

        /// <summary>
        /// child documents contained by this emitter
        /// </summary>
        public ChildDocument ChildDocument
        {
            get
            {
                return this.m_childDocument;
            }

            set
            {
                if (this.m_childDocument != value)
                {
                    this.m_childDocument = value;
                    NotifyDataChanged();
                }
            }
        }

        /// <summary>
        /// filed document contained by this emitter
        /// </summary>
        public FieldDocument FieldDocument
        {
            get
            {
                return this.m_filedDocument;
            }

            set
            {
                if (this.m_filedDocument != value)
                {
                    this.m_filedDocument = value;
                    NotifyDataChanged();
                }
            }
        }

        /// <summary>
        /// fluctuation document contained by this emitter
        /// </summary>
        public FluctuationDocument FluctuationDocument
        {
            get
            {
                return this.m_fluctunationDocument;
            }

            set
            {
                if (this.m_fluctunationDocument != value)
                {
                    this.m_fluctunationDocument = value;
                    NotifyDataChanged();
                }
            }
        }


        /// <summary>
        /// Get the data model proxy.
        /// </summary>
        public DataModelProxy DataModelProxy
        {
            get { return m_dataModelProxy; }
        }


        /// <summary>
        /// Check if the color 0 of the emitter data is locked to "Constant Color".
        /// </summary>
        public bool IsColor0Locked
        {
            get
            {
                #if BUILD_FOR_CTR
                    if ( this.EmitterData.AnimEditData.Color1.ColorType==Constances.ColorSettingType.Constant )
                        return false;
                    else
                        return true;
                #else
                    return false;
                #endif
            }
        }


        /// <summary>
        /// Check if the color 1 of the emitter data is locked to "Constant Color".
        /// </summary>
        public bool IsColor1Locked
        {
            get
            {
                #if BUILD_FOR_CTR
                    if ( this.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Constant )
                        return false;
                    else
                        return true;
                #else
                    return false;
                #endif
            }
        }

        #endregion

        #region Owner document

        /// <summary>
        /// Set owner document.
        /// This method can move the document to another owner.
        /// </summary>
        /// <param name="owner">The new owner.</param>
        /// <param name="iIndex">The index the document should be placed at.</param>
        /// <returns>True on success.</returns>
        public bool SetOwnerDocument( IEmitterSetDocument owner,
                                      int iIndex )
        {
            if ( owner==null )
                return false;

            if ( owner.Count>=TheApp.MAX_EMITTER_COUNT )
                return false;

            this.EmitterSetDocument.RemoveEmitterDocument( this );

            this.EmitterSetDocument = owner;
            this.EmitterSetDocument.AddEmitterDocument( this, iIndex );

            return true;
        }

        #endregion

        #region Data model for dev tab

        /// <summary>
        /// Initialize the data model proxy for the dev tab.
        /// </summary>
        private void InitDataModelProxy()
        {
            // Create and register data model proxy
            m_dataModelProxy = new DataModelProxy( this );
        }


        /// <summary>
        /// Determine if the data mode with the given ID is enabled.
        /// </summary>
        /// <param name="iDataModelID">The data model ID.</param>
        /// <returns>True on enabled.</returns>
        public bool IsDataModelEnabled( uint iDataModelID )
        {
            return true;
        }

        #endregion

        #region Create / remove child documents

        /// <summary>
        /// Create child document. The method WILL replace the original one
        /// with the created document, make sure you've saved the original
        /// one if you need it.
        /// </summary>
        /// <returns>The created child document or null on failure.</returns>
        public ChildDocument CreateChildDocument()
        {
            if ( this.ChildDocument!=null )
                RemoveChildDocument();

            ChildDocument doc  = new ChildDocument(this);
            this.ChildDocument = doc;
            this.ChildDocument.NotifyAddedToParent();

            /*
            string docPath;

            // Get the data source path of the emitter document
            docPath = this.DataScrPath + "." +
                      this.RelativeDataScrPath +
                      ".ChildData.BillboardPrimitiveFileSource";

            // Add the command listener.

            TheApp.CommandManager.AddCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                      docPath,
                                                      doc.OnPrimitivePathChanged);

            // Get the data source path of the emitter document
            docPath = this.DataScrPath + "." +
                      this.RelativeDataScrPath +
                      ".ChildTexPatData.UI_texPatFileName";

            // Add the command listener.
            TheApp.CommandManager.AddCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                      docPath,
                                                      doc.OnChildTexturePathChanged);
            */

            return doc;
        }


        /// <summary>
        /// Create field document. The method WILL replace the original one
        /// with the created document, make sure you've saved the original
        /// one if you need it.
        /// </summary>
        /// <returns>The created field document or null on failure.</returns>
        public FieldDocument CreateFieldDocument()
        {
            if ( this.FieldDocument!=null )
                RemoveFieldDocument();

            FieldDocument doc = new FieldDocument(this);

            this.FieldDocument = doc;
            this.FieldDocument.NotifyAddedToParent();

            return doc;
        }


        /// <summary>
        /// Create fluctuation document. The method WILL replace the original
        /// one with the created document, make sure you've saved the original
        /// one if you need it.
        /// </summary>
        /// <returns>The created fluctuation document or null on failure.</returns>
        public FluctuationDocument CreateFluctuationDocument()
        {
            if ( this.FluctuationDocument!=null )
                RemoveFluctuationDocument();

            FluctuationDocument doc = new FluctuationDocument(this);

            this.FluctuationDocument = doc;
            this.FluctuationDocument.NotifyAddedToParent();

            return doc;
        }


        /// <summary>
        /// Remove child document from the emitter document. The removed
        /// document is returned for further use.
        /// </summary>
        /// <returns>The removed document or null if it was already removed.</returns>
        public ChildDocument RemoveChildDocument()
        {
            ChildDocument doc = this.ChildDocument;
            if ( doc==null)
                return null;

            this.ChildDocument.NotifyRemovedFromParent();
            this.ChildDocument = null;

            //string docPath;

            /*
             *  Primitive変更のイベントリスナー
             */
            // Get the data source path of the emitter document
            /*
            docPath = this.DataScrPath + "." +
                      this.RelativeDataScrPath +
                      ".ChildData.BillboardPrimitiveFileSource";

            // Remove the command listener.
            TheApp.CommandManager.RemoveCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                         docPath,
                                                         doc.OnPrimitivePathChanged );
            */

            /*
             *  テクスチャ変更のイベントリスナー
             */
            // Get the data source path of the emitter document
            /*
            docPath = this.DataScrPath + "." +
                      this.RelativeDataScrPath +
                      ".ChildTexPatData.UI_texPatFileName";

            // Remove the command listener.
            TheApp.CommandManager.RemoveCommandListener( CmdEvtInterestTypes.EditedDataSource,
                                                         docPath,
                                                         doc.OnChildTexturePathChanged );
            */

            // Notify the document that it's been removed
            doc.NotifyRemovedFromParent();

            return doc;
        }


        /// <summary>
        /// Remove field document from the emitter document. The removed
        /// document is returned for further use.
        /// </summary>
        /// <returns>The removed document or null if it was already removed.</returns>
        public FieldDocument RemoveFieldDocument()
        {
            FieldDocument doc = this.FieldDocument;
            if ( doc==null )
                return null;

            this.FieldDocument.NotifyRemovedFromParent();
            this.FieldDocument = null;

            // Notify the document that it's been removed
            doc.NotifyRemovedFromParent();

            return doc;
        }


        /// <summary>
        /// Remove fluctuation document from the emitter document. The removed
        /// document is returned for further use.
        /// </summary>
        /// <returns>The removed document or null if it was already removed.</returns>
        public FluctuationDocument RemoveFluctuationDocument()
        {
            FluctuationDocument doc = this.FluctuationDocument;
            if ( doc==null )
                return null;

            this.FluctuationDocument.NotifyRemovedFromParent();
            this.FluctuationDocument = null;

            // Notify the document that it's been removed
            doc.NotifyRemovedFromParent();

            return doc;
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            NotifyRemovedFromParent();

            if ( this.IsRemoved==true )
                TheApp.UserConfigChanged += OnAppUserConfigChanged;

            base.NotifyAddedToParent();

            RegisterPrimitiveFileMonitor();

            TheApp.OnGammaCorrectionModeChanged += this.OnGammaCorrectionModeChanged;

            #if BUILD_FOR_CTR

            TheApp.CommandManager.BeforeExecution += OnBeforeCommandExecuted;
            TheApp.CommandManager.AddCommandListener( OnCommandExecuted );

            #endif

            DocumentManager.RegisterDocumentPath( this, this.DataScrPath );

            // Notify the documents that they have been added
            if ( this.ChildDocument!=null )
                this.ChildDocument.NotifyAddedToParent();

            if ( this.FieldDocument!=null )
                this.FieldDocument.NotifyAddedToParent();

            if ( this.FluctuationDocument!=null )
                this.FluctuationDocument.NotifyAddedToParent();

            DocumentManager.RegisterDataModelProxy( m_dataModelProxy );

            // Register animation table list.
            if ( m_editData.AnimTableList!=null )
            {
                TheApp.AnimationTableManager.RegisterAnimationTableList( this,
                                                                         m_editData.AnimTableList );
            }
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            // Unregister the animation table list.
            TheApp.AnimationTableManager.UnregisterAnimationTableList( this );

            // Notify the documents that they are going to been removed
            if ( this.ChildDocument!=null )
                this.ChildDocument.NotifyRemovedFromParent();

            if ( this.FieldDocument!=null )
                this.FieldDocument.NotifyRemovedFromParent();

            if ( this.FluctuationDocument!=null )
                this.FluctuationDocument.NotifyRemovedFromParent();

            if ( m_dataModelProxy!=null )
                DocumentManager.UnregisterDataModelProxy( m_dataModelProxy );

            TheApp.UserConfigChanged -= OnAppUserConfigChanged;

            DocumentManager.UnregisterDocumentPath( this, this.DataScrPath );

            #if BUILD_FOR_CTR

            TheApp.CommandManager.RemoveCommandListener( OnCommandExecuted );
            TheApp.CommandManager.BeforeExecution -= OnBeforeCommandExecuted;

            #endif

            TheApp.OnGammaCorrectionModeChanged -= this.OnGammaCorrectionModeChanged;

            UnregisterPrimitiveFileMonitor();

            base.NotifyRemovedFromParent();
        }

        #endregion

        #region Utility Functions

        internal static void CheckPrimitivePathValidity( string primitivePath,
                                                         IEmitterSetDocument emitterSetDocument )
        {
            if (emitterSetDocument == null)
                throw new ArgumentNullException("emitterSetDocument");

            if (string.IsNullOrEmpty(primitivePath) == false &&
                emitterSetDocument.FileLocation.Length > 0 &&
                DocumentConstants.ValidatePrimitivePath(emitterSetDocument.FilePath,
                                                        primitivePath) == false)
            {
                /*
                using (var context = new OptionalMessageBoxContext("PrimitiveUnreachableMsg"))
                {
                    context.AssignDetailButtonClickHandler(
                        delegate
                        {
                            HelpEx.ShowHelp(Form.ActiveForm,
                                          TheApp.HelpNamespace,
                                          "/html/guide/emitterset_concept/emitterset_concept.html#PRIMITIVE");
                        });

                    UIMessageBox.Show(StringResource.Get("WARNING_PRIMITIVE_LOC_NOT_REACHABLE",
                                                         Path.GetFileName(primitivePath)),
                                      UIMessageBoxButtons.OKDetail,
                                      MessageBoxIcon.Warning,
                                      MessageBoxDefaultButton.Button1);
                }
                */
            }
        }

        internal static void CheckTexturePathValidity( string texturePath,
                                                       IEmitterSetDocument emitterSetDocument )
        {
            if (emitterSetDocument == null)
                throw new ArgumentNullException("emitterSetDocument");

            if (string.IsNullOrEmpty(texturePath) == false &&
                emitterSetDocument.FileLocation.Length > 0 &&
                DocumentConstants.ValidateTexturePath(emitterSetDocument.FilePath,
                                                      texturePath) == false)
            {
                //TextureLocationUnreachable.DisplayDialog(texturePath);
                /*
                using (var context = new OptionalMessageBoxContext("TextureUnreachableMsg"))
                {
                    context.AssignDetailButtonClickHandler(
                        delegate
                        {
                            HelpEx.ShowHelp(Form.ActiveForm,
                                          TheApp.HelpNamespace,
                                          "/html/guide/emitterset_concept/emitterset_concept.html#TEXTURE");
                        });

                    UIMessageBox.Show(StringResource.Get("WARNING_TEXTURE_LOC_NOT_REACHABLE",
                                                         Path.GetFileName(texturePath)),
                                       UIMessageBoxButtons.OKDetail,
                                       MessageBoxIcon.Warning,
                                       MessageBoxDefaultButton.Button1);
                }
                */
            }
        }

        /// <summary>
        /// Check primitive location.
        /// If the primitive is not in the predefined primitive search directories,
        /// show the warning.
        /// </summary>
        private void CheckPrimitivePathValidity()
        {
            string primitivePath = EmitterData.BillboardPrimitiveFileSource;

            CheckPrimitivePathValidity(primitivePath, EmitterSetDocument);
        }

        /// <summary>
        /// Check texture location.
        /// If the texture is not in the predefined texture search directories,
        /// show the warning.
        /// </summary>
        private void CheckTexture0PathValidity()
        {
            string texturePath = EmitterData.TexPatData0.UI_texPatFileName;

            CheckTexturePathValidity( texturePath, EmitterSetDocument );
        }

        /// <summary>
        /// Check sub-texture location.
        /// If the texture is not in the predefined texture search directories,
        /// show the warning.
        /// </summary>
        private void CheckTexture1PathValidity()
        {
            string texturePath = EmitterData.TexPatData1.UI_texPatFileName;

            CheckTexturePathValidity( texturePath, EmitterSetDocument );
        }

        /// <summary>
        /// Check texture 2 location.
        /// If the texture is not in the predefined texture search directories,
        /// show the warning.
        /// </summary>
        private void CheckTexture2PathValidity()
        {
            string texturePath = EmitterData.TexPatData2.UI_texPatFileName;

            CheckTexturePathValidity( texturePath, EmitterSetDocument );
        }

        /// <summary>
        /// Checks if the assets are reachable, and report it otherwise
        /// </summary>
        public override void CheckAssetsReachability( bool bUpdateProjectTree )
        {
            int unreachableAssetCount = 0;
            int typeMismatchTextureCount = 0;

            // Primitive
            string primitivePath = this.EmitterData.BillboardPrimitiveFileSource;
            if ( DocumentConstants.NoSuchAssetPath( primitivePath ) )
            {
                ++unreachableAssetCount;
            }

            // Texture 0
            TextureResourceHelper.CheckTextureReachability(EmitterData.TexPatData0.UI_texPatFileName,
                ref unreachableAssetCount, ref typeMismatchTextureCount);

            // Texture 1
            TextureResourceHelper.CheckTextureReachability(EmitterData.TexPatData1.UI_texPatFileName,
                ref unreachableAssetCount, ref typeMismatchTextureCount);

            // Texture 2
            TextureResourceHelper.CheckTextureReachability(EmitterData.TexPatData2.UI_texPatFileName,
                ref unreachableAssetCount, ref typeMismatchTextureCount);

            // Child (supposed to be up to date; do not update from here since it is a bottom up update)
            if ( m_childDocument!=null )
            {
                unreachableAssetCount += m_childDocument.SelfUnreachableAssetCount;
                typeMismatchTextureCount += m_childDocument.SelfTypeMismatchTextureCount;
            }

            UpdateUnreachableAssetCount( unreachableAssetCount, bUpdateProjectTree );
            UpdateTypeMismatchTextureCount( typeMismatchTextureCount, bUpdateProjectTree );
        }

        /// <summary>
        /// Verify the texture paths of this emitter and/or its child are set.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <returns>True if all the texture paths are set.</returns>
        public bool VerifyTexturePathsAreSet(string docPath)
        {
            return VerifyTexturePaths(docPath, true, false);
        }

        /// <summary>
        /// Verify the texture paths of this emitter and/or its child.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <returns>True if all the texture paths are valid.</returns>
        public bool VerifyTexturePathsAreReachable(string docPath)
        {
            return VerifyTexturePaths(docPath, false, true);
        }

        /// <summary>
        /// Verify the texture paths of this emitter and/or its child.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <param name="bCheckTexNotSet">True to check if the texture is set.</param>
        /// <param name="bCheckPath">True to check if the path is reachable.</param>
        /// <returns>True if all the texture paths are set and valid.</returns>
        private bool VerifyTexturePaths( string docPath,
                                         bool bCheckTexNotSet,
                                         bool bCheckPath )
        {
            bool bValid = true;

            // Emitter texture 0
            bValid &= CheckTexturePathIsSetAndReachable( docPath,
                                                         bCheckTexNotSet,
                                                         bCheckPath,
                                                         this.EmitterData.TexPatData0.UI_texPatFileName,
                                                         res.Strings.LOG_MSG_TEXTURE_0_NOT_SET,
                                                         res.Strings.LOG_MSG_TEXTURE_0_NOT_FOUND );

            // Emitter texture 1
            bValid &= CheckTexturePathIsSetAndReachable( docPath,
                                                         false,
                                                         bCheckPath,
                                                         this.EmitterData.TexPatData1.UI_texPatFileName,
                                                         null,
                                                         res.Strings.LOG_MSG_TEXTURE_1_NOT_FOUND );

            // Emitter texture 2
            bValid &= CheckTexturePathIsSetAndReachable( docPath,
                                                         false,
                                                         bCheckPath,
                                                         this.EmitterData.TexPatData2.UI_texPatFileName,
                                                         null,
                                                         res.Strings.LOG_MSG_TEXTURE_2_NOT_FOUND );

            // Child texture
            if ( this.EmitterType==EmitterType.Complex &&
                 this.EmitterData.ChildFlag.Enable==true )
            {
                bValid &= CheckTexturePathIsSetAndReachable( docPath,
                                                             bCheckTexNotSet,
                                                             bCheckPath,
                                                             this.EmitterData.ChildTexPatData.UI_texPatFileName,
                                                             res.Strings.LOG_MSG_CHILD_TEXTURE_NOT_SET,
                                                             res.Strings.LOG_MSG_CHILD_TEXTURE_NOT_FOUND );
            }

            return bValid;
        }

        /// <summary>
        /// Verify the primitve paths of this emitter and/or its child.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <returns>True if all the primitive paths are valid.</returns>
        public bool VerifyPrimitivePathsAreReachable( string docPath )
        {
            bool bValid = true;

            // Emitter primitive
            bValid &= CheckPrimitivePathIsReachable( docPath,
                                                     this.EmitterData.BillboardPrimitiveFileSource,
                                                     res.Strings.LOG_MSG_PRIMITIVE_NOT_FOUND );

            // Child primitive
            if ( this.EmitterType==EmitterType.Complex &&
                 this.EmitterData.ChildFlag.Enable==true )
            {
                bValid &= CheckPrimitivePathIsReachable( docPath,
                                                         this.EmitterData.ChildData.BillboardPrimitiveFileSource,
                                                         res.Strings.LOG_MSG_CHILD_PRIMITIVE_NOT_FOUND);
            }

            return bValid;
        }


        /// <summary>
        /// Verify the texture path is set and reachable.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <param name="bCheckTexNotSet"></param>
        /// <param name="bCheckPath"></param>
        /// <param name="texturePath">Path of the texture.</param>
        /// <param name="notSetMessage">Message to log if the path is not set.</param>
        /// <param name="notFoundMessage">Message to log if the path is not reachable.</param>
        /// <returns>True if the texture path is set and valid.</returns>
        private bool CheckTexturePathIsSetAndReachable( string docPath,
                                                        bool bCheckTexNotSet,
                                                        bool bCheckPath,
                                                        string texturePath,
                                                        string notSetMessage,
                                                        string notFoundMessage )
        {
            bool bValid = true;

            if ( string.IsNullOrEmpty( texturePath ) )
            {
                if ( bCheckTexNotSet==true )
                {
                    TheApp.Logger.Warn.FormatMessage( notSetMessage,
                                                      this.EmitterSetDocument.Name,
                                                      this.Name);
                    bValid = false;
                }
            }
            else if ( bCheckPath==true &&
                      !string.IsNullOrEmpty( docPath ) )
            {
                string fullPath;
                if ( DocumentConstants.LocateTextureByFileName( docPath,
                                                                texturePath,
                                                                out fullPath )==false )
                {
                    TheApp.Logger.Warn.FormatMessage( notFoundMessage,
                                                      this.EmitterSetDocument.Name,
                                                      this.Name,
                                                      texturePath);
                    bValid = false;
                }
            }
            return bValid;
        }

        /// <summary>
        /// Verify the primitve path is reachable.
        /// </summary>
        /// <param name="docPath">Use this path as document path to verify the textures.</param>
        /// <param name="primitivePath">Path of the primitive.</param>
        /// <param name="notFoundMessage">Message to log if the path is not reachable.</param>
        /// <returns>True if the primitive path is valid.</returns>
        private bool CheckPrimitivePathIsReachable( string docPath,
                                                    string primitivePath,
                                                    string notFoundMessage )
        {
            bool bValid = true;

            if ( !string.IsNullOrEmpty( primitivePath ) &&
                 !string.IsNullOrEmpty( docPath ) )
            {
                string fullPath;
                if ( DocumentConstants.LocatePrimitiveByFileName( docPath,
                                                                  primitivePath,
                                                                  out fullPath )==false )
                {
                    TheApp.Logger.Warn.FormatMessage( notFoundMessage,
                                                      this.EmitterSetDocument.Name,
                                                      this.Name,
                                                      primitivePath );
                    bValid = false;
                }
            }
            return bValid;
        }

        /// <summary>
        /// notify child list has been changed
        /// </summary>
        public void NotifyDataChanged()
        {
//            ProjectManager.UpdateAll();
        }

        /// <summary>
        /// apply / convert assist data to original data value
        /// </summary>
        /// <param name="data">data to be converted</param>
        public bool ApplyAssistData(EmitterData data = null)
        {
            bool isForceSet = false;		// 値を強制セットしたか？
            return isForceSet;
        }

        /// <summary>
        /// Duplicate this emitter document with the given name and owner emitter set.
        /// </summary>
        /// <param name="name">Name for the new emitter document.</param>
        /// <param name="project">Owner emitter set of the new emitter document.</param>
        /// <returns>The created emitter document.</returns>
        public EmitterDocument Duplicate( string name,
                                          EmitterSetDocument owner )
        {
            EmitterData emitterData = new EmitterData( this.EmitterData );
            if ( emitterData==null )
                return null;

            emitterData.Name = string.Copy( name );

            EmitterDocument doc = new EmitterDocument( owner, emitterData );

            // Load texture file 0
            string fullPath = String.Empty;
            if ( DocumentIO.ReadTextureFile( this.FileLocation,
                                             this.EmitterData.TexPatData0.UI_texPatFileName,
                                             this.EmitterData.TexRes0,
                                             ref fullPath )==true )
            {
                this.EmitterData.TexPatData0.UI_texPatFileName = fullPath;
            }

            // Load texture file 1
            fullPath = String.Empty;
            if ( DocumentIO.ReadTextureFile( this.FileLocation,
                                             this.EmitterData.TexPatData1.UI_texPatFileName,
                                             this.EmitterData.TexRes1,
                                             ref fullPath )==true )
            {
                this.EmitterData.TexPatData1.UI_texPatFileName = fullPath;
            }

            // Load texture file 2
            fullPath = String.Empty;
            if ( DocumentIO.ReadTextureFile( this.FileLocation,
                                             this.EmitterData.TexPatData2.UI_texPatFileName,
                                             this.EmitterData.TexRes2,
                                             ref fullPath )==true )
            {
                this.EmitterData.TexPatData2.UI_texPatFileName = fullPath;
            }

            return doc;
        }

        /// <summary>
        /// Copy the data from the given source document data.
        /// </summary>
        /// <param name="srcData">The source document data.</param>
        public void Copy( EmitterData srcData )
        {
            // Remember the document name first.
            string name = this.Name;

            // Copy the data model.
            this.EmitterData.Set( srcData );

            // Changed the document name.
            this.EmitterData.Name = string.Copy( name );

            // Create / remove the child documents.
            if ( this.EmitterData.Type==EmitterType.Simple )
            {
                if ( this.ChildDocument!=null )
                    RemoveChildDocument();

                if ( this.FieldDocument!=null )
                    RemoveFieldDocument();

                if ( this.FluctuationDocument!=null )
                    RemoveFluctuationDocument();
            }
            else if ( this.EmitterData.Type==EmitterType.Complex )
            {
                if ( this.ChildDocument==null )
                    CreateChildDocument();

                if ( this.FieldDocument==null )
                    CreateFieldDocument();

                if ( this.FluctuationDocument==null )
                    CreateFluctuationDocument();
            }
        }

        /// <summary>
        /// Copy the field data from the given source document data.
        /// </summary>
        /// <param name="srcData"></param>
        public void CopyFieldData(EmitterData srcData)
        {
            this.EmitterData.FieldFlag.Set(srcData.FieldFlag);
            this.EmitterData.FieldCollisionData.Set(srcData.FieldCollisionData);
            this.EmitterData.FieldConvergenceData.Set(srcData.FieldConvergenceData);
            this.EmitterData.FieldMagnetData.Set(srcData.FieldMagnetData);
            this.EmitterData.FieldPosAddData.Set(srcData.FieldPosAddData);
            this.EmitterData.FieldRandomData.Set(srcData.FieldRandomData);
            this.EmitterData.FieldSpinData.Set(srcData.FieldSpinData);

            this.EmitterData.UI_hostioEmitterFlg.DispFieldCollision     = srcData.UI_hostioEmitterFlg.DispFieldCollision;
            this.EmitterData.UI_hostioEmitterFlg.DispFieldConvergence   = srcData.UI_hostioEmitterFlg.DispFieldConvergence;
            this.EmitterData.UI_hostioEmitterFlg.DispFieldMagnet        = srcData.UI_hostioEmitterFlg.DispFieldMagnet;
            this.EmitterData.UI_hostioEmitterFlg.DispFieldSpin          = srcData.UI_hostioEmitterFlg.DispFieldSpin;

            #region Copy animation tables

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_random_x;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_random_y;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_random_z;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_magnet;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_span_rot;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_span_outer;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_convergence;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_pos_add_x;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_pos_add_y;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.field_pos_add_z;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            #endregion
        }

        /// <summary>
        /// Copy the fluctuation data from the given source document data.
        /// </summary>
        /// <param name="srcData"></param>
        public void CopyFluctuationData(EmitterData srcData)
        {
            this.EmitterData.FluctuationData.Set(srcData.FluctuationData);
            this.EmitterData.FluctuationFlag.Set(srcData.FluctuationFlag);
        }

        /// <summary>
        /// Copy the child data from the given source document data.
        /// </summary>
        /// <param name="srcData"></param>
        public void CopyChildData(EmitterData srcData)
        {
            this.EmitterData.ChildData.Set(srcData.ChildData);
            this.EmitterData.ChildAssistData.Set(srcData.ChildAssistData);
            this.EmitterData.ChildFlag.Set(srcData.ChildFlag);
            this.EmitterData.ChildTexPatData.Set(srcData.ChildTexPatData);

            #region Copy animation tables

            {
                AnimTableTargetTypes target = AnimTableTargetTypes.gravity_child;

                AnimTableData dst = this.EmitterData.AnimTableList.FindAnimationTableData( target );
                AnimTableData src = srcData.AnimTableList.FindAnimationTableData( target );
                dst.Set( src );
            }

            #endregion
        }

        /// <summary>
        /// Check if the given document can be a child of this document.
        /// E.q. Emitter can be emitter set's child, but cannot be a child
        /// of an effect project document.
        /// </summary>
        /// <param name="doc">The child document.</param>
        /// <returns>True if the given document is a valid child document type.</returns>
        public override bool CanAddAsChildDoc( IDocument doc )
        {
            if ( doc is ChildDocument )
                return true;
            else if ( doc is FieldDocument )
                return true;
            else if ( doc is FluctuationDocument )
                return true;

            return false;
        }

        #endregion

        #region Drag & drop

        /// <summary>
        /// Check if the document can be dragged on project tree view.
        /// </summary>
        /// <returns>True if can be dragged.</returns>
        public override bool CanDrag()
        {
            return true;
        }


        /// <summary>
        /// Check if the given document can be dropped and inserted before this document.
        /// </summary>
        /// <param name="doc">The dropped document.</param>
        /// <returns>True if the given document can be dropped.</returns>
        public override bool CanDrop( IDocument doc )
        {
            if ( doc is EmitterDocument && doc.OwnerDocument==this.OwnerDocument )
                return true;

            return false;
        }

        #endregion

        #region Modification state

        /// <summary>
        /// Clear the modified flag.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public override int ClearModifiedFlag( bool bUpdateProjectTree )
        {
            if ( this.ChildDocument!=null )
                this.ChildDocument.ClearModifiedFlag( false );

            if ( this.FieldDocument!=null )
                this.FieldDocument.ClearModifiedFlag( false );

            if ( this.FluctuationDocument!=null )
                this.FluctuationDocument.ClearModifiedFlag( false );

            return base.ClearModifiedFlag( bUpdateProjectTree );
        }

        #endregion

        #region Event handlers

        /// <summary>
        /// Gamma correction mode changed event
        /// </summary>
        private void OnGammaCorrectionModeChanged()
        {
            ReloadTexture0(null);
            ReloadTexture1(null);
            ReloadTexture2(null);
        }


        /// <summary>
        /// Map the draw path name to the corresponding ID in the user config file.
        /// </summary>
        /// <returns>True if the ID is successfully mapped.</returns>
        public bool UpdateDrawPathID( bool bSilent )
        {
            if ( TheApp.UserConfigData==null )
            {
                // The user config data is empty, so set the draw path ID to zero.
                this.EmitterData.UI_drawPathID = 0;
                this.EmitterData.UserShaderCompileDef1 = string.Empty;
                this.EmitterData.UserShaderCompileDef2 = string.Empty;
                this.EmitterData.ChildData.UserShaderCompileDef1 = string.Empty;
                this.EmitterData.ChildData.UserShaderCompileDef2 = string.Empty;
                return false;
            }

            // No draw paths are set in the user config file,
            // and emitter draw path is also empty.
            // Just bail out.
            if ( TheApp.UserConfigData.DrawPaths==null ||
                 TheApp.UserConfigData.DrawPaths.Length<=0 )
            {
                return true;
            }

            UserConfigData.DrawPathData data =
                TheApp.UserConfigData.FindDrawPathByText( this.EmitterData.UI_drawPath );
            if ( data==null )
            {
                // Just skip the reset / updates if on console mode.
                if ( TheApp.ConsoleMode==true )
                    return false;

                // On silent mode, just skip the updating.
                if ( bSilent==true )
                    TheApp.SetUpdateDrawPathDialogAction( false );

                // The draw path data cannot be found in the new user config file.
                bool bUpdateDrawPath = false;

                if ( TheApp.ShouldSkipUpdateDrawPathDialog()==false )
                {
                    /*
                    App.IO.UpdateInvalidDrawPathDialog dialog =
                        new App.IO.UpdateInvalidDrawPathDialog( this.EmitterSetDocument.Name + " : " + this.Name );

                    DialogResult result =
                        MainFrame.SynchronizationContext.SafelyProduce( () => dialog.ShowDialog(TheApp.MainFrame) );
                    if ( result==DialogResult.Yes )
                        bUpdateDrawPath = true;
                    else
                        bUpdateDrawPath = false;

                    if ( dialog.IsApplyToAll==true )
                        TheApp.SetUpdateDrawPathDialogAction( bUpdateDrawPath );
                    */
                }
                else
                {
                    bUpdateDrawPath = TheApp.ShouldUpdateDrawPath();
                }

                // Should we update / reset the draw path?
                if ( bUpdateDrawPath==true )
                {
                    #region Create command and arguments to update the values

                    // Source data path to EmitterData.UI_drawPath
                    string srcPath1 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".UI_drawPath";

                    // Source data path to EmitterData.UI_drawPathID
                    string srcPath2 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".UI_drawPathID";

                    // Source data path to EmitterData.UserShaderCompileDef1
                    string srcPath3 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".UserShaderCompileDef1";

                    // Source data path to EmitterData.UserShaderCompileDef2
                    string srcPath4 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".UserShaderCompileDef2";

                    // Source data path to EmitterData.ChildData.UserShaderCompileDef1
                    string srcPath5 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".ChildData.UserShaderCompileDef1";

                    // Source data path to EmitterData.ChildData.UserShaderCompileDef2
                    string srcPath6 = this.DataScrPath + "." +
                                      this.RelativeDataScrPath +
                                      ".ChildData.UserShaderCompileDef2";

                    // The argument list for the command to execute.
                    /*
                    List<PropertyEditCommandArgs> argList = new List<PropertyEditCommandArgs>(2);

                    // Command argument for EmitterData.UI_drawPath
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath1),
                                                             this,
                                                             this.EmitterData.UI_drawPath,
                                                             TheApp.UserConfigData.DefaultDrawPath) );

                    // Command argument for EmitterData.UI_drawPathID
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath2),
                                                             this,
                                                             this.EmitterData.UI_drawPathID,
                                                             TheApp.UserConfigData.DefaultDrawPathID) );

                    // Command argument for EmitterData.UserShaderCompileDef1
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath3),
                                                             this,
                                                             this.EmitterData.UserShaderCompileDef1,
                                                             TheApp.UserConfigData.DefaultShaderCompileDef1) );

                    // Command argument for EmitterData.UserShaderCompileDef2
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath4),
                                                             this,
                                                             this.EmitterData.UserShaderCompileDef2,
                                                             TheApp.UserConfigData.DefaultShaderCompileDef2) );

                    // Command argument for EmitterData.ChildData.UserShaderCompileDef1
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath5),
                                                             this,
                                                             this.EmitterData.ChildData.UserShaderCompileDef1,
                                                             TheApp.UserConfigData.DefaultShaderCompileDef1) );

                    // Command argument for EmitterData.ChildData.UserShaderCompileDef2
                    argList.Add( new PropertyEditCommandArgs(new NWCore.Utility.Path(""),
                                                             new NWCore.Utility.Path(srcPath6),
                                                             this,
                                                             this.EmitterData.ChildData.UserShaderCompileDef2,
                                                             TheApp.UserConfigData.DefaultShaderCompileDef2) );

                    // The data path to display on the history dialog.
                    string displayDataPath = this.EmitterSetDocument.Name + "." + this.Name;

                    // Execute the command.
                    TheApp.CommandManager.Execute( "PropertyEditCommand",
                                                   argList,
                                                   res.Strings.COMMAND_DESC_UPDATE_INVALID_DRAW_PATH,
                                                   displayDataPath,
                                                   this );
                    */

                    #endregion
                }
                else
                {
                    // Keep the draw path name but set the ID to zero.
                    this.EmitterData.UI_drawPathID = 0;
                    this.EmitterData.UserShaderCompileDef1 = string.Empty;
                    this.EmitterData.UserShaderCompileDef2 = string.Empty;
                    this.EmitterData.ChildData.UserShaderCompileDef1 = string.Empty;
                    this.EmitterData.ChildData.UserShaderCompileDef2 = string.Empty;
                }
            }
            else if ( data.ID!=this.EmitterData.UI_drawPathID )
            {
                this.EmitterData.UI_drawPathID = data.ID;
                this.EmitterData.UserShaderCompileDef1 = data.ShaderCompileDef1;
                this.EmitterData.UserShaderCompileDef2 = data.ShaderCompileDef2;
                this.EmitterData.ChildData.UserShaderCompileDef1 = data.ShaderCompileDef1;
                this.EmitterData.ChildData.UserShaderCompileDef2 = data.ShaderCompileDef2;

                // The draw path ID has changed, the emitter needs to
                // be sent to the viewer again.
                //App.Viewer.Message.SendUtility.SendDocumentBinary( this.EmitterSetDocument );
            }

            return true;
        }

        /// <summary>
        /// Handle UserConfigChanged event from the application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAppUserConfigChanged( object sender,
                                             EventArgs e )
        {
            UpdateDrawPathID( false );
        }


        /// <summary>
        /// Handle ConfigChanged event from the application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAppConfigChanged( object sender,
                                         EventArgs e )
        {
            RegisterPrimitiveFileMonitor();
        }


        /// <summary>
        ///
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HelpButtonClicked( object sender,
                                        System.ComponentModel.CancelEventArgs e )
        {
            HelpProvider hpp = new HelpProvider();

            hpp.HelpNamespace = TheApp.HelpNamespace;

            /*
            HelpEx.ShowHelp(Form.ActiveForm,
                           TheApp.HelpNamespace,
                           "/html/guide/emitterset_concept/emitterset_concept.html#TEXTURE" );
            */
        }

        /// <summary>
        ///  Notify the texture resource has been modified
        /// </summary>
        public void OnTexture0ResourceModified()
        {
            // Get the data source path of the texture resource.
            string docPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".TexRes0";
            TextureResourceHelper.IssueTextureResourceModifiedCommand(docPath, this, EmitterData.TexRes0);
        }

        /// <summary>
        ///  Notify the texture resource has been modified
        /// </summary>
        public void OnTexture1ResourceModified()
        {
            // Get the data source path of the texture resource.
            string docPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".TexRes1";
            TextureResourceHelper.IssueTextureResourceModifiedCommand(docPath, this, EmitterData.TexRes1);
        }

        /// <summary>
        ///  Notify the texture resource has been modified
        /// </summary>
        public void OnTexture2ResourceModified()
        {
            // Get the data source path of the texture resource.
            string docPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".TexRes2";
            TextureResourceHelper.IssueTextureResourceModifiedCommand(docPath, this, EmitterData.TexRes2);
        }

        /// <summary>
        /// Reload texture.
        /// </summary>
        public bool ReloadTexture0(string path)
        {
            bool result = false;
            m_bUpdatingTexture0 = true;

            if ( path==null ||
                 path==this.EmitterData.TexPatData0.UI_texPatFileName )
            {
                string fullPath = String.Empty;
                result = DocumentIO.ReadTextureFile(this.FileLocation,
                                                    this.EmitterData.TexPatData0.UI_texPatFileName,
                                                    this.EmitterData.TexRes0,
                                                    ref fullPath);

                OnTexture0ResourceModified();
                CheckAssetsReachability(true);
            }
            m_bUpdatingTexture0 = false;

            return result;
        }

        /// <summary>
        /// Reload texture 1.
        /// </summary>
        /// <returns></returns>
        public bool ReloadTexture1(string path)
        {
            bool result = false;
            m_bUpdatingTexture1 = true;

            if ( path==null ||
                 path==this.EmitterData.TexPatData1.UI_texPatFileName )
            {
                string fullPath = String.Empty;
                result = DocumentIO.ReadTextureFile(this.FileLocation,
                                                    this.EmitterData.TexPatData1.UI_texPatFileName,
                                                    this.EmitterData.TexRes1,
                                                    ref fullPath);

                OnTexture1ResourceModified();
                CheckAssetsReachability(true);
            }
            m_bUpdatingTexture1 = false;

            return result;
        }

        /// <summary>
        /// Reload texture 2.
        /// </summary>
        /// <returns></returns>
        public bool ReloadTexture2(string path)
        {
            bool result = false;
            m_bUpdatingTexture2 = true;

            if ( path==null ||
                 path==this.EmitterData.TexPatData2.UI_texPatFileName )
            {
                string fullPath = String.Empty;
                result = DocumentIO.ReadTextureFile(this.FileLocation,
                                                    this.EmitterData.TexPatData2.UI_texPatFileName,
                                                    this.EmitterData.TexRes2,
                                                    ref fullPath);

                OnTexture2ResourceModified();
                CheckAssetsReachability(true);
            }
            m_bUpdatingTexture2 = false;

            return result;
        }

        /// <summary>
        /// Handle primitive path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnPrimitivePathChanged( ICommand command,
                                            CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null )
                return;
            */

            RegisterPrimitiveFileMonitor();

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckPrimitivePathValidity();
            }

            CheckAssetsReachability(true);
        }

        /// <summary>
        /// Handle texture 0 path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnTexture0PathChanged( ICommand command,
                                           CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null )
                return;
            */

            if ( m_bUpdatingTexture0==true )
                return;

            m_bUpdatingTexture0 = true;

            // Get the data source path of the texture resource.
            string docPath = DataScrPath + "." +
                RelativeDataScrPath +
                ".TexPatData0.UI_texPatFileName";

            string texPatFileName = EmitterData.TexPatData0.UI_texPatFileName;
            //TextureResourceHelper.ReadTextureAndIssueCommand(docPath, propCommand.ControlPath, this, EmitterData.TexRes0, ref texPatFileName);
            EmitterData.TexPatData0.UI_texPatFileName = texPatFileName;

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckTexture0PathValidity();
            }
            OnTexture0ResourceModified();

            m_bUpdatingTexture0 = false;

            CheckAssetsReachability(true);
        }

        /// <summary>
        /// Handle texture 1 path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnTexture1PathChanged( ICommand command,
                                           CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null )
                return;
            */

            if ( m_bUpdatingTexture1==true )
                return;

            m_bUpdatingTexture1 = true;

            // Get the data source path of the texture resource.
            string docPath = DataScrPath + "." +
                RelativeDataScrPath +
                ".TexPatData1.UI_texPatFileName";

            string texPatFileName = EmitterData.TexPatData1.UI_texPatFileName;
            //TextureResourceHelper.ReadTextureAndIssueCommand(docPath, propCommand.ControlPath, this, EmitterData.TexRes1, ref texPatFileName);
            EmitterData.TexPatData1.UI_texPatFileName = texPatFileName;

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckTexture1PathValidity();
            }
            OnTexture1ResourceModified();

            m_bUpdatingTexture1 = false;

            CheckAssetsReachability(true);
        }

        /// <summary>
        /// Handle texture 2 path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnTexture2PathChanged( ICommand command,
                                          CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null )
                return;
            */

            if ( m_bUpdatingTexture2==true )
                return;

            m_bUpdatingTexture2 = true;

            // Get the data source path of the texture resource.
            string docPath = DataScrPath + "." +
                RelativeDataScrPath +
                ".TexPatData2.UI_texPatFileName";

            string texPatFileName = EmitterData.TexPatData2.UI_texPatFileName;
            //TextureResourceHelper.ReadTextureAndIssueCommand(docPath, propCommand.ControlPath, this, EmitterData.TexRes2, ref texPatFileName);
            EmitterData.TexPatData2.UI_texPatFileName = texPatFileName;

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckTexture2PathValidity();
            }
            OnTexture2ResourceModified();

            m_bUpdatingTexture2 = false;

            CheckAssetsReachability(true);
        }


        /// <summary>
        /// Handle command executed.
        /// </summary>
        /// <param name="command">The command executed.</param>
        /// <param name="executionType">The command execution type.</param>
        /// <param name="targetDocument">The target document.</param>
        private void OnCommandExecuted( ICommand command,
                                        CommandExecutionTypes executionType,
                                        IDocument targetDocument )
        {
            /*
            PropertyEditCommand cmd = command as PropertyEditCommand;
            if ( cmd==null )
                return;

            HandleColorTypeCommandExecuted( cmd, executionType, targetDocument );
            HandleCombinerColorTypeCommandExecuted( cmd, executionType, targetDocument );
            */
        }


        /// <summary>
        /// Handle BeforeCommandExecuted event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="args">The event arguments.</param>
        private void OnBeforeCommandExecuted( object sender,
                                              System.ComponentModel.CancelEventArgs args )
        {
            HandleColorTypeBeforeCommandExecuted( args );
        }

        #endregion

        #region Handle commands on specific data sources

        #region EmitterData.AnimEditData.Color*.ColorType

        /// <summary>
        /// Handle before command executed on color type data sources.
        /// </summary>
        /// <param name="args">The event arguments.</param>
        private void HandleColorTypeBeforeCommandExecuted( System.ComponentModel.CancelEventArgs args )
        {
            #if BUILD_FOR_CTR

            BeforeCommandExecutionEventArgs myArgs = args as BeforeCommandExecutionEventArgs;
            if ( myArgs==null )
                return;

            if ( myArgs.ExecutionType!=CommandExecutionTypes.Execute )
                return;

            PropertyEditCommand cmd = myArgs.Command as PropertyEditCommand;
            if ( cmd==null )
                return;

            // Don't care if only use color 0.
            if ( this.EmitterData.UI_colorType==Constances.ColorCommonType.ConstColor ||
                 this.EmitterData.UI_colorType==Constances.ColorCommonType.Modulate )
            {
                return;
            }

            // Compose data source paths.
            string tmpPath = this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimEditData";

            NWCore.Utility.Path color0Path =
                new NWCore.Utility.Path( tmpPath + ".Color0.ColorType" );

            NWCore.Utility.Path color1Path =
                new NWCore.Utility.Path( tmpPath + ".Color1.ColorType" );

            // Is any of the colors using animation?
            NWCore.Utility.Path targetPath = null;
            if ( this.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Animation ||
                 this.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Random )
            {
                targetPath = color1Path;
            }
            else if ( this.EmitterData.AnimEditData.Color1.ColorType==Constances.ColorSettingType.Animation ||
                      this.EmitterData.AnimEditData.Color1.ColorType==Constances.ColorSettingType.Random )
            {
                targetPath = color0Path;
            }
            else
            {
                return;
            }

            // At least one of the colors is using animation, check if the other
            // is about to be set to anything other than constant color.
            foreach ( PropertyEditCommandArgs cmdArgs in cmd.CommandArguments )
            {
                if ( cmdArgs.DataSrcPath.Match(targetPath)==false )
                    continue;

                if ( (cmdArgs.ModifiedValue is Constances.ColorSettingType)==false )
                    continue;

                if ( cmdArgs.UserData is string )
                {
                    string strUserData = cmdArgs.UserData as string;
                    if ( strUserData=="IgnoreColorTypeChecking" )
                        continue;
                }

                Constances.ColorSettingType modifiedColorType =
                    (Constances.ColorSettingType)cmdArgs.ModifiedValue;
                if ( modifiedColorType!=Constances.ColorSettingType.Constant )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning, res.Strings.MSG_OPERATION_CANCELLED );
                    myArgs.Cancel = true;
                    return;
                }
            }

            #endif
        }


        /// <summary>
        /// Handle command executed on color type data sources.
        /// </summary>
        /// <param name="command">The command executed.</param>
        /// <param name="executionType">The command execution type.</param>
        /// <param name="targetDocument">The target document.</param>
        /*
        private void HandleColorTypeCommandExecuted( PropertyEditCommand cmd,
                                                     CommandExecutionTypes executionType,
                                                     IDocument targetDocument )
        {
            #if BUILD_FOR_CTR

            if ( cmd==null )
                return;

            // Don't care if only use color 0.
            if ( this.EmitterData.UI_colorType==Constances.ColorCommonType.ConstColor ||
                 this.EmitterData.UI_colorType==Constances.ColorCommonType.Modulate )
            {
                return;
            }

            // Compose data source paths.
            string tmpPath = this.DataScrPath + "." + this.RelativeDataScrPath + ".AnimEditData";

            NWCore.Utility.Path color0Path =
                new NWCore.Utility.Path( tmpPath + ".Color0.ColorType" );

            NWCore.Utility.Path color1Path =
                new NWCore.Utility.Path( tmpPath + ".Color1.ColorType" );

            // Look for the command arguments we care for.
            PropertyEditCommandArgs     cmdArgs       = null;
            NWCore.Utility.Path   targetPath    = null;
            Constances.ColorSettingType origColorType = Constances.ColorSettingType.Constant;
            foreach ( PropertyEditCommandArgs args in cmd.CommandArguments )
            {
                if ( (args.ModifiedValue is Constances.ColorSettingType)==false )
                    continue;

                Constances.ColorSettingType colorType =
                    (Constances.ColorSettingType)args.ModifiedValue;
                if ( colorType!=Constances.ColorSettingType.Animation &&
                     colorType!=Constances.ColorSettingType.Random )
                {
                    continue;
                }

                // Already handled when pasting the whole color property page.
                if ( args.UserData is string )
                {
                    string strUserData = args.UserData as string;
                    if ( strUserData=="IgnoreColorTypeChecking" )
                        continue;
                }

                if ( args.DataSrcPath.Match(color0Path)==true )
                {
                    cmdArgs       = args;
                    targetPath    = color1Path;
                    origColorType = this.EmitterData.AnimEditData.Color1.ColorType;
                    break;
                }
                else if ( args.DataSrcPath.Match(color1Path)==true )
                {
                    cmdArgs       = args;
                    targetPath    = color0Path;
                    origColorType = this.EmitterData.AnimEditData.Color0.ColorType;
                    break;
                }
            }

            // We don't care about this command, bail out.
            if ( cmdArgs==null || targetPath==null )
                return;

            if ( executionType==CommandExecutionTypes.Execute )
            {
                bool bFound = false;
                foreach ( PropertyEditCommandArgs args in cmd.CommandArguments )
                {
                    if ( args.DataSrcPath.Match(targetPath)==true )
                    {
                        bFound = true;
                        cmdArgs.UserData = args.OriginalValue;
                        break;
                    }
                }

                if ( bFound==false )
                    cmdArgs.UserData = origColorType;
            }

            PropertyEditCommandArgs arg = null;
            if ( executionType==CommandExecutionTypes.Execute ||
                 executionType==CommandExecutionTypes.Redo )
            {
                arg =
                    new PropertyEditCommandArgs( cmdArgs.ControlPath,
                                                 targetPath,
                                                 this,
                                                 cmdArgs.UserData,
                                                 Constances.ColorSettingType.Constant );
            }
            else if ( executionType==CommandExecutionTypes.Undo )
            {
                arg =
                    new PropertyEditCommandArgs( cmdArgs.ControlPath,
                                                 targetPath,
                                                 this,
                                                 Constances.ColorSettingType.Constant,
                                                 cmdArgs.UserData );
            }
            else
            {
                return;
            }

            PropertyEditCommand newCmd = new PropertyEditCommand( arg );

            newCmd.CanUndo = false;

            TheApp.CommandManager.ScheduleExec( newCmd );

            #endif
        }
        */

        #endregion

        #region EmitterData.UI_colorType

        /// <summary>
        /// Handle command executed on color type data sources.
        /// </summary>
        /// <param name="command">The command executed.</param>
        /// <param name="executionType">The command execution type.</param>
        /// <param name="targetDocument">The target document.</param>
        /*
        private void HandleCombinerColorTypeCommandExecuted( PropertyEditCommand cmd,
                                                             CommandExecutionTypes executionType,
                                                             IDocument targetDocument )
        {
            #if BUILD_FOR_CTR

            if ( cmd==null )
                return;

            // Compose data source paths.
            string tmpPath = this.DataScrPath + "." + this.RelativeDataScrPath;
            NWCore.Utility.Path targetPath =
                new NWCore.Utility.Path( tmpPath + ".UI_colorType" );

            // Look for the command arguments we care for.
            PropertyEditCommandArgs cmdArgs = null;
            foreach ( PropertyEditCommandArgs args in cmd.CommandArguments )
            {
                if ( (args.ModifiedValue is Constances.ColorCommonType)==false )
                    continue;

                if ( args.DataSrcPath.Match(targetPath)==false )
                    continue;

                cmdArgs = args;

                break;
            }

            // We don't care about this command, bail out.
            if ( cmdArgs==null )
                return;

            if ( executionType==CommandExecutionTypes.Execute )
            {

                if ( this.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Constant ||
                     this.EmitterData.AnimEditData.Color1.ColorType==Constances.ColorSettingType.Constant )
                {
                    return;
                }

                Constances.ColorCommonType colorType =
                    (Constances.ColorCommonType)cmdArgs.ModifiedValue;
                if ( colorType==Constances.ColorCommonType.ConstColor ||
                     colorType==Constances.ColorCommonType.Modulate )
                {
                    return;
                }

                // Save the original value for undo.
                cmdArgs.UserData = this.EmitterData.AnimEditData.Color1.ColorType;

                TheApp.OutputLogMsg( NWCore.LogLevels.Warning, res.Strings.WARNING_COLOR1_TYPE_FORCED_CONSTANT );
            }

            if ( (cmdArgs.UserData is Constances.ColorSettingType)==false )
                return;

            NWCore.Utility.Path modifyPath =
                new NWCore.Utility.Path( tmpPath + ".AnimEditData.Color1.ColorType" );

            PropertyEditCommandArgs arg = null;
            if ( executionType==CommandExecutionTypes.Execute ||
                 executionType==CommandExecutionTypes.Redo )
            {
                arg =
                    new PropertyEditCommandArgs( cmdArgs.ControlPath,
                                                 modifyPath,
                                                 this,
                                                 cmdArgs.UserData,
                                                 Constances.ColorSettingType.Constant );
            }
            else if ( executionType==CommandExecutionTypes.Undo )
            {
                arg =
                    new PropertyEditCommandArgs( cmdArgs.ControlPath,
                                                 modifyPath,
                                                 this,
                                                 Constances.ColorSettingType.Constant,
                                                 cmdArgs.UserData );
            }
            else
            {
                return;
            }

            PropertyEditCommand newCmd = new PropertyEditCommand( arg );

            newCmd.CanUndo = false;

            TheApp.CommandManager.ScheduleExec( newCmd );

            #endif
        }
        */

        #endregion

        #endregion

        #region Primitive file monitor

        /// <summary>
        /// Register file system watcher for the primitive file.
        /// </summary>
        private void RegisterPrimitiveFileMonitor()
        {
            /*
            if ( m_primitiveFileWatcher==null )
                return;
            */

            UnregisterPrimitiveFileMonitor();

            string primitiveFilePath = this.EmitterData.BillboardPrimitiveFileSource;

            if ( string.IsNullOrEmpty(primitiveFilePath)==true ||
                 Path.IsPathRooted(primitiveFilePath)==false )
            {
                return;
            }

            string dirPath = System.IO.Path.GetDirectoryName( primitiveFilePath );
            if ( System.IO.Path.IsPathRooted(dirPath)==false )
                return;

            /*
            m_primitiveFileWatcher.DelayToWait = 1000; // 1000 ms
            m_primitiveFileWatcher.Path                = dirPath;
            m_primitiveFileWatcher.EnableRaisingEvents = true;

            m_primitiveFileWatcher.Changed += new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Deleted += new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Renamed += new RenamedEventHandler( OnPrimitiveFileRenamed );
            m_primitiveFileWatcher.Created += new FileSystemEventHandler( OnPrimitiveFileChanged );
            */
        }


        /// <summary>
        /// Unregister file system watcher for the primitive file.
        /// </summary>
        private void UnregisterPrimitiveFileMonitor()
        {
            /*
            if ( m_primitiveFileWatcher==null )
                return;

            m_primitiveFileWatcher.EnableRaisingEvents = false;

            m_primitiveFileWatcher.Changed -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Deleted -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Renamed -= new RenamedEventHandler( OnPrimitiveFileRenamed );
            m_primitiveFileWatcher.Created -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            */
        }


        /// <summary>
        /// Reload primitive file.
        /// </summary>
        private void ReloadPrimitive()
        {
            // Source data path to EmitterData.BillboardPrimitiveFileSource
            string srcPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".BillboardPrimitiveFileSource";

            /*
            PropertyEditCommandArgs args =
                new PropertyEditCommandArgs( new NWCore.Utility.Path( "" ),
                                             new NWCore.Utility.Path( srcPath ),
                                             this,
                                             this.EmitterData.BillboardPrimitiveFileSource,
                                             this.EmitterData.BillboardPrimitiveFileSource );
            PropertyEditCommand cmd = new PropertyEditCommand( args );
            cmd.CanUndo = false;

            TheApp.CommandManager.ScheduleExec( cmd );
            */
        }


        /// <summary>
        /// Handle event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnPrimitiveFileChanged( object sender,
                                             System.IO.FileSystemEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.EmitterData.BillboardPrimitiveFileSource;
            if ( e.FullPath!=primitiveFilePath )
                return;

            //MainFrame.SynchronizationContext.Post( delegate { this.ReloadPrimitive(); } );
        }


        /// <summary>
        /// Handle Rename event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnPrimitiveFileRenamed( object sender,
                                             System.IO.RenamedEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.EmitterData.BillboardPrimitiveFileSource;
            if ( e.FullPath!=primitiveFilePath && e.OldFullPath!=primitiveFilePath )
                return;

            //MainFrame.SynchronizationContext.Post( delegate { this.ReloadPrimitive(); } );
        }

        #endregion
    }
}
