﻿// ========================================================================
// <copyright file="ChildDocument.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
//using App.Command;
using App.IO;
//using App.PropertyEdit;
//using App.Controls;
using App.Utility;
using NWCore.DataModel;

namespace App.Data
{
    /// <summary>
    /// パーティクル・セット用ドキュメント
    /// </summary>
    public sealed class ChildDocument : BaseProjectPanelDocument
    {
        #region Member Variables

        private EmitterDocument   m_emitterDoc           = null;    // parent/owner
        private bool              m_bUpdatingTexture     = false;
        //private CustomFileSystemWatcher m_primitiveFileWatcher = new CustomFileSystemWatcher();

        #endregion

        #region Properties

        /// <summary>
        /// アイコンのイメージキー
        /// </summary>
        public override string ImageKey
        {
            get
            {
                string baseName = this.ObjectID.ToString();
                DialogResult result;
                if (this.EmitterDocument.EmitterData.ChildFlag.Enable)
                {
                    if (this.HasUnreachableAsset)
                    {
                        return baseName + Document.UnreachableAssetKey;
                    }
                    else if (this.HasTypeMismatchTexture && Config.ShouldShowMessageBox("LinearEditTextureWarning", out result))
                    {
                        return baseName + Document.TextureTypeMismatchKey;
                    }
                    else
                    {
                        return baseName + Document.ActiveIconKey;
                    }
                }
                else
                {
                    return baseName + Document.NonactiveIconKey;
                }
            }
        }

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                return res.Strings.CHILD_DEFAULT_NAME;
            }
            set
            {
            }
        }

        /// <summary>
        /// オーナードキュメント。
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.EmitterDocument; }
        }

        /// <summary>
        /// Emitter that this child belongs to
        /// </summary>
        public EmitterDocument EmitterDocument
        {
            get { return this.m_emitterDoc; }
        }

        /// <summary>
        /// EmitterData
        /// </summary>
        public EmitterData EditData
        {
            get { return (this.EmitterDocument != null ? this.EmitterDocument.EmitterData : null); }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "ChildData";
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return this.EmitterDocument.DataScrPath
                    + "."
                    + this.EmitterDocument.RelativeDataScrPath;
            }
        }

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ChildDocument(EmitterDocument emitterDoc)
            : base(res.Strings.CHILD_DEFAULT_NAME)
        {
            this.Name = res.Strings.CHILD_DEFAULT_NAME;
            this.m_emitterDoc = emitterDoc;

            TheApp.ConfigChanged += OnAppConfigChanged;
        }

        #endregion

        #region オブジェクト

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.Child; }
        }

        #endregion

        #region Utility Functions

        // Method body factorized with EmitterDocument.CheckTexturePathValidity static method
        // Please avoid duplicated code
        private void CheckTexturePathValidity()
        {
            string texturePath = EmitterDocument.EmitterData.ChildTexPatData.UI_texPatFileName;
            EmitterSetDocument esetDoc = EmitterDocument.OwnerDocument as EmitterSetDocument;

            EmitterDocument.CheckTexturePathValidity( texturePath, esetDoc );
        }

        /// <summary>
        /// Check primitive location.
        /// If the primitive is not in the predefined primitive search directories,
        /// show the warning.
        /// </summary>
        private void CheckPrimitivePathValidity()
        {
            string primitivePath = EmitterDocument.EmitterData.ChildData.BillboardPrimitiveFileSource;
            EmitterSetDocument esetDoc = EmitterDocument.OwnerDocument as EmitterSetDocument;

            EmitterDocument.CheckPrimitivePathValidity( primitivePath, esetDoc );
        }

        /// <summary>
        /// Checks if the assets are reachable, and report it otherwise
        /// </summary>
        public override void CheckAssetsReachability(bool bUpdateProjectTree)
        {
            int unreachableAssetCount = 0;
            int typeMismatchTextureCount = 0;

            if (this.EmitterDocument.EmitterData.ChildFlag.Enable)
            {
                string primitivePath = this.EmitterDocument.EmitterData.ChildData.BillboardPrimitiveFileSource;
                if ( DocumentConstants.NoSuchAssetPath( primitivePath ) )
                {
                    ++unreachableAssetCount;
                }

                TextureResourceHelper.CheckTextureReachability(EmitterDocument.EmitterData.ChildTexPatData.UI_texPatFileName,
                    ref unreachableAssetCount, ref typeMismatchTextureCount);
            }

            UpdateUnreachableAssetCount(unreachableAssetCount, bUpdateProjectTree);
            UpdateTypeMismatchTextureCount(typeMismatchTextureCount, bUpdateProjectTree);

            (this.OwnerDocument as EmitterDocument).CheckAssetsReachability(bUpdateProjectTree);
        }

        #endregion

        #region Event handlers


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            RegisterPrimitiveFileMonitor();

            TheApp.OnGammaCorrectionModeChanged += this.OnGammaCorrectionModeChanged;
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            TheApp.OnGammaCorrectionModeChanged -= this.OnGammaCorrectionModeChanged;

            UnregisterPrimitiveFileMonitor();

            base.NotifyRemovedFromParent();
        }


        /// <summary>
        /// Handle ConfigChanged event from the application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAppConfigChanged( object sender,
                                         EventArgs e )
        {
            RegisterPrimitiveFileMonitor();
        }


        /// <summary>
        /// Gamma correction mode changed event
        /// </summary>
        private void OnGammaCorrectionModeChanged()
        {
            ReloadTexture(null);
        }

        /// <summary>
        ///  Notify the texture resource has been modified
        /// </summary>
        public void OnTextureResourceModified()
        {
            // Get the data source path of the texture resource.
            string docPath = this.DataScrPath + "." +
                                this.RelativeDataScrPath +
                                ".TextureRes";
            TextureResourceHelper.IssueTextureResourceModifiedCommand(docPath, this,
                EmitterDocument.EmitterData.ChildData.TextureRes);
        }

        /// <summary>
        /// Reload texture.
        /// </summary>
        public bool ReloadTexture(string path)
        {
            bool result = false;
            m_bUpdatingTexture = true;

            if ( path==null ||
                 path==EmitterDocument.EmitterData.ChildTexPatData.UI_texPatFileName )
            {
                string fullPath = String.Empty;
                result = DocumentIO.ReadTextureFile( this.FileLocation,
                                                     EmitterDocument.EmitterData.ChildTexPatData.UI_texPatFileName,
                                                     EmitterDocument.EmitterData.ChildData.TextureRes,
                                                     ref fullPath );

                OnTextureResourceModified();
                CheckAssetsReachability(true);
            }
            m_bUpdatingTexture = false;

            return result;
        }


        /// <summary>
        /// Handle texture path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnPrimitivePathChanged( ICommand command,
                                            CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null )
                return;
            */

            RegisterPrimitiveFileMonitor();

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckPrimitivePathValidity();
            }

            CheckAssetsReachability(true);
        }

        /// <summary>
        /// Handle texture path being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnChildTexturePathChanged( ICommand command,
                                               CommandExecutionTypes executionType )
        {
            /*
            PropertyEditCommand propCommand = command as PropertyEditCommand;
            if ( propCommand==null)
                return;
            */

            if ( m_bUpdatingTexture==true )
                return;

            m_bUpdatingTexture = true;

            EmitterData data = this.EmitterDocument.EmitterData;

            // Get the data source path of the texture resource.
            string docPath = EmitterDocument.DataScrPath + "." +
                EmitterDocument.RelativeDataScrPath +
                ".ChildTexPatData.UI_texPatFileName";

            string texPatFileName = data.ChildTexPatData.UI_texPatFileName;
            //TextureResourceHelper.ReadTextureAndIssueCommand(docPath, propCommand.ControlPath, this, data.ChildData.TextureRes, ref texPatFileName);
            data.ChildTexPatData.UI_texPatFileName = texPatFileName;

            if ( executionType==CommandExecutionTypes.Execute )
            {
                CheckTexturePathValidity();
            }
            OnTextureResourceModified();

            m_bUpdatingTexture = false;

            CheckAssetsReachability(true);
        }

        #endregion

        #region Primitive file monitor

        /// <summary>
        /// Register file system watcher for the primitive file.
        /// </summary>
        private void RegisterPrimitiveFileMonitor()
        {
            /*
            if ( m_primitiveFileWatcher==null )
                return;
            */

            UnregisterPrimitiveFileMonitor();

            string primitiveFilePath = this.EditData.ChildData.BillboardPrimitiveFileSource;

            if ( string.IsNullOrEmpty(primitiveFilePath)==true ||
                 Path.IsPathRooted(primitiveFilePath)==false )
            {
                return;
            }

            string dirPath = System.IO.Path.GetDirectoryName( primitiveFilePath );
            if ( System.IO.Path.IsPathRooted(dirPath)==false )
                return;

            /*
            m_primitiveFileWatcher.DelayToWait = 1000; // 1000 ms
            m_primitiveFileWatcher.Path                = dirPath;
            m_primitiveFileWatcher.EnableRaisingEvents = true;

            m_primitiveFileWatcher.Changed += new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Deleted += new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Renamed += new RenamedEventHandler( OnPrimitiveFileRenamed );
            m_primitiveFileWatcher.Created += new FileSystemEventHandler( OnPrimitiveFileChanged );
            */
        }


        /// <summary>
        /// Unregister file system watcher for the primitive file.
        /// </summary>
        private void UnregisterPrimitiveFileMonitor()
        {
            /*
            if ( m_primitiveFileWatcher==null )
                return;

            m_primitiveFileWatcher.EnableRaisingEvents = false;

            m_primitiveFileWatcher.Changed -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Deleted -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            m_primitiveFileWatcher.Renamed -= new RenamedEventHandler( OnPrimitiveFileRenamed );
            m_primitiveFileWatcher.Created -= new FileSystemEventHandler( OnPrimitiveFileChanged );
            */
        }


        /// <summary>
        /// Reload primitive file.
        /// </summary>
        private void ReloadPrimitive()
        {
            /*
            // Source data path to EmitterData.BillboardPrimitiveFileSource
            string srcPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath +
                             ".BillboardPrimitiveFileSource";

            PropertyEditCommandArgs args =
                new PropertyEditCommandArgs( new NWCore.Utility.Path( "" ),
                                             new NWCore.Utility.Path( srcPath ),
                                             this,
                                             this.EditData.ChildData.BillboardPrimitiveFileSource,
                                             this.EditData.ChildData.BillboardPrimitiveFileSource );
            PropertyEditCommand cmd = new PropertyEditCommand( args );
            cmd.CanUndo = false;

            TheApp.CommandManager.ScheduleExec( cmd );
            */
        }


        /// <summary>
        /// Handle event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnPrimitiveFileChanged( object sender,
                                             System.IO.FileSystemEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.EditData.ChildData.BillboardPrimitiveFileSource;
            if ( e.FullPath!=primitiveFilePath )
                return;

            //MainFrame.SynchronizationContext.Post( delegate { this.ReloadPrimitive(); } );
        }


        /// <summary>
        /// Handle Rename event from the file system watcher.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnPrimitiveFileRenamed( object sender,
                                             System.IO.RenamedEventArgs e )
        {
            if ( Config.Data.Option.AutoPrimitiveReload==false )
                return;

            string primitiveFilePath = this.EditData.ChildData.BillboardPrimitiveFileSource;
            if ( e.FullPath!=primitiveFilePath && e.OldFullPath!=primitiveFilePath )
                return;

            //MainFrame.SynchronizationContext.Post( delegate { this.ReloadPrimitive(); } );
        }

        #endregion
    }
}
