﻿// ========================================================================
// <copyright file="UserShaderUIDefinition.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Globalization;
using System.Diagnostics;

namespace App.Data
{
    /// <summary>
    /// Class for storing/loading user shader UI definition data.
    /// </summary>
    public class UserShaderUIDefinition : IXmlSerializable
    {
        #region Enumerator for the user shader UI control types

        /// <summary>
        /// Enumerator for the user shader UI control types.
        /// </summary>
        public enum UserShaderUIControlTypes
        {
            Unknow   = -1,

            CheckBox = 0,
            RadioBtnGroup,
            RadioButton,
            Slider
        }

        #endregion

        #region Class for UI group definition for user shader parameters

        /// <summary>
        /// Class for UI group definition for user shader parameters.
        /// </summary>
        public class UIGroupDefinition
        {
            #region Constructor

            /// <summary>
            /// Default constructor.
            /// </summary>
            public UIGroupDefinition()
            {
            }

            #endregion

            #region Initialize <= the kind of controls be in the group is decided here

            /// <summary>
            /// Initialize the data.
            /// </summary>
            /// <param name="groupType">The group type to initialize to.</param>
            /// <returns>True on success.</returns>
            public bool Init( UserShaderUIControlTypes groupType )
            {
                m_groupType = groupType;

                this.Visible = true;

                switch ( groupType )
                {
                    case UserShaderUIControlTypes.CheckBox :
                        {
                            this.Label = res.Strings.UI_CAFE_EXT_USER_SHADER_FLAGS;

                            m_controls = new CheckBoxUserUIDef[NumCheckBoxDef];

                            // Initialize the control definitions.
                            for ( int i=0;i<m_controls.Length;++i )
                            {
                                BaseUserUIControlDef item = new CheckBoxUserUIDef();
                                if ( item==null )
                                    continue;

                                m_controls[i] = item;

                                // Setup the default label.
                                item.Index = i;

                                if ( item.Init()==false )
                                    return false;
                            }
                        }
                        break;

                    case UserShaderUIControlTypes.RadioBtnGroup :
                        {
                            this.Label = res.Strings.UI_CAFE_EXT_USER_SHADER_FLAGS;

                            m_childGroups = new UIGroupDefinition[NumRadioBtnGroups];

                            // Initialize the child group definitions.
                            for ( int i=0;i<m_childGroups.Length;++i )
                            {
                                UIGroupDefinition item = new UIGroupDefinition();
                                if ( item==null )
                                    continue;

                                item.Index = i;

                                if ( item.Init( UserShaderUIControlTypes.RadioButton )==false )
                                    return false;

                                m_childGroups[i] = item;
                            }
                        }
                        break;

                    case UserShaderUIControlTypes.RadioButton :
                        {
                            this.Label = res.Strings.UI_CAFE_EXT_USER_SHADER_FLAGS;

                            m_controls = new RadioButtonUserUIDef[NumRadioButtonDef];

                            // Initialize the control definitions.
                            for ( int i=0;i<m_controls.Length;++i )
                            {
                                BaseUserUIControlDef item = new RadioButtonUserUIDef();
                                if ( item==null )
                                    continue;

                                m_controls[i] = item;

                                // Setup the default label.
                                item.Index = i;

                                if ( item.Init()==false )
                                    return false;
                            }
                        }
                        break;

                    case UserShaderUIControlTypes.Slider :
                        {
                            this.Label = res.Strings.UI_CAFE_EXT_PARAM_USER_SHADER_PARAMS;

                            m_controls = new FloatSliderUserUIDef[NumSliderDef];

                            // Initialize the control definitions.
                            for ( int i=0;i<m_controls.Length;++i )
                            {
                                BaseUserUIControlDef item = new FloatSliderUserUIDef();
                                if ( item==null )
                                    continue;

                                m_controls[i] = item;

                                // Setup the default label.
                                item.Index = i;

                                if ( item.Init()==false )
                                    return false;
                            }
                        }
                        break;
                }

                return true;
            }

            #endregion

            #region Properties

            /// <summary>Get the index of the UI group.</summary>
            public int Index { get; private set; }

            /// <summary>Get or set the flag indicating whether this group is visible.</summary>
            public bool Visible { get; set; }

            /// <summary>Get or set the label of the group.</summary>
            public string Label { get; set; }

            /// <summary>Get the enumerator for the child group definitions.</summary>
            public IEnumerable<UIGroupDefinition> ChildGroups
            {
                get
                {
                    if ( m_childGroups==null )
                    {
                        yield return null;
                    }
                    else
                    {
                        foreach ( UIGroupDefinition child in m_childGroups )
                        {
                            yield return child;
                        }
                    }
                }
            }

            /// <summary>Get the count of the control definitions.</summary>
            public int ControlDefCount
            {
                get
                {
                    if ( m_controls==null )
                        return 0;
                    return m_controls.Length;
                }
            }

            /// <summary>Get the enumerator for the control definitions in the group.</summary>
            public IEnumerable<BaseUserUIControlDef> ControlDefs
            {
                get
                {
                    if ( m_controls==null )
                    {
                        yield return null;
                    }
                    else
                    {
                        foreach ( BaseUserUIControlDef def in m_controls )
                        {
                            yield return def;
                        }
                    }
                }
            }

            #endregion

            #region Load group definition

            /// <summary>
            /// Load the definition from the given XML element.
            /// </summary>
            /// <param name="xml">The XML element containing the definition.</param>
            /// <returns>True on success.</returns>
            public bool LoadDefinition( XmlElement xml )
            {
                // Load the label for the group.
                /*
                if ( xml.HasAttribute(App.Controls.UIManager.XMLUIPrimaryLabelNodeName)==true )
                {
                    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
                }
                else if ( xml.HasAttribute(App.Controls.UIManager.XMLUISecondaryLabelNodeName)==true )
                {
                    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
                }
                */

                // Load the visibility flag for the group.
                if ( xml.HasAttribute("Visible")==true )
                {
                    bool bVisible = true;
                    if ( bool.TryParse( xml.GetAttribute("Visible"), out bVisible )==true )
                        this.Visible = bVisible;
                }

                // Load the index for the group.
                int iIndex;
                if ( int.TryParse(xml.GetAttribute("Index"),
                                  out iIndex)==false )
                {
                    return false;
                }

                this.Index = iIndex;

                // Load the template definition.
                XmlElement  template = null;
                XmlNodeList nodes    = xml.GetElementsByTagName( "Template" );
                if ( nodes!=null && nodes.Count>0 )
                    template = nodes[0] as XmlElement;

                // Load the control definitions.
                foreach ( XmlNode node in xml.ChildNodes )
                {
                    XmlElement child = node as XmlElement;
                    if ( child==null )
                        continue;

                    if ( child.Name=="Group" )
                    {
                        UserShaderUIControlTypes grpType;
                        if ( Enum.TryParse(child.GetAttribute("Type"),
                                           out grpType)==false )
                        {
                            continue;
                        }

                        if ( int.TryParse(child.GetAttribute("Index"),
                                          out iIndex )==false )
                        {
                            continue;
                        }

                        if ( iIndex<0 || iIndex>=m_childGroups.Length )
                            continue;

                        UIGroupDefinition group = m_childGroups[iIndex];
                        if ( group==null )
                            continue;

                        group.Init( grpType );
                        group.HideAll();
                        group.LoadDefinition( child );
                    }
                    else if ( child.Name==m_groupType.ToString() )
                    {
                        if ( int.TryParse(child.GetAttribute("Index"), out iIndex)==false )
                            continue;

                        if ( iIndex<0 || iIndex>=m_controls.Length )
                            continue;

                        // The control definitions should all be created and initialized.
                        // Just return false if they were not.
                        // This means someone tries to load definitions without initialize
                        // first.
                        BaseUserUIControlDef def = m_controls[iIndex];
                        if ( def==null )
                            return false;

                        // Load the control definitions.
                        if ( def.LoadDefinition(child, template)==false )
                            return false;
                    }
                }

                return true;
            }

            #endregion

            #region Utility methods

            /// <summary>
            /// Set the visibility of all the definitions to false.
            /// </summary>
            public void HideAll()
            {
                this.Visible = false;

                if ( m_childGroups!=null )
                {
                    foreach ( UIGroupDefinition def in m_childGroups )
                    {
                        if ( def!=null )
                            def.HideAll();
                    }
                }

                if ( m_controls!=null )
                {
                    foreach ( BaseUserUIControlDef def in m_controls )
                    {
                        if ( def!=null )
                            def.Visible = false;
                    }
                }
            }

            #endregion

            #region Constants <= the number of controls in the group is defined here

            private const int NumCheckBoxDef    = 32;
            private const int NumRadioBtnGroups = 4;
            private const int NumRadioButtonDef = 8;
            private const int NumSliderDef      = 32;

            #endregion

            #region Member variables

            private UserShaderUIControlTypes     m_groupType   = UserShaderUIControlTypes.Unknow;
            private UIGroupDefinition[]          m_childGroups = null;
            private BaseUserUIControlDef[] m_controls    = null;

            #endregion
        }

        #endregion

        #region Constructors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserShaderUIDefinition()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool Init( int iIndex )
        {
            this.CallbackId   = iIndex;
            this.Label   = string.Empty;
            this.Visible = true;

            m_vertexShaderPaths.Clear();
            m_fragmentShaderPaths.Clear();

            if ( m_cbGroup==null || m_cbGroup.Init(UserShaderUIControlTypes.CheckBox)==false )
                return false;

            if ( m_rbGroup==null || m_rbGroup.Init(UserShaderUIControlTypes.RadioBtnGroup)==false )
                return false;

            if ( m_sldGroup==null || m_sldGroup.Init(UserShaderUIControlTypes.Slider)==false )
                return false;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get the index of the UI definition.</summary>
        public int CallbackId { get; set; }

        /// <summary>Get the label of the UI definition.</summary>
        public string Label
        {
            get
            {
                if ( string.IsNullOrEmpty(m_label)==true )
                    return string.Format( res.Strings.UI_CAFE_EXT_PARAM_SETTING, this.CallbackId+1 );
                return m_label;
            }
            private set { m_label = value; }
        }

        /// <summary>Get or set the flag indicating whether this control is visible.</summary>
        public bool Visible { get; set; }


        /// <summary>Enumerate vertex shader paths.</summary>
        public List<string> VertexShaderPaths
        {
            get { return m_vertexShaderPaths; }
        }


        /// <summary>Enumerate fragment shader paths.</summary>
        public List<string> FragmentShaderPaths
        {
            get { return m_fragmentShaderPaths; }
        }


        /// <summary>
        /// Get the check box group definition.
        /// </summary>
        public UIGroupDefinition CheckBoxGroupDefinition
        {
            get { return m_cbGroup; }
        }


        /// <summary>
        /// Get the radio button groups definitions.
        /// </summary>
        public UIGroupDefinition RadioBtnGroupDefinition
        {
            get { return m_rbGroup; }
        }


        /// <summary>
        /// Get the slider group definition.
        /// </summary>
        public UIGroupDefinition SliderGroupDefinition
        {
            get { return m_sldGroup; }
        }

        #endregion

        #region UI group definitions

        /// <summary>
        /// Get the UI group definition with the given group type.
        /// </summary>
        /// <param name="groupType">The group type.</param>
        /// <returns>The UI group definition.</returns>
        public UIGroupDefinition GetGroupDef( UserShaderUIControlTypes groupType )
        {
            switch ( groupType )
            {
                case UserShaderUIControlTypes.CheckBox :
                    return m_cbGroup;

                case UserShaderUIControlTypes.RadioBtnGroup :
                    return m_rbGroup;

                case UserShaderUIControlTypes.Slider :
                    return m_sldGroup;
            }

            return null;
        }


        /// <summary>
        /// Set the visibility of all the definitions to false.
        /// </summary>
        public void HideAll()
        {
            if ( m_cbGroup!=null )
                m_cbGroup.HideAll();

            if ( m_rbGroup!=null )
                m_rbGroup.HideAll();

            if ( m_sldGroup!=null )
                m_sldGroup.HideAll();
        }

        #endregion

        #region Implemented for IXmlSerializable

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml( XmlReader reader )
        {
            // First reset the definitions.
            this.Init( 0 );
            this.HideAll();

            if ( reader.MoveToContent()==XmlNodeType.Element &&
                 reader.LocalName=="UserShaderUIDefinition" )
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml( reader.ReadOuterXml() );
                SetFromXMLDocument( doc.DocumentElement );
            }
        }

        /// <summary>
        /// Set this according to the XML document
        /// </summary>
        /// <param name="root">Root of the XML document</param>
        private void SetFromXMLDocument( XmlElement root )
        {
            Debug.Assert( root!=null );

            // Load the visibility flag of the definition.
            {
                bool bVisible;
                if ( bool.TryParse(root.GetAttribute("Visible"),
                                   out bVisible)==false )
                {
                    bVisible = true;
                }

                // Set the visibility flag of the definition.
                this.Visible = bVisible;
            }

            // Load the label for the definition.
            /*
            if ( root.HasAttribute(App.Controls.UIManager.XMLUIPrimaryLabelNodeName)==true )
            {
                this.Label = root.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
            }
            else if ( root.HasAttribute(App.Controls.UIManager.XMLUISecondaryLabelNodeName)==true )
            {
                this.Label = root.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
            }
            else
            */
            {
                this.Label = string.Empty;
            }

            foreach ( XmlNode node in root.ChildNodes )
            {
                XmlElement child = node as XmlElement;
                if ( child==null )
                    continue;

                if ( child.Name=="VertexShaderPaths" )
                    ReadShaderPaths( child, ref m_vertexShaderPaths );
                else if ( child.Name=="FragmentShaderPaths" )
                    ReadShaderPaths( child, ref m_fragmentShaderPaths );
                else if ( child.Name!="Group" )
                    continue;

                UserShaderUIControlTypes grpType;
                if ( Enum.TryParse( child.GetAttribute( "Type" ),
                                    out grpType )==false )
                {
                    continue;
                }

                UIGroupDefinition group = GetGroupDef( grpType );
                if ( group==null )
                    continue;

                group.Init( grpType );
                group.HideAll();
                group.LoadDefinition( child );
            }
        }


        /// <summary>
        /// Read shader file paths.
        /// </summary>
        /// <param name="node">The XML element containing the shader paths.</param>
        /// <param name="shaderPaths">The list to store the paths.</param>
        private void ReadShaderPaths( XmlElement node,
                                      ref List<string> shaderPaths )
        {
            // Clear the list first.
            shaderPaths.Clear();

            foreach ( XmlNode child in node.ChildNodes )
            {
                if ( child.Name!="ShaderPath" )
                    continue;

                string filePath =
                    RequireDocument.GetFullPathForRequireDocument( child.InnerText.Trim() );
                if ( string.IsNullOrEmpty(filePath)==true )
                    continue;

                filePath = App.Utility.ShellUtility.ToAbsolutePath( filePath,
                                                                    System.IO.Directory.GetCurrentDirectory() );

                try
                {
                    if ( System.IO.File.Exists(filePath)==false )
                    {
                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             res.Strings.WARNING_CUSTOM_SHADER_FILE_NOT_FOUND,
                                             UserShaderUIDefDeserializationBlock.CurrentFilePath,
                                             filePath );
                        continue;
                    }
                }
                catch
                {
                    // The file path is invalid.
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.WARNING_CUSTOM_SHADER_FILE_NOT_FOUND,
                                         UserShaderUIDefDeserializationBlock.CurrentFilePath,
                                         filePath );
                    continue;
                }

                shaderPaths.Add( filePath );
            }
        }


        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml( XmlWriter writer )
        {
            // Not implemented.
        }


        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        #endregion

        #region Member variables

        private const int         m_iNumRBGroups = 4;

        private string            m_label        = string.Empty;

        private UIGroupDefinition m_cbGroup      = new UIGroupDefinition();
        private UIGroupDefinition m_rbGroup      = new UIGroupDefinition();
        private UIGroupDefinition m_sldGroup     = new UIGroupDefinition();

        private List<string>      m_vertexShaderPaths   = new List<string>();
        private List<string>      m_fragmentShaderPaths = new List<string>();

        #endregion
    }

    #region Class for holding information while deserializing .usd files

    /// <summary>
    /// Class for holding information needed while deserializing user shader UI definition files.
    /// </summary>
    public class UserShaderUIDefDeserializationBlock : IDisposable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="fileName"></param>
        public UserShaderUIDefDeserializationBlock( string fileName )
        {
            UserShaderUIDefDeserializationBlock.CurrentFilePath = fileName;
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            UserShaderUIDefDeserializationBlock.CurrentFilePath = string.Empty;
        }


        /// <summary>Get the path of the user shader definition file that is currently being loaded.</summary>
        public static string CurrentFilePath { get; private set; }
    }

    #endregion
}
