﻿// ========================================================================
// <copyright file="CustomShaderUIDefList.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Globalization;

using App.Utility;

namespace App.Data
{
    /// <summary>
    /// Class for storing/deserializing a link to a .usd(user shader definition) file.
    /// </summary>
    public class CustomShaderUIDefLinkData
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public CustomShaderUIDefLinkData()
        {
            this.CallbackId          = -1;
            this.FilePath       = string.Empty;
            this.DefinitionData = null;
        }


        /// <summary>Get or set the index of the linked user shader definition file.</summary>
        public int CallbackId { get; set; }


        /// <summary>Get or set the path of the linked user shader definition file.</summary>
        public string FilePath { get; set; }


        /// <summary>Get or set the user shader definition data.</summary>
        public UserShaderUIDefinition DefinitionData { get; set; }


        /// <summary>
        /// Read the contents from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element to read from.</param>
        /// <returns>The loaded data on success or null on failure.</returns>
        public static CustomShaderUIDefLinkData ReadXml( XmlElement xml )
        {
            if ( xml==null )
                return null;

            if ( xml.Name!="CustomShaderUIDefLinkData" )
                return null;

            // Parse the index of the definition.
            int callbackId;
            if ( int.TryParse(xml.GetAttribute("CallbackId"),
                              NumberStyles.Integer,
                              CultureInfo.InvariantCulture,
                              out callbackId)==false )
            {
                return null;
            }

            if ( callbackId<=0 || callbackId>TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.ERROR_USERSHADER_DEF_LIST_XML_INDEX_INVALID,
                                     callbackId );
                return null;
            }

            // Parse the file path of the user shader definition.
            string filePath =
                RequireDocument.GetFullPathForRequireDocument( xml.InnerText.Trim() );
            if ( string.IsNullOrEmpty(filePath)==true )
                return null;

            filePath = ShellUtility.ToAbsolutePath( filePath,
                                                    Directory.GetCurrentDirectory() );

            try
            {
                if ( File.Exists(filePath)==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_USERSHADER_DEF_LIST_XML_DEF_FILE_NOT_FOUND,
                                         Path.GetFileName(filePath) );
                    return null;
                }
            }
            catch
            {
                // The file path might be invalid, bail out.
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.ERROR_USERSHADER_DEF_LIST_XML_DEF_FILE_PATH_INVALID,
                                     Path.GetFileName(filePath) );
                return null;
            }

            // Load the user shader definition file.
            UserShaderUIDefinition definition = null;
            try
            {
                // Create the serializer.
                XmlSerializer serializer =
                    new XmlSerializer( typeof(UserShaderUIDefinition) );

                // Load the file.
                using ( FileStream stream = new FileStream(filePath, FileMode.Open) )
                {
                    using ( new UserShaderUIDefDeserializationBlock(filePath) )
                    {
                        definition =
                            serializer.Deserialize( stream ) as UserShaderUIDefinition;
                    }
                }
            }
            catch ( IOException ex )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.ERROR_CUSTOM_SHADER_FAILED_LOADING,
                                     Path.GetFileName(filePath),
                                     ex.Message );
                return null;
            }
            catch ( Exception ex )
            {
                string message = string.Empty;
                if ( ex==null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_USERSHADER_UI_DEF_XML_PARSING_FAILED,
                                         Path.GetFileName(filePath),
                                         string.Empty );
                }
                else if ( ex.InnerException!=null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_USERSHADER_UI_DEF_XML_PARSING_FAILED,
                                         Path.GetFileName(filePath),
                                         ex.InnerException.Message );
                }
                else if ( ex!=null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_USERSHADER_UI_DEF_XML_PARSING_FAILED,
                                         Path.GetFileName(filePath),
                                         ex.Message );
                }
                return null;
            }

            if ( definition==null )
                return null;

            definition.CallbackId = callbackId - 1;

            // Create the instance.
            CustomShaderUIDefLinkData data = new CustomShaderUIDefLinkData();

            data.CallbackId     = callbackId - 1;
            data.FilePath       = filePath;
            data.DefinitionData = definition;

            return data;
        }
    }


    /// <summary>
    /// Class for storing/deserializing the .usd(user shader definition) file list.
    /// </summary>
    public class CustomShaderUIDefList : IXmlSerializable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public CustomShaderUIDefList()
        {
            for ( int i=0;i<TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS;++i )
                m_definitionList.Add( null );
        }


        /// <summary>
        /// Initialize default user shader definitions.
        /// </summary>
        public void InitDefaultData()
        {
            for ( int i=0;i<TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS;++i )
            {
                UserShaderUIDefinition definition = new UserShaderUIDefinition();
                definition.Init( i );

                m_definitionList[i] = new CustomShaderUIDefLinkData();

                m_definitionList[i].CallbackId          = i;
                m_definitionList[i].DefinitionData = definition;
            }
        }


        /// <summary>Get or set the list of the user shader definitions link data.</summary>
        public List<CustomShaderUIDefLinkData> DefinitionList
        {
            get { return m_definitionList; }
        }


        /// <summary>Enumerate the general vertex shader paths.</summary>
        public List<string> GeneralVertexShaderPaths
        {
            get { return m_generalVtxShaderPaths; }
        }


        /// <summary>Enumerate the general fragment shader paths.</summary>
        public List<string> GeneralFragmentShaderPaths
        {
            get { return m_generalFrgShaderPaths; }
        }


        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml( XmlReader reader )
        {
            // Clear the definitions first.
            for ( int i=0;i<TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS;++i )
                m_definitionList[i] = null;

            if ( reader.MoveToContent()!=XmlNodeType.Element ||
                 reader.LocalName!="CustomShaderUIDefList" )
            {
                return;
            }

            XmlDocument doc = new XmlDocument();
            doc.LoadXml( reader.ReadOuterXml() );

            XmlElement root = doc.DocumentElement;
            if ( root==null )
                throw new XmlReadException();

            foreach ( XmlNode child in root.ChildNodes )
            {
                if ( child.Name.Length==GENERAL_USER_SHADER_PATHS_TAG_NAME.Length &&
                     child.Name==GENERAL_USER_SHADER_PATHS_TAG_NAME )
                {
                    ReadGeneralUserShaderPaths( child );
                    continue;
                }

                CustomShaderUIDefLinkData data =
                    CustomShaderUIDefLinkData.ReadXml( child as XmlElement );
                if ( data==null )
                    continue;

                if ( data.CallbackId<0 || data.CallbackId>=TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS )
                    continue;

                m_definitionList[data.CallbackId] = data;
            }

            // Create default definitions.
            for ( int i=0;i<TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS;++i )
            {
                if ( m_definitionList[i]!=null && m_definitionList[i].DefinitionData!=null )
                    continue;

                UserShaderUIDefinition definition = new UserShaderUIDefinition();
                definition.Init( i );

                if ( m_definitionList[i]==null )
                    m_definitionList[i] = new CustomShaderUIDefLinkData();

                m_definitionList[i].CallbackId          = i;
                m_definitionList[i].DefinitionData = definition;
            }
        }


        /// <summary>
        /// Read the general user shader paths.
        /// </summary>
        /// <param name="node">The XML node that contains the paths.</param>
        private void ReadGeneralUserShaderPaths( XmlNode node )
        {
            m_generalVtxShaderPaths.Clear();
            m_generalFrgShaderPaths.Clear();
            foreach ( XmlNode child in node.ChildNodes )
            {
                if ( child.Name=="VertexShaderPaths" )
                    ReadShaderPaths( child, ref m_generalVtxShaderPaths );
                else if ( child.Name=="FragmentShaderPaths" )
                    ReadShaderPaths( child, ref m_generalFrgShaderPaths );
            }
        }


        /// <summary>
        /// Read shader file paths.
        /// </summary>
        /// <param name="node">The XML node containing the shader paths.</param>
        /// <param name="shaderPaths">The list to store the paths.</param>
        private void ReadShaderPaths( XmlNode node,
                                      ref List<string> shaderPaths )
        {
            // Clear the list first.
            shaderPaths.Clear();

            foreach ( XmlNode child in node.ChildNodes )
            {
                if ( child.Name!="ShaderPath" )
                    continue;

                string filePath =
                    RequireDocument.GetFullPathForRequireDocument( child.InnerText.Trim() );
                if ( string.IsNullOrEmpty(filePath)==true )
                    continue;

                filePath = ShellUtility.ToAbsolutePath( filePath, Directory.GetCurrentDirectory() );

                try
                {
                    if ( System.IO.File.Exists(filePath)==false )
                    {
                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             res.Strings.WARNING_CUSTOM_SHADER_FILE_NOT_FOUND,
                                             UserShaderUIDefDeserializationBlock.CurrentFilePath,
                                             filePath );
                        continue;
                    }
                }
                catch
                {
                    // The file path is invalid.
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.WARNING_CUSTOM_SHADER_FILE_NOT_FOUND,
                                         UserShaderUIDefDeserializationBlock.CurrentFilePath,
                                         filePath );
                    continue;
                }

                shaderPaths.Add( filePath );
            }
        }


        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml( XmlWriter writer )
        {
            // Not implemented.
        }


        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }


        private readonly static string GENERAL_USER_SHADER_PATHS_TAG_NAME = "GeneralCustomShaders";

        private List<CustomShaderUIDefLinkData> m_definitionList = new List<CustomShaderUIDefLinkData>();

        private List<string> m_generalVtxShaderPaths = new List<string>();
        private List<string> m_generalFrgShaderPaths = new List<string>();
    }
}
