﻿/*---------------------------------------------------------------------------*
  Project:  NintendoWare
  File:     eft_Particle.vsh

  Copyright (C)2011-2013 Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *---------------------------------------------------------------------------*/


//---------------------------------------------------
// デプステクスチャからデプス値を取得する( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_GET_DEPTH_VALUE_FROM_TEXTURE
float GetDepthValueFromTexture( vec4 projection )
{
    float depth = texture2DProj( sysDepthBufferTexture, projection ).r;
    depth = - ( sysScreenFarMultNearFar ) / ( depth  * ( sysScreenSubFarNear ) - sysScreenFar );
    return depth;
}
#endif

//---------------------------------------------------
// デプス値を取得する( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_GET_DEPTH_VALUE
float GetDepthValue( vec4 position )
{
    float depth = position.z / position.w;
    depth = - ( sysScreenFarMultNearFar ) / ( depth * ( sysScreenSubFarNear ) - sysScreenFar );
    return depth;
}
#endif


 //---------------------------------------------------
// ローカル座標系の法線ベクトル取得( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_GET_LOCAL_NORMAL
vec3 GetLocalNormal()
{
#ifdef _USE_NORMAL_ATTR
    return sysNormalAttr;
#else
    return vec3( 0, 0, 1 );
#endif
}
#endif


//---------------------------------------------------
// ローカル座標系の接線ベクトル取得( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_GET_LOCAL_TANGENT
vec3 GetLocalTangent()
{
#ifdef _USE_TANGENT_ATTR
    return sysTangentAttr.xyz;
#else
    return vec3( 1, 0, 0 );
#endif
}
#endif


//---------------------------------------------------
// ローカル座標系の従法線ベクトル取得( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_GET_LOCAL_BINORMAL
vec3 GetLocalBinormal()
{
#if defined( _USE_NORMAL_ATTR ) && defined( _USE_TANGENT_ATTR )
    return cross( sysNormalAttr, sysTangentAttr.xyz ) * sysTangentAttr.w;
#else
    return vec3( 0, 1, 0 );
#endif
}
#endif

//---------------------------------------------------
// プリミティブのスケールZを取得( カスタマイズ可能 )
// X/Y どちらを適用するか分岐します。
//---------------------------------------------------

#if 0
#ifndef USE_USR_GET_PRIMITIVE_SCALE_Z
float GetPrimitiveScaleZ()
{
    float f = GET_BIT_FLAG( BIT_FLAG_PRIMITIVE_SCALE_Y_TO_Z );
    return f * sysPtclScale.y + ( 1 - f ) * sysPtclScale.x;
}
#endif
#else
#ifndef USE_USR_GET_PRIMITIVE_SCALE_Z
float GetPrimitiveScaleZ()
{
// CHECK_BIT_FLAGが利用出来ないため、応急処置
#if 0
    if ( CHECK_BIT_FLAG( BIT_FLAG_PRIMITIVE_SCALE_Y_TO_Z ) )
    {
        return sysPtclScale.y;
    }
    else
    {
        return sysPtclScale.x;
    }
#endif
    return sysPtclScale.y;
}
#endif
#endif

//--------------------------------------------------------------
// カスタムシェーダ処理
//--------------------------------------------------------------
#ifndef USE_USR_FINAL_ADJUSTMENT_VERTEX_PROCSS
void FinalAdjustmentVertexProcess(){}
#endif

#ifndef USE_USR_ADJUSTMENT_WORLD_POSITION
void AdjustmentWorldPosition() {}
#endif

//--------------------------------------------------------------
// エミッタプラグイン処理
//--------------------------------------------------------------
#ifndef USE_EMITTER_PLUGIN_FINAL_ADJUSTMENT_VERTEX_PROCSS
void EP_FinalAdjustmentVertexProcess(){}
#endif

#ifndef USE_EMITTER_PLUGIN_ADJUSTMENT_WORLD_POSITION
void EP_AdjustmentWorldPosition(){}
#endif

#ifndef _USE_STRIPE // ストライプが使用されてない場合のみ（ストライプでは先に定義済み）
//---------------------------------------------------
// Zバッファの奥行きを取得して、指定のz値が隠れているならを、見えているなら1を返す
//---------------------------------------------------
#ifndef USE_USR_CHECK_DEPTH_BUFFER
float CheckDepthBuffer( vec2 ofs, vec4 outDepthTexCoordProj, float pz )
{
    float tz = GetDepthValueFromTexture( outDepthTexCoordProj + vec4( ofs, 0, 0 ) );
    float dis = tz - pz;
    return step( 0.0, dis );
}
#endif

//---------------------------------------------------
// 遮蔽チェック
//---------------------------------------------------
#ifndef USE_USR_VERTEX_MASKING_CHECK
float CalcMaskingCheck( vec4 ptclFragCoordCenter, float sysPtclCenterDepth )
{
    float alphaRate = 1.0;

    vec4 outDepthTexCoordProj = ptclFragCoordCenter;

    // ピクセルZを取得する
    float cpz = sysPtclCenterDepth;

    // 外側と内側のサンプリング点のデプスバッファとの前後関係を取得
    float r = alphaFunc0.y;
    float out_sum = 0.0;
    float in_sum = 0.0;

    // 外側の点
    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*1.0*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*1.0*r, outDepthTexCoordProj, cpz );

    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*0.7*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*0.7*r, outDepthTexCoordProj, cpz );

    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*0.4*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*0.4*r, outDepthTexCoordProj, cpz );

    // 内側の点
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.2, -0.4 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.2, 0.4 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.4, -0.2 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.4, 0.2 )*1.2*r, outDepthTexCoordProj, cpz );

    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.1, -0.2 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.1, 0.2 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.2, -0.1 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.2, 0.1 )*1.0*r, outDepthTexCoordProj, cpz );

    in_sum += 5.0 * CheckDepthBuffer( vec2( 0, 0 )*r, outDepthTexCoordProj, cpz );

    // 外側の遮蔽率と、内側の遮蔽率を個別に乗算して減衰(内側が全点隠れたら消えるようにするため)
    alphaRate *= clamp( out_sum / 35.0, 0.0, 1.0 );
    alphaRate *= clamp( in_sum / 7.0, 0.0, 1.0 );

    return alphaRate;
}
#endif
#endif

//---------------------------------------------------
// ソフトエッジ処理( カスタマイズ可能 )
//---------------------------------------------------
#ifndef USE_USR_VERTEX_SOFT_EDGE
float CalcVertexSoftEdge()
{
    float alphaRate = 1.0;

    // デプスオフセット版
    float sum       = 0.0;
    float rad       = alphaFunc0.y;
    float offset    = alphaFunc0.x;

    // パーティクル中心
    float cpz  = sysPtclVertexDepthValue;
    float ctz  = sysPtclTextureDepthValue;

    float cdis = ctz - cpz;
    cdis /= offset;
    cdis = clamp( cdis, 0.0, 1.0 );

    alphaRate = cdis;

#ifdef _VFX_NSDK01215_COMPATIBLE
#else
    // 円形サンプリング
    vec4 nearPos[16];
    nearPos[0]  = vec4(  1.0   ,  0.0   , offset, 0.0 );
    nearPos[1]  = vec4(  0.9239,  0.3827, offset, 0.0 );
    nearPos[2]  = vec4(  0.7071,  0.7071, offset, 0.0 );
    nearPos[3]  = vec4(  0.3827,  0.9239, offset, 0.0 );
    nearPos[4]  = vec4(  0.0   ,  1.0   , offset, 0.0 );
    nearPos[5]  = vec4( -0.9239,  0.3827, offset, 0.0 );
    nearPos[6]  = vec4( -0.7071,  0.7071, offset, 0.0 );
    nearPos[7]  = vec4( -0.3827,  0.9239, offset, 0.0 );
    nearPos[8]  = vec4( -1.0   ,  0.0   , offset, 0.0 );
    nearPos[9]  = vec4( -0.9239, -0.3827, offset, 0.0 );
    nearPos[10] = vec4( -0.7071, -0.7071, offset, 0.0 );
    nearPos[11] = vec4( -0.3827, -0.9239, offset, 0.0 );
    nearPos[12] = vec4(  0.0   , -1.0   , offset, 0.0 );
    nearPos[13] = vec4(  0.9239, -0.3827, offset, 0.0 );
    nearPos[14] = vec4(  0.7071, -0.7071, offset, 0.0 );
    nearPos[15] = vec4(  0.3827, -0.9239, offset, 0.0 );

    for( int i = 0; i < 16; ++i )
    {
        vec4 pos = sysFragCoordVary + nearPos[i] * rad;

        float pz = GetDepthValue( pos );
        float tz = GetDepthValueFromTexture( pos );

        float dis = tz - pz;

        dis = clamp( dis, 0.0, 1.0 );
        sum += dis;
    }

    alphaRate = mix( 0.0, 1.0, sum / 16 );
#endif
    return alphaRate;
}
#endif



//------------------------------------------------------------------------------
// YZX 回転行列を生成します。
//------------------------------------------------------------------------------
mat4 MakeRotationMatrixYZX( vec3 rotate )
{
    //回転の軸は常にWorld空間に固定されます

    mat4 rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = sinX * sinY;
    float opt2 = cosX * sinY;
    float opt3 = sinX * cosY;
    float opt4 = cosX * cosY;

    rotMat[0][0] =  cosY * cosZ;        // (cosY * cosZ)
    rotMat[0][1] =  opt4 * sinZ + opt1; // (cosX * cosY * sinZ) + (sinX * sinY)
    rotMat[0][2] =  opt3 * sinZ - opt2; // (sinX * cosY * sinZ) - (cosX * sinY)
    rotMat[0][3] =  0.0;

    rotMat[1][0] = -sinZ;               //-(sinZ)
    rotMat[1][1] =  cosZ * cosX;        // (cosX * cosZ)
    rotMat[1][2] =  cosZ * sinX;        // (sinX * cosZ)
    rotMat[1][3] =  0.0;

    rotMat[2][0] =  sinY * cosZ;        // (sinY * cosZ)
    rotMat[2][1] =  opt2 * sinZ - opt3; // (cosX * sinY * sinZ) - (sinX * cosY)
    rotMat[2][2] =  opt1 * sinZ + opt4; //-(sinX * sinY * sinZ) + (cosX * cosY)
    rotMat[2][3] =  0.0;

    rotMat[3][0] = 0.0;
    rotMat[3][1] = 0.0;
    rotMat[3][2] = 0.0;
    rotMat[3][3] = 1.0;

    return rotMat;
}

//------------------------------------------------------------------------------
// XYZ 回転行列を生成します。
//------------------------------------------------------------------------------
mat4 MakeRotationMatrixXYZ( vec3 rotate )
{
    //回転の軸は常にWorld空間に固定されます

    mat4 rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosX * sinZ;
    float opt2 = cosX * cosZ;
    float opt3 = sinX * sinZ;
    float opt4 = sinX * cosZ;

    rotMat[0][0] = cosY * cosZ;         // (cosY * cosZ)
    rotMat[0][1] = cosY * sinZ;         // (cosY * sinZ)
    rotMat[0][2] = -sinY;               //-(sinY)
    rotMat[0][3] = 0.0;

    rotMat[1][0] = opt4 * sinY - opt1;  // (sinX * sinY * cosZ) - (cosX * sinZ)
    rotMat[1][1] = opt3 * sinY + opt2;  // (sinX * sinY * sinZ) + (cosX * cosZ)
    rotMat[1][2] = sinX * cosY;         // (sinX * cosY)
    rotMat[1][3] = 0.0;

    rotMat[2][0] = opt2 * sinY + opt3;  // (cosX * sinY * cosZ) + (sinX * sinZ)
    rotMat[2][1] = opt1 * sinY - opt4;  // (cosX * sinY * sinZ) - (sinX * sinZ)
    rotMat[2][2] = cosX * cosY;         // (cosX * cosY)
    rotMat[2][3] = 0.0;

    rotMat[3][0] = 0.0;
    rotMat[3][1] = 0.0;
    rotMat[3][2] = 0.0;
    rotMat[3][3] = 1.0;

    return rotMat;
}

//------------------------------------------------------------------------------
// ZXY 回転行列を生成します。
//------------------------------------------------------------------------------
mat4 MakeRotationMatrixZXY( vec3 rotate )
{
    //回転の軸は常にWorld空間に固定されます

    mat4 rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosY * cosZ;
    float opt2 = cosY * sinZ;
    float opt3 = sinY * cosZ;
    float opt4 = sinY * sinZ;

    rotMat[0][0] = sinX * opt4 + opt1;  // (sinX * sinY * sinZ) + (cosY * cosZ)
    rotMat[0][1] = cosX * sinZ;         // (cosX * sinZ)
    rotMat[0][2] = sinX * opt2 - opt3;  // (sinX * cosY * sinZ) - (sinY * cosZ)
    rotMat[0][3] = 0.0;

    rotMat[1][0] = sinX * opt3 - opt2;  // (sinX * sinY * cosZ) - (cosY * sinZ)
    rotMat[1][1] = cosX * cosZ;         // (cosX * cosZ)
    rotMat[1][2] = sinX * opt1 + opt4;  // (sinX * cosY * cosZ) + (sinY * sinZ)
    rotMat[1][3] = 0.0;

    rotMat[2][0] = cosX * sinY;         // (cosX * sinY)
    rotMat[2][1] = -sinX;               //-(sinX)
    rotMat[2][2] = cosX * cosY;         // (cosX * cosY)
    rotMat[2][3] = 0.0;

    rotMat[3][0] = 0.0;
    rotMat[3][1] = 0.0;
    rotMat[3][2] = 0.0;
    rotMat[3][3] = 1.0;

    return rotMat;
}

//------------------------------------------------------------------------------
//  揺らぎ 処理
//------------------------------------------------------------------------------
// 揺らぎ : Sin波
float CalcFluctuation_Sin( vec4 fluctuation, float time )
{
    #define amplitude    fluctuation.x
    #define cycle        fluctuation.y
    #define phaseRandom  fluctuation.z
    #define phaseInit    fluctuation.w

    float t = ( ( time + phaseInit ) / cycle ) + ( phaseRandom * sysPtclRandomX );   //現在フレーム位置（割合）

    // 正弦波の揺らぎ量を計算。
    // 掛け率 1.0 から amplitude 分減算する。
    // amplitude := [0.0, 1.0]

    float flucValue =  1.0 - ( ( cos( t * 6.2831853 ) + 1.0 ) * 0.5 * amplitude );   //揺らぎ量

    return flucValue;
}

// 揺らぎ : のこぎり波
float CalcFluctuation_SawTooth( vec4 fluctuation, float time )
{
    #define amplitude    fluctuation.x
    #define cycle        fluctuation.y
    #define phaseRandom  fluctuation.z
    #define phaseInit    fluctuation.w

    float t = (( time + phaseInit ) / cycle ) + ( phaseRandom * sysPtclRandomX );    //現在フレーム位置（割合）
    float flucValue = abs( ( 1.0 - fract( t ) * amplitude ) );                        //揺らぎ量

    return flucValue;
}

// 揺らぎ : 矩形波
float CalcFluctuation_Rect( vec4 fluctuation, float time )
{
    #define amplitude    fluctuation.x
    #define cycle        fluctuation.y
    #define phaseRandom  fluctuation.z
    #define phaseInit    fluctuation.w

    float t = (( time + phaseInit ) / cycle ) + ( phaseRandom * sysPtclRandomX );    //現在フレーム位置（割合）
    float flucValue = abs( 1.0 - ( 1.0 - step( 0.5, fract( t ) ) ) * amplitude );       //揺らぎ量

    return flucValue;
}


//------------------------------------------------------------------------------
// ディレクショナル計算
//------------------------------------------------------------------------------
void CalcDirectionalParticleType( vec3 localPos, vec3 upVector )
{
    vec3 basisX;
    vec3 basisY;
    vec3 basisZ;

    mat4 matVelLook = mat4( 1.0, 0.0, 0.0, 0.0,
                            0.0, 1.0, 0.0, 0.0,
                            0.0, 0.0, 1.0, 0.0,
                            0.0, 0.0, 0.0, 1.0 );

    basisY.xyz = sysPtclWorldPosDelta;
    basisY = normalize( basisY );

    basisX = cross( upVector, basisY );
    basisX = normalize( basisX );

    basisZ = cross( basisX, basisY );

    matVelLook[0][0] = basisX.x;
    matVelLook[1][0] = basisY.x;
    matVelLook[2][0] = basisZ.x;
    matVelLook[3][0] = 0.0;
    matVelLook[0][1] = basisX.y;
    matVelLook[1][1] = basisY.y;
    matVelLook[2][1] = basisZ.y;
    matVelLook[3][1] = 0.0;
    matVelLook[0][2] = basisX.z;
    matVelLook[1][2] = basisY.z;
    matVelLook[2][2] = basisZ.z;
    matVelLook[3][2] = 0.0;

#ifdef _USE_ROTATE
    matVelLook *= sysPtclRotateMatrix;
#endif

    gl_Position.xyz = ( matVelLook * vec4( localPos, 0 ) ).xyz + sysPtclWorldPos;

    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();
    sysWorldNormalVary    = ( matVelLook * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( matVelLook * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( matVelLook * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;
}

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - ビルボードパーティクル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_BILLBOARD
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

#ifdef _USE_ROTATE
    mat4 transposeMat   = transpose( bldMat ) * sysPtclRotateMatrix;
#else
    mat4 transposeMat   = transpose( bldMat );
#endif
    sysWorldNormalVary    = ( transposeMat * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( transposeMat * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( transposeMat * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;

#ifdef _USE_ROTATE
    vec3 temp = ( sysPtclRotateMatrix * vec4( localPos, 1 ) ).xyz;
    gl_Position.xyz = ( vec4( temp, 1 ) * bldMat ).xyz + sysPtclWorldPos;
#else
    gl_Position.xyz = ( vec4( localPos, 1 ) * bldMat ).xyz + sysPtclWorldPos;
#endif

}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - 高機能ビルボードパーティクル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_COMPLEX_BILLBOARD
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

    // ビルボードマトリクス
    vec3 viewUp = vec3( sysViewMatrix[1].x, sysViewMatrix[1].y, sysViewMatrix[1].z );

    vec3 zAxis = normalize( sysEyePosition.xyz - sysPtclWorldPos.xyz );
    vec3 xAxis = normalize( cross( zAxis, -viewUp ) );
    vec3 yAxis = cross( zAxis, xAxis );

    mat4 cmpBillboardMat = mat4(
        xAxis.x, yAxis.x, zAxis.x, 0,
        xAxis.y, yAxis.y, zAxis.y, 0,
        xAxis.z, yAxis.z, zAxis.z, 0,
              0,       0,       0, 1
        );

#ifdef _USE_ROTATE
    mat4 transposeMat   = transpose( cmpBillboardMat ) * sysPtclRotateMatrix;
#else
    mat4 transposeMat   = transpose( cmpBillboardMat );
#endif

    sysWorldNormalVary    = ( transposeMat * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( transposeMat * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( transposeMat * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;

#ifdef _USE_ROTATE
    vec3 temp = ( sysPtclRotateMatrix * vec4( localPos, 1 ) ).xyz;
    gl_Position.xyz = ( vec4( temp, 1 ) * cmpBillboardMat ).xyz + sysPtclWorldPos;
#else
    gl_Position.xyz = ( vec4( localPos, 1 ) * cmpBillboardMat ).xyz + sysPtclWorldPos;
#endif
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - Yビルボードパーティクル（カメラを向く）
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_Y_BILLBOARD
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

    sysWorldNormalVary    = ( sysPtclRotateMatrix * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( sysPtclRotateMatrix * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( sysPtclRotateMatrix * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;

    vec3 temp = ( sysPtclRotateMatrix * vec4( localPos, 1 ) ).xyz;
    gl_Position.xyz = temp + sysPtclWorldPos;
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - Yビルボードパーティクル（カメラと平行）
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

    sysWorldNormalVary    = ( sysPtclRotateMatrix * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( sysPtclRotateMatrix * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( sysPtclRotateMatrix * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;

    vec3 temp = ( sysPtclRotateMatrix * vec4( localPos, 1 ) ).xyz;
    gl_Position.xyz = temp + sysPtclWorldPos;
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - 立体風ビルボード
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_WITH_SCALE_Z
void ProcessParticleType( vec3 localPos )
{
    // 基本的な向きは標準ビルボードと同じ
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

#ifdef _USE_ROTATE
    mat4 transposeMat   = transpose( bldMat ) * sysPtclRotateMatrix;
#else
    mat4 transposeMat   = transpose( bldMat );
#endif
    sysWorldNormalVary    = ( transposeMat * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( transposeMat * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( transposeMat * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;

#ifdef _USE_ROTATE
    vec3 temp = ( sysPtclRotateMatrix * vec4( localPos, 1 ) ).xyz;
    gl_Position.xyz = ( vec4( temp, 1 ) * bldMat ).xyz + sysPtclWorldPos;
#else
    gl_Position.xyz = ( vec4( localPos, 1 ) * bldMat ).xyz + sysPtclWorldPos;
#endif
}
#endif


//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - XYパーティクル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_POLYGON_XY
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

    mat4    tempMat;
    tempMat[0]            = vec4( sysEmitterRT[0].xyz, sysPtclWorldPos.x );
    tempMat[1]            = vec4( sysEmitterRT[1].xyz, sysPtclWorldPos.y );
    tempMat[2]            = vec4( sysEmitterRT[2].xyz, sysPtclWorldPos.z );
    tempMat[3]            = vec4( 0.0, 0.0, 0.0, 1.0 );

#ifdef _USE_ROTATE
    gl_Position           = ( sysPtclRotateMatrix * vec4( localPos, 1 ) );
    gl_Position           = gl_Position * tempMat;
    sysWorldNormalVary    = ( sysPtclRotateMatrix * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( sysPtclRotateMatrix * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( sysPtclRotateMatrix * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;
#else
    gl_Position           = vec4( localPos, 1 ) * tempMat;
#endif

    sysWorldNormalVary    = ( vec4( sysWorldNormalVary,   0.0 ) * tempMat ).xyz;
    sysWorldTangentVary   = ( vec4( sysWorldTangentVary,  0.0 ) * tempMat ).xyz;
    sysWorldBinormalVary  = ( vec4( sysWorldBinormalVary, 0.0 ) * tempMat ).xyz;
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - XZパーティクル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_POLYGON_XZ
void ProcessParticleType( vec3 localPos )
{
    sysWorldNormalVary    = GetLocalNormal();
    sysWorldTangentVary   = GetLocalTangent();
    sysWorldBinormalVary  = GetLocalBinormal();

    vec4    yzFlipPos;
    yzFlipPos.xyz             = localPos.xzy;
    yzFlipPos.w               = 1.0;
    sysWorldNormalVary.xyz    = sysWorldNormalVary.xzy;
    sysWorldTangentVary.xyz   = sysWorldTangentVary.xzy;
    sysWorldBinormalVary.xyz  = sysWorldBinormalVary.xzy;
    yzFlipPos.z               = -yzFlipPos.z;
    sysWorldNormalVary.z      = -sysWorldNormalVary.z;
    sysWorldTangentVary.z     = -sysWorldTangentVary.z;
    sysWorldBinormalVary.z    = -sysWorldBinormalVary.z;

#ifdef _USE_ROTATE
    gl_Position           = sysPtclRotateMatrix * yzFlipPos;
    sysWorldNormalVary    = ( sysPtclRotateMatrix * vec4( sysWorldNormalVary,   0.0 ) ).xyz;
    sysWorldTangentVary   = ( sysPtclRotateMatrix * vec4( sysWorldTangentVary,  0.0 ) ).xyz;
    sysWorldBinormalVary  = ( sysPtclRotateMatrix * vec4( sysWorldBinormalVary, 0.0 ) ).xyz;
#else
    gl_Position           = yzFlipPos;
#endif

    mat4    tempMat;
    tempMat[0]          = vec4( sysEmitterRT[0].xyz, sysPtclWorldPos.x );
    tempMat[1]          = vec4( sysEmitterRT[1].xyz, sysPtclWorldPos.y );
    tempMat[2]          = vec4( sysEmitterRT[2].xyz, sysPtclWorldPos.z );
    tempMat[3]          = vec4( 0.0, 0.0, 0.0, 1.0 );

    gl_Position         = gl_Position * tempMat;
    sysWorldNormalVary    = ( vec4( sysWorldNormalVary,   0.0 ) * tempMat ).xyz;
    sysWorldTangentVary   = ( vec4( sysWorldTangentVary,  0.0 ) * tempMat ).xyz;
    sysWorldBinormalVary  = ( vec4( sysWorldBinormalVary, 0.0 ) * tempMat ).xyz;
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - ディレクショナルYパーティクル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_VEL_LOOK
void ProcessParticleType( vec3 localPos )
{
    vec3 emitterUp = -eyeVec.xyz;
    CalcDirectionalParticleType( localPos, emitterUp );
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - ディレクショナルポリゴン
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_VEL_LOOK_POLYGON
void ProcessParticleType( vec3 localPos )
{
    vec3 emitterUp;
    emitterUp.x      = sysEmitterRT[0].y;
    emitterUp.y      = sysEmitterRT[1].y;
    emitterUp.z      = sysEmitterRT[2].y;

    if( cross( emitterUp.xyz, sysPtclWorldPosDelta.xyz ).x == 0 )
    {
        // MEMO: エミッタ上方向とパーティクル進行方向が重なると、
        //       ディレクショナルポリゴンの右方向ベクトルの計算に失敗するので、
        //       事前に一時変数をちょっとエミッタz方向にずらしておく（こうすると後の外積が0にならない）
        emitterUp.xyz += vec3( sysEmitterRT[0].z, sysEmitterRT[1].z, sysEmitterRT[2].z ) * 0.001f;
    }

    // MEMO: そのままだとY軸下方向にポリゴンが向くので、ここで裏返しておく。
    //       根本対処は CalcDirectionalParticleType() のリファクタリング時に行った方が無難。
    CalcDirectionalParticleType( localPos, -emitterUp );
}
#endif

//------------------------------------------------------------------------------
// 各ビルボードタイプごとの処理 - 立体風ビルボードディレクショナル
//------------------------------------------------------------------------------
#ifdef _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL
void ProcessParticleType( vec3 localPos )
{
    vec3 emitterUp;
    emitterUp.x      = sysEmitterRT[0].y;
    emitterUp.y      = sysEmitterRT[1].y;
    emitterUp.z      = sysEmitterRT[2].y;

    CalcDirectionalParticleType( localPos, emitterUp );
}
#endif

//------------------------------------------------------------------------------
// タイムからベクトルを生成
//------------------------------------------------------------------------------
vec3 GetVectorFromTime( vec3 localVec, float time )
{
    #define airRegist               ubVec.x             // 空気抵抗
    #define gravityX                ubGrav.x            // 重力X
    #define gravityY                ubGrav.y            // 重力Y
    #define gravityZ                ubGrav.z            // 重力Z
    #define gravityScale            ubGrav.w            // 重力スケール

    float k = airRegist;
    float t = time;
    vec3  gravity   = vec3 ( gravityX, gravityY, gravityZ );

    if ( k == 1.0 )
    {
        // 重力のみ
        gravity *= ( gravityScale * ( t * t * 0.5 ) );
    }
    else
    {
        // 重力 + 空気抵抗
        float klog = log( k );
        float kpow = pow( k, t );
        float tempK = 1.0 / ( 1.0 - k );
        float tempKPow = ( t - ( ( kpow - 1.0 ) / klog ) );
        float integration = tempK * tempKPow;

        gravity *= ( gravityScale * integration );
    }

#ifdef _WORLD_GRAVITY
    {
        float gravX = gravity.x;
        float gravY = gravity.y;
        float gravZ = gravity.z;
        gravity.x =  gravX * sysEmitterRT[0].x + gravY * sysEmitterRT[1].x + gravZ * sysEmitterRT[2].x;
        gravity.y =  gravX * sysEmitterRT[0].y + gravY * sysEmitterRT[1].y + gravZ * sysEmitterRT[2].y;
        gravity.z =  gravX * sysEmitterRT[0].z + gravY * sysEmitterRT[1].z + gravZ * sysEmitterRT[2].z;
    }
#endif

    // 空気抵抗値を加味した時間位置
    float velTime = ( airRegist == 1.0 ) ? time : ( 1.0 - pow( airRegist, time ) )/( 1.0 - airRegist );

    // 運動量ランダムを含める。
    vec3 ret = ( localVec * velTime ) + gravity;

    // 移動量の積算を計算
    return ret;
}

//------------------------------------------------------------------------------
// 回転行列生成
//------------------------------------------------------------------------------
#if defined( _USE_ROTATE ) || defined( _PARTICLE_TYPE_Y_BILLBOARD ) || defined( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL )
mat4 CalcRotationMatrix( float time )
{
    #define rotInitX             ubRot0.x       // 初期値 X
    #define rotInitY             ubRot0.y       // 初期値 Y
    #define rotInitZ             ubRot0.z       // 初期値 Z
    #define rotInitRndX          ubRot1.x       // 初期値ランダム X
    #define rotInitRndY          ubRot1.y       // 初期値ランダム Y
    #define rotInitRndZ          ubRot1.z       // 初期値ランダム Z
    #define rotVelX              ubRot2.x       // 加算値 X
    #define rotVelY              ubRot2.y       // 加算値 Y
    #define rotVelZ              ubRot2.z       // 加算値 Z
    #define rotVelRegist         ubRot2.w       // 減衰値
    #define rotVelRandX          ubRot3.x       // 加算値ランダム X
    #define rotVelRandY          ubRot3.y       // 加算値ランダム Y
    #define rotVelRandZ          ubRot3.z       // 加算値ランダム Z

    // ※sysPtclRandom変数は初回に乱数を与えられて以降は固定

    // 乱数を増やす
    float tempRandX = ( sysPtclRandomX + sysPtclRandomY ) / 2.0;
    float tempRandY = ( sysPtclRandomY + sysPtclRandomZ ) / 2.0;
    float tempRandZ = ( sysPtclRandomZ + sysPtclRandomX ) / 2.0;
    // [-1.0, 1.0]
    tempRandX = (tempRandX - 0.5) * 2.0;
    tempRandY = (tempRandY - 0.5) * 2.0;
    tempRandZ = (tempRandZ - 0.5) * 2.0;

    float tempRandX2 = sysPtclRandomX;
    float tempRandY2 = sysPtclRandomY;
    float tempRandZ2 = sysPtclRandomZ;
    // [-0.5, 0.5]
    tempRandX2 = (tempRandX2 - 0.5);
    tempRandY2 = (tempRandY2 - 0.5);
    tempRandZ2 = (tempRandZ2 - 0.5);

    // 回転速度（ランダム量込み）
    float rVelX = rotVelX + ( tempRandX * rotVelRandX );
    float rVelY = rotVelY + ( tempRandY * rotVelRandY );
    float rVelZ = rotVelZ + ( tempRandZ * rotVelRandZ );

    // 初期回転
#ifdef _USE_ROTATE
    float rInitX = sysInitRotateAttr.x;
    float rInitY = sysInitRotateAttr.y;
    float rInitZ = sysInitRotateAttr.z;
#else
    float rInitX = 0.0;
    float rInitY = 0.0;
    float rInitZ = 0.0;
#endif

    // 回転方向ランダム
#if 0
    if ( CHECK_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_X ) )
    {
        if ( sysPtclRandomY > 0.5 )
        {
            rVelX = -rVelX;
            rInitX = -rInitX;
        }
    }
    if ( CHECK_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_Y ) )
    {
        if ( sysPtclRandomZ > 0.5 )
        {
            rVelY = -rVelY;
            rInitY = -rInitY;
        }
    }
    if ( CHECK_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_Z ) )
    {
        if ( sysPtclRandomX > 0.5 )
        {
            rVelZ = -rVelZ;
            rInitZ = -rInitZ;
        }
    }
#else
    {
        // MEMO: 一行目で、フラグのON/OFFを1/0で取得
        //       二行目で、乱数が0.5f以上なら1が、未満なら0が乗算され、最終的な1/0が返る（回転方向を反転するか否かの）
        float fx = GET_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_X );
        float fy = GET_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_Y );
        float fz = GET_BIT_FLAG( BIT_FLAG_ROTATE_DIR_RANDOM_Z );
        fx *= GET_FLAG_RANDOM_HALF( sysPtclRandomY );
        fy *= GET_FLAG_RANDOM_HALF( sysPtclRandomZ );
        fz *= GET_FLAG_RANDOM_HALF( sysPtclRandomX );

        // MEMO: 方向反転する場合（fx,fy,fz）が1.0の場合、逆方向の2倍のベクトルを加算して向きを反転
        rVelX  += -2.0 * fx * rVelX;
        rInitX += -2.0 * fx * rInitX;
        rVelY  += -2.0 * fy * rVelY;
        rInitY += -2.0 * fy * rInitY;
        rVelZ  += -2.0 * fz * rVelZ;
        rInitZ += -2.0 * fz * rInitZ;
    }
#endif

    //回転初期値（ランダム量込み）
    float rotX = rInitX + ( tempRandX2 * rotInitRndX );
    float rotY = rInitY + ( tempRandY2 * rotInitRndY );
    float rotZ = rInitZ + ( tempRandZ2 * rotInitRndZ );

    //減衰率を計算
    float rotRegist = pow( rotVelRegist, time );
    float rotTime = ( rotVelRegist == 1.0 ) ? time : ( 1.0 - rotRegist )/( 1.0 - rotVelRegist );

    //現在時刻の回転量を計算
    float nRotX = ( rVelX * rotTime ) + rotX;
    float nRotY = ( rVelY * rotTime ) + rotY;
    float nRotZ = ( rVelZ * rotTime ) + rotZ;

    // ビルボードY値を計算
#ifdef _PARTICLE_TYPE_Y_BILLBOARD
    float billboardY = atan( -( eyePos.x - sysPtclWorldPos.x ), ( eyePos.z - sysPtclWorldPos.z ) );
    nRotY -= billboardY;
#endif

    // ビルボードY（カメラに平行）の値を計算
#ifdef _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL
    float billboardY = atan( eyeVec.x, eyeVec.z );
    nRotY += billboardY;
#endif

    // 回転タイプにより分岐
#ifdef _ROTATE_YZX
    mat4 ret = MakeRotationMatrixYZX( vec3( nRotX, nRotY, nRotZ ) );
#endif
#ifdef _ROTATE_XYZ
    mat4 ret = MakeRotationMatrixXYZ( vec3( nRotX, nRotY, nRotZ ) );
#endif
#ifdef _ROTATE_ZXY
    mat4 ret = MakeRotationMatrixZXY( vec3( nRotX, nRotY, nRotZ ) );
#endif

    return ret;
}
#endif


//------------------------------------------------------------------------------
// main
//------------------------------------------------------------------------------
#ifndef USE_USR_VERTEX_MAIN
void main()
{
    #define colorScale          ubCol0.x        // カラースケール

    // パーティクル時間
    sysPtclTime = sysEmitterTime - sysPtclEmitTime;
    // CPUパーティクル時には、sysPtclEmitTimeが InitRot.w に書かれるため、
    // パーティクル時間が負になり、不正なタイミングで描画される可能性がある。
    if ( sysPtclTime < 0 )
    {
        DiscardVertex();
        return;
    }

    // 寿命を全うしたパーティクルは描画しない
    if ( sysPtclTime >= sysPtclLife )
    {
        DiscardVertex();
        return;
    }

    // 最終アルファを1.0からスタート。
    sysMasterAlpha = 1.0;

#if defined( _USE_NN_VFX ) && defined( _PARTICLE )

    // sysPosAttr を gl_VertexID から捻出する
    sysPosAttr.x = ( gl_VertexID & 1 ) != 0 ? 0.5 : -0.5;
    sysPosAttr.y = ( gl_VertexID & 2 ) != 0 ? 0.5 : -0.5;
    sysPosAttr.z = 0.0;

    // 頂点インデックス
    sysIndexAttr = gl_VertexID;

#else

    // 頂点インデックス
    sysIndexAttr = int( sysPosAttr.w );

#endif

    // エミッタRTマトリクス生成
    MakeEmitterRTMatrix();

    //--------------------------------------------------------------
    // 揺らぎ
    //--------------------------------------------------------------
    float ptclFluctuationX = 1.0;
    float ptclFluctuationY = 1.0;
    float ptclFluctuationZ = 1.0;

#if 0
#if defined( _FLUCTUATION_SCALE_ENABLE ) || defined( _FLUCTUATION_ALPHA_ENABLE )
    vec4 flucX = vec4 ( fluc.pm0.x, fluc.pm0.z, fluc.pm1.x, fluc.pm1.z );
    if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SIN   ) )      ptclFluctuationX = CalcFluctuation_Sin( flucX, sysPtclTime );
    else if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SAW_TOOTH ) )  ptclFluctuationX = CalcFluctuation_SawTooth( flucX, sysPtclTime );
    else if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_RECT  ) )      ptclFluctuationX = CalcFluctuation_Rect( flucX, sysPtclTime );
    ptclFluctuationY = ptclFluctuationX;
#endif
#ifdef _FLUCTUATION_SCALE_Y_SEPARATE
    vec4 flucY = vec4 ( fluc.pm0.y, fluc.pm0.w, fluc.pm1.y, fluc.pm1.w );
    if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SIN  ) )       ptclFluctuationY = CalcFluctuation_Sin( flucY, sysPtclTime );
    else if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SAW_TOOTH ) )  ptclFluctuationY = CalcFluctuation_SawTooth( flucY, sysPtclTime );
    else if ( CHECK_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_RECT ) )       ptclFluctuationY = CalcFluctuation_Rect( flucY, sysPtclTime );
#endif
#else

#if defined( _FLUCTUATION_SCALE_ENABLE ) || defined( _FLUCTUATION_ALPHA_ENABLE )
    {
        vec4 flucX = vec4 ( fluc.pm0.x, fluc.pm0.z, fluc.pm1.x, fluc.pm1.z );
        vec3 v;
        v.x = GET_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SIN );
        v.y = GET_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_SAW_TOOTH );
        v.z = GET_BIT_FLAG( BIT_FLAG_FLUCTUATION_TYPE_RECT );
        vec3 ret;
        ret.x = CalcFluctuation_Sin( flucX, sysPtclTime );
        ret.y = CalcFluctuation_SawTooth( flucX, sysPtclTime );
        ret.z = CalcFluctuation_Rect( flucX, sysPtclTime );
        ptclFluctuationX = dot( v, ret );

#ifndef _FLUCTUATION_SCALE_Y_SEPARATE
        // スケール揺らぎYを個別指定しない＝XYZ連動
        ptclFluctuationY = ptclFluctuationX;
        ptclFluctuationZ = ptclFluctuationX;    // スケールZ も連動
#else
        vec4 flucY = vec4 ( fluc.pm0.y, fluc.pm0.w, fluc.pm1.y, fluc.pm1.w );
        ret.x = CalcFluctuation_Sin( flucY, sysPtclTime );
        ret.y = CalcFluctuation_SawTooth( flucY, sysPtclTime );
        ret.z = CalcFluctuation_Rect( flucY, sysPtclTime );
        ptclFluctuationY = dot( v, ret );
#endif
    }
#endif
#endif

    //--------------------------------------------------------------
    // パーティクルスケール
    //--------------------------------------------------------------
    // 入力スケール
    sysPtclScale.xyz = sysScaleAttr.xyz;

    {
        //--------------------------------------------------------------
        // TODO: sysPtclRandomX をいかなる場合もデッドストリップさせないための仕込み。sysPtclRandomX がデッドストリップ可になったら外す。
        //--------------------------------------------------------------
        sysPtclScale.x += clamp( sysPtclRandomX, 0, 0 ) ;  // 計算上の意味はない。
    }

#if _MACRO_BRANCH_8KEY_ANIME
#ifdef _SCALE_ANIM_1_KEY
    sysPtclScale.xyz *= scaleAnim.v[0].xyz;
#endif
#ifdef _SCALE_ANIM_2_KEY
    sysPtclScale.xyz *= Calculate2KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_3_KEY
    sysPtclScale.xyz *= Calculate3KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_4_KEY
    sysPtclScale.xyz *= Calculate4KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_5_KEY
    sysPtclScale.xyz *= Calculate5KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_6_KEY
    sysPtclScale.xyz *= Calculate6KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_7_KEY
    sysPtclScale.xyz *= Calculate7KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#ifdef _SCALE_ANIM_8_KEY
    sysPtclScale.xyz *= Calculate8KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#else
#ifdef _SCALE_ANIM
    sysPtclScale.xyz *= Calculate8KeyAnimScale( scaleAnimKeyNum, sysPtclTime, scaleLoopRate, scaleIsLoopInitRandom, sysPtclLife, sysPtclRandomX ).xyz;
#endif
#endif

    // スケール揺らぎ反映
#if defined( _FLUCTUATION_SCALE_ENABLE )
    sysPtclScale.x *= ptclFluctuationX;
    sysPtclScale.y *= ptclFluctuationY;
#ifndef _PARTICLE
    // プリミティブの場合、Zも揺らぐ
    sysPtclScale.z *= ptclFluctuationZ;
#endif
#endif
    // エミッタパーティクルスケールを適用
    sysPtclScale.x *= sysEmitterParticleScaleX;
    sysPtclScale.y *= sysEmitterParticleScaleY;
    sysPtclScale.z *= sysEmitterParticleScaleZ;

#ifdef _PARTICLE
#if defined( _PARTICLE_TYPE_WITH_SCALE_Z ) || defined( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL )
    // 立体風ビルボード系の場合
    // スケール値を保存
    vec3 scaleOrigin = sysPtclScale.xyz;
#else
    // それ以外の場合、パーティクルのZスケールは1.0fにしておく
    sysPtclScale.z = 1.0f;
#endif
#endif

    //--------------------------------------------------------------
    // GPUパーティクル タイム版
    //--------------------------------------------------------------
#ifdef _CALC_TYPE_GPU_TIME
    // 現時間のベクトルを生成
#ifndef _VFX_ENABLE_PRECISE_GPU_COUNTER_MODE
    vec3 ptclLocalVec = GetVectorFromTime( sysLocalVecAttr.xyz, sysPtclTime + sysFrameRate );
#else
    vec3 ptclLocalVec = GetVectorFromTime( sysLocalVecAttr.xyz, sysPtclTime );
#endif

    // sysPtclTime-frameRate 時間のベクトルを生成
#ifndef _VFX_ENABLE_PRECISE_GPU_COUNTER_MODE
    vec3 ptclPreLocalVec = GetVectorFromTime( sysLocalVecAttr.xyz, sysPtclTime );
    if( sysPtclTime < 0.001 )
    {
        // これ以上前のフレームは存在しないので、方向だけ与える
        vec3 norm = normalize( sysLocalVecAttr.xyz ) * 0.001;
        ptclPreLocalVec = ptclLocalVec - norm;
    }
#else
    vec3 ptclPreLocalVec = GetVectorFromTime( sysLocalVecAttr.xyz, sysPtclTime - sysFrameRate );
#endif

    // パーティクルのローカル位置をタイムから生成
    float dynamicsRand = sysScaleAttr.w;
    sysPtclLocalPos = sysLocalPosAttr.xyz + ptclLocalVec * dynamicsRand ;
    //sysPtclLocalPos = sysLocalPosAttr.xyz + ptclLocalVec;

    // パーティクル世界座標移動差分
    sysPtclWorldPosDelta = ApplyEmitterRotateMatrix( ptclLocalVec - ptclPreLocalVec ).xyz;

#if defined(_PARTICLE_TYPE_VEL_LOOK) || defined(_PARTICLE_TYPE_VEL_LOOK_POLYGON)
    // 座標差分が一定値以下の場合は、初速を姿勢として扱う
    float scalar = length( sysPtclWorldPosDelta );
    if ( scalar < 0.001 )
    {
        sysPtclWorldPosDelta = ApplyEmitterRotateMatrix( ptclLocalVec );
    }
#endif

#endif

    //--------------------------------------------------------------
    // GPUパーティクル ストリームアウト版
    //--------------------------------------------------------------
#ifdef _CALC_TYPE_GPU_SO

#ifdef _USE_NN_VFX

    // パーティクルのローカル位置をCPUで計算済み入力値を利用
    sysPtclLocalPos = sysLocalPosAttr.xyz;

#ifdef _USE_LOCAL_DIFF_ATTR
    // パーティクル世界座標移動差分をCPUで計算済みベクトルを利用
    sysPtclWorldPosDelta = ApplyEmitterRotateMatrix( sysLocalDiffAttr.xyz ).xyz;

    // 位置差分の値により、ディレクショナル系が描画されない症状への対応
    sysPtclWorldPosDelta.z += 0.000001;
#endif

#else

    sysPtclLocalPos = sysInPos.xyz;

    // TODO : パーティクル世界座標移動差分をとりあえずベクトルを入力
    sysPtclWorldPosDelta = ApplyEmitterRotateMatrix( sysInVec.xyz ).xyz;

    // 位置差分の値により、ディレクショナル系が描画されない症状への対応
    sysPtclWorldPosDelta.z += 0.000001;

#endif

#endif

    //--------------------------------------------------------------
    // CPUパーティクル
    //--------------------------------------------------------------
#ifdef _CALC_TYPE_CPU
    // パーティクルのローカル位置をCPUで計算済み入力値を利用
    sysPtclLocalPos = sysLocalPosAttr.xyz;

#ifdef _USE_LOCAL_DIFF_ATTR

    // パーティクル世界座標移動差分をCPUで計算済みベクトルを利用
    sysPtclWorldPosDelta = ApplyEmitterRotateMatrix( sysLocalDiffAttr.xyz ).xyz;

    // 位置差分の値により、ディレクショナル系が描画されない症状への対応
    sysPtclWorldPosDelta.z += 0.000001;

#endif

#endif


    // パーティクル中心世界座標
    sysPtclWorldPos = ApplyEmitterMatrix( sysPtclLocalPos );

    // パーティクルの回転値を生成
#if defined( _USE_ROTATE ) || defined( _PARTICLE_TYPE_Y_BILLBOARD ) || defined( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL )
    sysPtclRotateMatrix = CalcRotationMatrix( sysPtclTime );
#endif

    //--------------------------------------------------------------
    // 頂点処理
    //--------------------------------------------------------------
    // カメラオフセット
#ifdef _CAMERA_OFFSET
    vec3 cameraOffset = CalculateCameraOffset( sysPtclWorldPos );
#endif

#ifdef _CAMERA_OFFSET_FIXED_SIZE
    float fixSizeZOffsetRatio = 1.0;
    vec3 cameraOffset = vec3( 0, 0, 0 );
    {
        float zWorldOffset = ubCenter.z;
#ifdef _CONNECT_PTCL_SCALE_TO_Z_OFFSET
        // エミッタを含むパーティクルスケールの X値 を入れる。（XYが非等価は想定しない）
        zWorldOffset *= sysPtclScale.x;
#endif
        vec3 eyePos = sysEyePosition.xyz;
        vec3 eyeVec = normalize( sysEyeVector.xyz );
        vec3 ptclPos = sysPtclWorldPos.xyz;
        vec3 cameraDist = ptclPos - eyePos;
        vec3 cameraDir = normalize( -cameraDist ); // ＋方向がカメラ側なので反対向きにする
        cameraOffset = cameraDir * zWorldOffset;

        // 大きさが変わらないように拡大・縮小する（相似比を使う）
        float cDistFrom = length( cameraDist );
        float cDistTo   = length( cameraDist + cameraOffset );

        fixSizeZOffsetRatio = cDistTo / cDistFrom;
        sysPtclScale.xyz *= fixSizeZOffsetRatio;
    }
#endif

    // 各ビルボードタイプごとの処理
    // 内部でgl_Position.xyzの更新が行われます。
    ProcessParticleType( vec3( ( sysPosAttr.xy + ( ubCenter.xy * 0.5 ) ) * sysPtclScale.xy, sysPosAttr.z * sysPtclScale.z ) );

    {
        // 世界座標を調整する。
        EP_AdjustmentWorldPosition();   // エミッタプラグインのコールバック呼び出し
        AdjustmentWorldPosition();      // カスタムシェーダのコールバック呼び出し
    }

    // 以降、Ptclのワールド座標は確定した前提。
#ifdef _PARTICLE
#if defined( _PARTICLE_TYPE_WITH_SCALE_Z ) || defined( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL )

#define DOT(A, B) ( A.x * B.x + A.y * B.y + A.z * B.z )

#define worldAxisY = vec3( 0, 1, 0 );                                                                   // ワールドY軸方向

#ifdef _PARTICLE_TYPE_WITH_SCALE_Z
    vec3 localAxisX = ( vec3( sysEmitterRT[0][0], sysEmitterRT[1][0], sysEmitterRT[2][0] ) );  // 仮想箱系のX軸方向
    vec3 localAxisY = ( vec3( sysEmitterRT[0][1], sysEmitterRT[1][1], sysEmitterRT[2][1] ) );  // 仮想箱系のY軸方向
    vec3 localAxisZ = ( vec3( sysEmitterRT[0][2], sysEmitterRT[1][2], sysEmitterRT[2][2] ) );  // 仮想箱系のZ軸方向
#endif
#ifdef _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL
    vec3 localAxisZ = normalize( sysPtclWorldPosDelta.xyz );        // 仮想箱系のZ軸方向
    vec3 localAxisX = ( cross( vec3( 0, 1, 0 ), localAxisZ ) ); // 仮想箱系のX軸方向
    vec3 localAxisY = ( cross( localAxisZ, localAxisX ) ); // 仮想箱系のY軸方向
#endif
    vec3 iOffset = sysPtclWorldPos;//エミッタ位置＆Ptcl位置

    // カメラ方向（仮想箱座標系）を計算
    vec3 camLook  = normalize( iOffset - eyePos.xyz );              // 視線方向
    vec3 camLookH = normalize( DOT( camLook, localAxisX ) * localAxisX + DOT( camLook, localAxisZ ) * localAxisZ );   // 視線方向（水平面）
    vec3 camRight = normalize( cross( camLook, localAxisY ) );      // カメラ右方向
    vec3 camUp    = ( cross( camRight, camLook ) );        // カメラ上方向

    // 各種三角関数の値
    float cosY = DOT( camLook, localAxisY );
    float cosXH = DOT( camLookH, localAxisX );
    float cosZH = DOT( camLookH, localAxisZ );

    float sqCosX = cosXH * cosXH;
    float sq4CosX = sqCosX * sqCosX;

    float sqSinX = 1.0 - sqCosX;
    float sinX = sqrt( sqSinX );
    float sq4SinX = sqSinX * sqSinX;

    float sinY = sqrt( 1.0 - cosY * cosY );
    float sinZ = sqrt( 1.0 - cosZH * cosZH );

    // 俯瞰したときの矩形位置を計算: 正面～右側～背後～左側
    float a0 = -clamp( cosZH, -1, 0 );
    float a1 = -clamp( cosXH, -1, 0 );
    float a2 = clamp( cosZH, 0, 1 );
    float a3 = clamp( cosXH, 0, 1 );

    // 俯角からのスクリーンY方向の高さを決める
    float cX = DOT( camLook, localAxisX );
    float cZ = DOT( camLook, localAxisZ );
    float sX = 1.0 - cX * cX;
    float sZ = 1.0 - cZ * cZ;

    // 水平から見た時の矩形位置を計算
    //ToDo: b0, b1 の値は要調整
    float b0 = ( ( sinX * scaleOrigin.x ) + ( sinZ * scaleOrigin.z ) ) * 0.5 * sinY * sinY;
    float b1 = ( ( scaleOrigin.y * abs( sinY ) ) + ( ( sX * scaleOrigin.x ) * abs( sq4CosX ) + ( sZ * scaleOrigin.z ) * abs( sq4SinX ) ) * abs( cosY ) ) * 0.5;

    vec3 bR = camRight * b0;
    vec3 bU = camUp * b1;

    // y
    // 0-3
    // 1-2 x

    // n: x y
    // 0: - +
    // 1: - -
    // 2: + -
    // 3: + +

    int signU = ( gl_VertexID & 0x2 ) - 1;
    int signV = -( ( gl_VertexID + 1 ) & 0x2 ) + 1;

    vec3 pHorizon = signU * bR + signV * bU;

    float d = sign( DOT( camLook, localAxisY ) );
    if( d == 0 ){ d = 1; }
    vec3 tempX = localAxisX * scaleOrigin.x * 0.5;
    vec3 tempZ = localAxisZ * scaleOrigin.z * 0.5;
    vec3 pVertical =
        (      signU * tempX * a0 +  d * signV * tempZ * a0 ) +
        (  d * signV * tempX * a1 -      signU * tempZ * a1 ) +
        (     -signU * tempX * a2 -  d * signV * tempZ * a2 ) +
        ( -d * signV * tempX * a3 +      signU * tempZ * a3 ) ;

    vec3 pMorph= pHorizon * abs( sinY ) + pVertical * abs( cosY );
    gl_Position.xyz += pMorph + iOffset - gl_Position.xyz;

#endif
#endif

    // 速度スケールY
#ifdef _VELOCITY_SCALE_Y
    {
        // 粒の位置と頂点の位置が重なってないときだけ動かす
        vec3 vSub = gl_Position.xyz - sysPtclWorldPos;
        if( length( vSub ) > 0.0 )
        {
#if defined( _CALC_TYPE_GPU_SO ) && !defined( _USE_NN_VFX )
            // MEMO: eft2 かつ GPU+SO の時は速度が入ってくる
            vec3 vScaleDir = sysPtclWorldPosDelta.xyz;
#else
            // MEMO: それ以外の構成では移動量が入ってくるので、速度換算
            vec3 vScaleDir;
            vec3 vDelta = sysPtclWorldPosDelta.xyz;
            if( sysFrameRate > 0.0 )
            {
#if defined( _CALC_TYPE_GPU_SO )
                // コンピュートシェーダ時は、コンピュートシェーダ内で / sysFrameRate が行われる
                vScaleDir = vDelta.xyz;
#else
                vScaleDir = vDelta.xyz / sysFrameRate;
#endif
            }
            else
            {
#if defined( _CALC_TYPE_GPU_TIME )
                // GPU ではフレームレート0のときに Delta も無くなるので、
                // ほんの少し進めた位置との差分を取って速度化する（接線方向を求める）
                // こうしないと重力分の速度が乗らない。
                float tempFrameRate = 1.0;
                vec3 ptclLocalVecTemp = GetVectorFromTime( sysLocalVecAttr.xyz, sysPtclTime + tempFrameRate );
                vDelta = ApplyEmitterRotateMatrix( ptclLocalVecTemp - ptclPreLocalVec ).xyz;
                vScaleDir = vDelta.xyz;
#else
                // CPU/GPU+SO では Delta 自体は非零なので、速度ベクトルの長さに揃える。
                vScaleDir = vDelta.xyz / length( vDelta.xyz ) * length( sysLocalVecAttr.xyz );
#if defined( _CALC_TYPE_CPU )
                // TODO: CPUエミッタの場合の特殊処理。dynamicsRand のエミッタタイプごとのかけ違いを整える必要がある
                float dynamicsRand = sysScaleAttr.w;
                vScaleDir *= dynamicsRand;
#endif
#endif
            }
#endif
            vec3 npos = normalize( vSub );
            if( length( vDelta.xyz ) > 0.0 )
            {
                vec3 ndiff = normalize( vDelta );
                float power = dot( ndiff, npos );
                gl_Position.xyz += vScaleDir.xyz * addVecToScale.x * power;
            }
        }
    }
#endif

    // カメラオフセット値を加算
#if defined( _CAMERA_OFFSET ) || defined( _CAMERA_OFFSET_FIXED_SIZE )
    sysPtclWorldPos += cameraOffset;
    gl_Position.xyz += cameraOffset;
#endif
    gl_Position.w   = 1;

#if defined( _PARTICLE_SCALE_LIMIT_NEAR ) || defined( _PARTICLE_SCALE_LIMIT_FAR )
    #define scaleLimitDistNear  ubCamScaleLimit.x
    #define scaleLimitDistFar   ubCamScaleLimit.y
    float dist = length( sysPtclWorldPos.xyz - sysEyePosition.xyz );
    vec3 offsetVec = gl_Position.xyz - sysPtclWorldPos.xyz;

#ifdef _CAMERA_OFFSET_FIXED_SIZE
    // サイズ固定カメラオフセットがある場合、先に計算していたスケール補正の逆数を掛けて計算する
    offsetVec /= fixSizeZOffsetRatio;
#endif

#ifdef _LINK_FOVY_TO_SCALE_LIMIT
    float fovyRatio = sysCurrentFovy / sysBaseFovy;
#endif

#endif
#ifdef _PARTICLE_SCALE_LIMIT_NEAR
    // カメラ距離スケール「近」
    {
        vec3 vOffset = offsetVec * ( min( scaleLimitDistNear, dist ) / scaleLimitDistNear );
#ifdef _LINK_FOVY_TO_SCALE_LIMIT
        vOffset *= fovyRatio;
#endif
        gl_Position.xyz = vOffset + sysPtclWorldPos.xyz;
    }
#endif
#ifdef _PARTICLE_SCALE_LIMIT_FAR
    // カメラ距離スケール「遠」
    {
#ifdef _PARTICLE_SCALE_LIMIT_NEAR
        // MEMO: カメラ距離スケール「近」が設定されていない場合は、この条件分岐をサボる。
        //       コードの見た目的によろしくないが…
        if( dist > scaleLimitDistNear )
#endif
        {
            vec3 vOffset = offsetVec * ( max( scaleLimitDistFar, dist ) / scaleLimitDistFar );
#ifdef _LINK_FOVY_TO_SCALE_LIMIT
            vOffset *= fovyRatio;
#endif
            gl_Position.xyz = vOffset + sysPtclWorldPos.xyz;
        }
    }
#endif

    //v_outWorldPosを記憶
    sysWorldPositionVary = gl_Position.xyz;

    // 視線ベクトル
    sysEyeVecVary.xyz = sysWorldPositionVary - eyePos.xyz;

    // スクリーン座標系へ変換
    vec4 viewPos = vec4( gl_Position.xyz, 1 );  // ※保存しておく（下の Zオフセットで使う）
    gl_Position = gl_Position * sysViewProjectionMatrix;

    // デプスオフセット
#ifdef _DEPTH_OFFSET
    //---------------------------------------------------------------------------
    //  zWorldOffset: Zオフセット量（※ワールド座標系での距離）
    //
    //  ビュー座標系でオフセットした時の深度値を求め、
    //  ピクセルシェーダに送られる深度値がそうなるように上書きする。
    //  単純に実装すると負荷が高いので、式展開をして端折った実装を行う。
    //  ※等価な実装をコメントとして以下にメモしておく。
    //---------------------------------------------------------------------------

#ifdef _CONNECT_PTCL_SCALE_TO_Z_OFFSET
    // エミッタを含むパーティクルスケールの X値 を入れる。（XYが非等価は想定しない）
    float zWorldOffset = ubCenter.z * sysPtclScale.x;
#else
    // スケールを乗算しない
    float zWorldOffset = ubCenter.z;
#endif
    viewPos = viewPos * sysViewMatrix;          // ビュー座標系に変換
    viewPos.z += zWorldOffset;                  // ビュー座標系で Z+ 方向（手前）にオフセットする
    viewPos = viewPos * sysProjectionMatrix;    // 射影変換
    float targetZ = viewPos.z / viewPos.w;      // 深度値を取得

    gl_Position.z = gl_Position.w * targetZ;
#endif

#ifdef _AVOID_Z_FIGHTING
    // Zファイティング抑制
    // 深度値空間上で固定値のオフセット
    gl_Position.z -= sysZOffsetForZFighting;
#endif

    // ピクセルシェーダでのスクリーン座標参照用
    sysFragCoordVary = sysTexCoordTransformMatrix * gl_Position;

    // パーティクルの四隅のデプス値を計算
    sysPtclVertexDepthValue = GetDepthValue( sysFragCoordVary );

    // デプスバッファテクスチャからデプス値を取得
    sysPtclTextureDepthValue = GetDepthValueFromTexture( sysFragCoordVary );

    // パーティクルの中心座標がスクリーン座標でどこに置かれるか
#ifdef _DEPTH_OFFSET
    sysPtclFragCoordCenter = sysTexCoordTransformMatrix * ( vec4( sysPtclWorldPos, 1 ) * sysViewProjectionMatrix );
    float coffset = offsetParam - ( offsetParam * sysPtclFragCoordCenter.z ) / sysPtclFragCoordCenter.w;
    sysPtclFragCoordCenter.z -= coffset;
#else
    sysPtclFragCoordCenter = sysTexCoordTransformMatrix * ( vec4( sysPtclWorldPos, 1 ) * sysViewProjectionMatrix );
#endif

    // パーティクルの中心のデプス値を計算
    sysPtclVertexDepthValueCenter = GetDepthValue( sysPtclFragCoordCenter );


    //--------------------------------------------------------------
    // カラー/アルファ 0 処理
    //--------------------------------------------------------------
    // カラー0

#ifdef _COLOR_0_FIXED
    sysColor0Vary.rgb = color0Anim.v[0].xyz;
#endif
#ifdef _COLOR_0_RANDOM
    int index0 = int ( sysPtclRandomX * color0AnimKeyNum );
    sysColor0Vary.rgb = color0Anim.v[index0].xyz;
#endif

#if _MACRO_BRANCH_8KEY_ANIME
#ifdef _COLOR_0_ANIM_1_KEY
    sysColor0Vary.rgb = color0Anim.v[0].xyz;
#endif
#ifdef _COLOR_0_ANIM_2_KEY
    sysColor0Vary.rgb = Calculate2KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_3_KEY
    sysColor0Vary.rgb = Calculate3KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_4_KEY
    sysColor0Vary.rgb = Calculate4KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_5_KEY
    sysColor0Vary.rgb = Calculate5KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_6_KEY
    sysColor0Vary.rgb = Calculate6KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_7_KEY
    sysColor0Vary.rgb = Calculate7KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_0_ANIM_8_KEY
    sysColor0Vary.rgb = Calculate8KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#else
#ifdef _COLOR_0_ANIM
    sysColor0Vary.rgb = Calculate8KeyAnimColor0( color0AnimKeyNum, sysPtclTime, color0LoopRate, color0IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#endif
    // アルファ0
#ifdef _ALPHA_0_FIXED
    sysColor0Vary.a = alpha0Anim.v[0].x;
#endif
#if _MACRO_BRANCH_8KEY_ANIME
#ifdef _ALPHA_0_ANIM_1_KEY
    sysColor0Vary.a = alpha0Anim.v[0].x;
#endif
#ifdef _ALPHA_0_ANIM_2_KEY
    sysColor0Vary.a = Calculate2KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_3_KEY
    sysColor0Vary.a = Calculate3KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_4_KEY
    sysColor0Vary.a = Calculate4KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_5_KEY
    sysColor0Vary.a = Calculate5KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_6_KEY
    sysColor0Vary.a = Calculate6KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_7_KEY
    sysColor0Vary.a = Calculate7KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_0_ANIM_8_KEY
    sysColor0Vary.a = Calculate8KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#else
#ifdef _ALPHA_0_ANIM
    sysColor0Vary.a = Calculate8KeyAnimAlpha0( alpha0AnimKeyNum, sysPtclTime, alpha0LoopRate, alpha0IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#endif

    // カラー0 / アルファ0 継承
#ifdef _COLOR0_INHERIT
    sysColor0Vary.rgb *= sysColor0Attr.rgb;
#endif
#ifdef _ALPHA0_INHERIT
    sysColor0Vary.a *= sysColor0Attr.a;
#endif

    // エミッタカラーを乗算
    sysColor0Vary *= sysEmitterColor0;

    // カラースケール
    sysColor0Vary.rgb *= colorScale;

    //--------------------------------------------------------------
    // カラー/アルファ 1 処理
    //--------------------------------------------------------------
    // カラー1
#ifdef _COLOR_1_FIXED
    sysColor1Vary.rgb = color1Anim.v[0].rgb;
#endif
#ifdef _COLOR_1_RANDOM
    int index1 = int ( sysPtclRandomX * color1AnimKeyNum );
    sysColor1Vary.rgb = color1Anim.v[index1].rgb;
#endif

#if _MACRO_BRANCH_8KEY_ANIME
#ifdef _COLOR_1_ANIM_1_KEY
    sysColor1Vary.rgb = color1Anim.v[0].rgb;
#endif
#ifdef _COLOR_1_ANIM_2_KEY
    sysColor1Vary.rgb = Calculate2KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_3_KEY
    sysColor1Vary.rgb = Calculate3KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_4_KEY
    sysColor1Vary.rgb = Calculate4KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_5_KEY
    sysColor1Vary.rgb = Calculate5KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_6_KEY
    sysColor1Vary.rgb = Calculate6KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_7_KEY
    sysColor1Vary.rgb = Calculate7KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#ifdef _COLOR_1_ANIM_8_KEY
    sysColor1Vary.rgb = Calculate8KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#else
#ifdef _COLOR_1_ANIM
    sysColor1Vary.rgb = Calculate8KeyAnimColor1( color1AnimKeyNum, sysPtclTime, color1LoopRate, color1IsLoopInitRandom, sysPtclLife, sysPtclRandomX );
#endif
#endif
    // アルファ1
#ifdef _ALPHA_1_FIXED
    sysColor1Vary.a = alpha1Anim.v[0].x;
#endif

#if _MACRO_BRANCH_8KEY_ANIME
#ifdef _ALPHA_1_ANIM_1_KEY
    sysColor1Vary.a = alpha1Anim.v[0].x;
#endif
#ifdef _ALPHA_1_ANIM_2_KEY
    sysColor1Vary.a = Calculate2KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_3_KEY
    sysColor1Vary.a = Calculate3KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_4_KEY
    sysColor1Vary.a = Calculate4KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_5_KEY
    sysColor1Vary.a = Calculate5KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_6_KEY
    sysColor1Vary.a = Calculate6KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_7_KEY
    sysColor1Vary.a = Calculate7KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#ifdef _ALPHA_1_ANIM_8_KEY
    sysColor1Vary.a = Calculate8KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#else
#ifdef _ALPHA_1_ANIM
    sysColor1Vary.a = Calculate8KeyAnimAlpha1( alpha1AnimKeyNum, sysPtclTime, alpha1LoopRate, alpha1IsLoopInitRandom, sysPtclLife, sysPtclRandomX ).x;
#endif
#endif

    // カラー0 / アルファ0 継承
#ifdef _COLOR1_INHERIT
    sysColor1Vary.rgb *= sysColor1Attr.rgb;
#endif
#ifdef _COLOR1_INHERIT
    sysColor1Vary.a *= sysColor1Attr.a;
#endif

    // エミッタカラーを乗算
    sysColor1Vary *= sysEmitterColor1;

    // カラースケール
    sysColor1Vary.rgb *= colorScale;

    // プリミティブ頂点カラー
#ifdef _USE_COLOR0_ATTR
    sysPrimitiveColorVary = sysVertexColor0Attr;
#else
    sysPrimitiveColorVary = vec4( 1 );
#endif

    //--------------------------------------------------------------
    // テクスチャ  パターン・座標アニメーション処理
    //--------------------------------------------------------------
    int     ptclTexturePtn0;    // テクスチャパターン 0
    int     ptclTexturePtn1;    // テクスチャパターン 1
    int     ptclTexturePtn2;    // テクスチャパターン 2
    ptclTexturePtn0 = ptclTexturePtn1 = ptclTexturePtn2 = 0;

    // スフィアマップ テクスチャ座標を計算する
#ifdef _USE_SPHERE_MAP
    vec2 sphereMapTexCoord;
    vec3 viewPosition = normalize( ( vec4( sysWorldPositionVary, 1 ) * sysViewMatrix ).xyz );
    vec3 viewNormal   = normalize( ( vec4( sysWorldNormalVary,   0 ) * sysViewMatrix ).xyz );
    viewNormal.z = clamp( viewNormal.z, 0.2, 1.0 );     // 暫定処理（zの負数を回避）
    vec3 r = reflect( viewPosition, viewNormal );
    r.z = r.z + 1.0;
    float m = 2.0 * sqrt( dot(r, r) );
    sphereMapTexCoord.x =  ( r.x / m ) + 0.5;
    sphereMapTexCoord.y = -( r.y / m ) + 0.5;
#endif

#ifdef _PARTICLE
    vec4 defuv = vec4( sysPosAttr.x, sysPosAttr.y, sysPosAttr.x, sysPosAttr.y );
#endif
#ifdef _PRIMITIVE
    vec4 defuv = vec4( sysTexCoordAttr.x, sysTexCoordAttr.y, sysTexCoordAttr.z, sysTexCoordAttr.w );
#endif

#ifndef _TEXTURE0_USE_UV1
    vec2 inUv0   = defuv.xy;
#else
    vec2 inUv0   = defuv.zw;
#endif
#ifndef _TEXTURE1_USE_UV1
    vec2 inUv1   = defuv.xy;
#else
    vec2 inUv1   = defuv.zw;
#endif
#ifndef _TEXTURE2_USE_UV1
    vec2 inUv2   = defuv.xy;
#else
    vec2 inUv2   = defuv.zw;
#endif

    vec4 outUv0  = vec4( 0 );
    vec4 outUv1  = vec4( 0 );
    vec4 outUv2  = vec4( 0 );

    // テクスチャ描画サイズを固定( U / V )
#ifdef _TEX0_KEEP_TEXTURE_SIZE_U
    inUv0.x *= sysPtclScale.x;
#endif
#ifdef _TEX1_KEEP_TEXTURE_SIZE_U
    inUv1.x *= sysPtclScale.x;
#endif
#ifdef _TEX2_KEEP_TEXTURE_SIZE_U
    inUv2.x *= sysPtclScale.x;
#endif
#ifdef _TEX0_KEEP_TEXTURE_SIZE_V
    inUv0.y *= sysPtclScale.y;
#endif
#ifdef _TEX1_KEEP_TEXTURE_SIZE_V
    inUv1.y *= sysPtclScale.y;
#endif
#ifdef _TEX2_KEEP_TEXTURE_SIZE_V
    inUv2.y *= sysPtclScale.y;
#endif


#ifdef _TEX_0_SPHERE_MAP
        inUv0.xy = sphereMapTexCoord;
#endif
#ifdef _TEX_1_SPHERE_MAP
        inUv1.xy = sphereMapTexCoord;
#endif
#ifdef _TEX_2_SPHERE_MAP
        inUv2.xy = sphereMapTexCoord;
#endif

    // テクスチャ反転ランダム
#ifdef _PARTICLE
#ifdef _TEXTURE0_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_U ) ) { if ( sysPtclRandomX > 0.5 ) inUv0.x = -inUv0.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_V ) ) { if ( sysPtclRandomY > 0.5 ) inUv0.y = -inUv0.y; }
#endif
#ifdef _TEXTURE1_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_U ) ) { if ( sysPtclRandomZ > 0.5 ) inUv1.x = -inUv1.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_V ) ) { if ( sysPtclRandomW > 0.5 ) inUv1.y = -inUv1.y; }
#endif
#ifdef _TEXTURE2_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_U ) ) { if ( sysPtclRandomY > 0.5 ) inUv2.x = -inUv2.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_V ) ) { if ( sysPtclRandomZ > 0.5 ) inUv2.y = -inUv2.y; }
#endif
#endif

#ifdef _PRIMITIVE
#ifdef _TEXTURE0_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_U ) ) { if ( sysPtclRandomX > 0.5 ) inUv0.x = 1.0 - inUv0.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_V ) ) { if ( sysPtclRandomY > 0.5 ) inUv0.y = 1.0 - inUv0.y; }
#endif
#ifdef _TEXTURE1_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_U ) ) { if ( sysPtclRandomZ > 0.5 ) inUv1.x = 1.0 - inUv1.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_V ) ) { if ( sysPtclRandomW > 0.5 ) inUv1.y = 1.0 - inUv1.y; }
#endif
#ifdef _TEXTURE2_ENABLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_U ) ) { if ( sysPtclRandomY > 0.5 ) inUv2.x = 1.0 - inUv2.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_V ) ) { if ( sysPtclRandomZ > 0.5 ) inUv2.y = 1.0 - inUv2.y; }
#endif
#endif

    vec3 texPtnRandomSeed;
    texPtnRandomSeed.x = texPtnRandomSeed.y = texPtnRandomSeed.z = sysPtclRandomX;

    // テクスチャ座標アニメーションを計算
    // 乱数を共有するかしないかの設定
    ivec3 texShareRandom = ivec3( 0, 0, 0 );
    if( !CHECK_BIT_FLAG_1( BIT_FLAG_TEX_ANIM_FIXED_RANDOM ) )
    {
        // 乱数を共有しない
        texShareRandom.y = 1;
        texShareRandom.z = 2;

        texPtnRandomSeed.x = sysPtclRandomX;
        texPtnRandomSeed.y = sysPtclRandomY;
        texPtnRandomSeed.z = sysPtclRandomZ;
    }

    // テクスチャ0パターンアニメーション コマ値を計算
#if defined( _TEXTURE0_ENABLE ) && defined( _TEX_0_PATTERN_ANIM )
    ptclTexturePtn0 = CalculateTexPtnAnim0( sysPtclTime, sysPtclLife, texPtnRandomSeed.x );
#endif

    // テクスチャ1パターンアニメーション コマ値を計算
#if defined( _TEXTURE1_ENABLE ) && defined( _TEX_1_PATTERN_ANIM )
    ptclTexturePtn1 = CalculateTexPtnAnim1( sysPtclTime, sysPtclLife, texPtnRandomSeed.y );
#endif

    // テクスチャ2パターンアニメーション コマ値を計算
#if defined( _TEXTURE2_ENABLE ) && defined( _TEX_2_PATTERN_ANIM )
    ptclTexturePtn2 = CalculateTexPtnAnim2( sysPtclTime, sysPtclLife, texPtnRandomSeed.z );
#endif

    vec2 animDir = vec2( 1, 1 );
    vec4 randomSeed = vec4( sysPtclRandomX, sysPtclRandomY, sysPtclRandomZ, sysPtclRandomW );

#ifdef _TEXTURE0_ENABLE
#if defined( _TEX_0_SHIFT_ANIM ) || defined( _TEX_0_SHIFT_ANIM_ROTATE ) || defined( _TEX_0_PATTERN_ANIM )
#ifdef _TEX_0_SHIFT_ANIM_ROTATE
    outUv0 = CalculateTextureShiftAnimWithRotate( inUv0, texShiftAnim0, sysPtclTime, ptclTexturePtn0, texShareRandom.x, randomSeed );
#else
    outUv0 = CalculateTextureShiftAnim( inUv0, texShiftAnim0, sysPtclTime, ptclTexturePtn0, texShareRandom.x, randomSeed );
#endif
#else
#ifdef _PARTICLE
    outUv0.x = ( inUv0.x + 0.5 ) *  texShiftAnim0.pm4.x * ( 1 / texShiftAnim0.pm4.z );
    outUv0.y = ( inUv0.y - 0.5 ) * -texShiftAnim0.pm4.y * ( 1 / texShiftAnim0.pm4.w );
#endif
#ifdef _PRIMITIVE
    outUv0.xy = inUv0.xy * texShiftAnim0.pm4.xy * ( 1 / texShiftAnim0.pm4.zw );
#endif
#endif
#endif


#ifdef _TEXTURE1_ENABLE
#if defined( _TEX_1_SHIFT_ANIM ) || defined( _TEX_1_SHIFT_ANIM_ROTATE ) || defined( _TEX_1_PATTERN_ANIM )
#ifdef _TEX_1_SHIFT_ANIM_ROTATE
    outUv1 = CalculateTextureShiftAnimWithRotate( inUv1, texShiftAnim1, sysPtclTime, ptclTexturePtn1, texShareRandom.y, randomSeed );
#else
    outUv1 = CalculateTextureShiftAnim( inUv1, texShiftAnim1, sysPtclTime, ptclTexturePtn1, texShareRandom.y, randomSeed );
#endif
#else
#ifdef _PARTICLE
    outUv1.x = ( inUv1.x + 0.5 ) *  texShiftAnim1.pm4.x * ( 1 / texShiftAnim1.pm4.z );
    outUv1.y = ( inUv1.y - 0.5 ) * -texShiftAnim1.pm4.y * ( 1 / texShiftAnim1.pm4.w );
#endif
#ifdef _PRIMITIVE
    outUv1.xy = inUv1.xy * texShiftAnim1.pm4.xy * ( 1 / texShiftAnim1.pm4.zw );
#endif
#endif
#endif

#ifdef _TEXTURE2_ENABLE
#if defined( _TEX_2_SHIFT_ANIM ) || defined( _TEX_2_SHIFT_ANIM_ROTATE ) || defined( _TEX_2_PATTERN_ANIM )
#ifdef _TEX_2_SHIFT_ANIM_ROTATE
    outUv2 = CalculateTextureShiftAnimWithRotate( inUv2, texShiftAnim2, sysPtclTime, ptclTexturePtn2, texShareRandom.z, randomSeed );
#else
    outUv2 = CalculateTextureShiftAnim( inUv2, texShiftAnim2, sysPtclTime, ptclTexturePtn2, texShareRandom.z, randomSeed );
#endif
#else
#ifdef _PARTICLE
    outUv2.x = ( inUv2.x + 0.5 ) *  texShiftAnim2.pm4.x * ( 1 / texShiftAnim2.pm4.z );
    outUv2.y = ( inUv2.y - 0.5 ) * -texShiftAnim2.pm4.y * ( 1 / texShiftAnim2.pm4.w );
#endif
#ifdef _PRIMITIVE
    outUv2.xy = inUv2.xy * texShiftAnim2.pm4.xy * ( 1 / texShiftAnim2.pm4.zw );
#endif
#endif
#endif

    // 計算結果を書き込み
    sysTexCoord01Vary.x = outUv0.x;
    sysTexCoord01Vary.y = outUv0.y;
    sysTexCoord01Vary.z = outUv1.x;
    sysTexCoord01Vary.w = outUv1.y;
    sysTexCoord2Vary.x  = outUv2.x;
    sysTexCoord2Vary.y  = outUv2.y;

    //--------------------------------------------------------------
    // アルファ処理
    //--------------------------------------------------------------
    // アルファ係数
    sysPtclAlphaRate        = 1.0;
    float nearDistAlphaRate = 1.0;
    float farDistAlphaRate  = 1.0;
    float maskAlphaRate     = 1.0;
    float softEdgeRate      = 1.0;

    //------------------------------
    // 遮蔽チェック処理
    //------------------------------
#ifdef _MASKING_CHECK
    maskAlphaRate = CalcMaskingCheck( sysPtclFragCoordCenter, sysPtclVertexDepthValueCenter );
    sysPtclAlphaRate *= maskAlphaRate;
#endif

    //------------------------------
    // ソフトパーティクル処理
    //------------------------------
#ifdef _VERTEX_SOFT
    softEdgeRate = CalcVertexSoftEdge();
    sysPtclAlphaRate *= softEdgeRate;
#endif

    //------------------------------
    // フレネル処理
    //------------------------------
#ifdef _VERTEX_FRESNEL_ALPHA
    float fresnel = CalcFresnelRate( sysEyeVecVary.xyz, sysWorldNormalVary );
    sysPtclAlphaRate *= fresnel;
#endif

    //------------------------------
    // 近距離アルファ
    //------------------------------
#ifdef _NEAR_DIST_ALPHA
    #define nearAlphaMin    alphaFunc1.x
    #define nearAlphaMax    alphaFunc1.y
    nearDistAlphaRate = ( sysPtclVertexDepthValueCenter - nearAlphaMin )/( nearAlphaMax - nearAlphaMin );
    nearDistAlphaRate = clamp( nearDistAlphaRate, 0.0, 1.0 );
    sysPtclAlphaRate *= nearDistAlphaRate;
#endif

    //------------------------------
    // 遠距離アルファ
    //------------------------------
#ifdef _FAR_DIST_ALPHA
    #define farAlphaMin    alphaFunc1.z
    #define farAlphaMax    alphaFunc1.w
    farDistAlphaRate = ( sysPtclVertexDepthValueCenter - farAlphaMin )/( farAlphaMax - farAlphaMin );
    farDistAlphaRate = clamp( farDistAlphaRate, 0, 1 );
    farDistAlphaRate = 1- farDistAlphaRate;
    sysPtclAlphaRate *= farDistAlphaRate;
#endif

    // アルファ係数をピクセルに送る
    // フェードのアルファ値もここで処理
    sysMasterAlpha *= ( sysPtclAlphaRate * sysEmitterFadeAlpha );

    // アルファ 揺らぎ
#if defined( _FLUCTUATION_ALPHA_ENABLE )
    sysMasterAlpha *= ptclFluctuationX;
#endif

    // 寿命係数
    sysLifeRatio = sysPtclTime / sysPtclLife;

    // シェーダアニメーションを計算
#if defined( _SHADER_ANIM ) || defined( _VFX_COMBINER_SHADER_ANIM_ENABLED )
    vec3 vShaderAnim = Calculate8KeyAnimShader( shaderAnimKeyNum, sysPtclTime, 0, 0, sysPtclLife, sysPtclRandomX ).xyz;
    sysShaderAnim  = vShaderAnim.x;
    sysShaderAnimY = vShaderAnim.y;
    sysShaderAnimZ = vShaderAnim.z;
#else
    sysShaderAnim  = shaderAnim.v[0].x;
    sysShaderAnimY = shaderAnim.v[0].y;
    sysShaderAnimZ = shaderAnim.v[0].z;
#endif

    // 乱数をフラグメントシェーダに送る
    sysParticleRandom = sysRandomAttr;

    //--------------------------------------------------------------
    // 最終調整用コールバック
    //--------------------------------------------------------------
    EP_FinalAdjustmentVertexProcess();  // エミッタプラグインのコールバック
    FinalAdjustmentVertexProcess();     // カスタムシェーダのコールバック

    //--------------------------------------------------------------
    // αが0の場合、カメラの後ろに飛ばしてピクセルシェーダから弾く。
    //--------------------------------------------------------------
    if( farDistAlphaRate <= 0 || nearDistAlphaRate <= 0 )
    {
        DiscardVertex();
        return;
    }

#ifdef _TARGET_VK
    gl_Position.y = -gl_Position.y;     // For Vulkan.
    #ifdef _DEPTH_MODE_NEAR_IS_MINUS_W
    gl_Position.z = (gl_Position.z + gl_Position.w) / 2.0;
    #endif
#endif
}
#endif
