﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using EffectMaker.DataModel.DataModels;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// DataModelManager.
    /// </summary>
    public class DataModelManager
    {
        /// <summary>Post constructors for each data model type.</summary>
        private static Dictionary<Type, Action<DataModelBase>> dataModelPostConstructors =
            new Dictionary<Type, Action<DataModelBase>>();

        /// <summary>
        /// Set delegate action to be executed after constructor of the specified data model type.
        /// </summary>
        /// <param name="dataModelType">The data model type.</param>
        /// <param name="action">The action to be executed.</param>
        /// <returns>True on success.</returns>
        public static bool SetPostConstructForType(
            Type dataModelType,
            Action<DataModelBase> action)
        {
            dataModelPostConstructors[dataModelType] = action;
            return true;
        }

        /// <summary>
        /// Get delegate action to be executed after constructor of the specified data model type.
        /// </summary>
        /// <param name="dataModelType">The data model type.</param>
        /// <returns>The action to be executed, or null if no action is set.</returns>
        public static Action<DataModelBase> GetPostConstructForType(Type dataModelType)
        {
            Action<DataModelBase> postConstructor;
            if (dataModelPostConstructors.TryGetValue(dataModelType, out postConstructor) == false)
            {
                return null;
            }

            return postConstructor;
        }

        /// <summary>
        /// 指定したタイプのプロパティ名のデフォルト値を取得する.
        /// </summary>
        /// <param name="type">クラスタイプ.</param>
        /// <param name="propertyName">取得したい値のプロパティ名.</param>
        /// <param name="result">結果の値.</param>
        /// <returns>成功した場合trueを返す.</returns>
        public static bool GetDataModelDefaultValue(
            Type type,
            string propertyName,
            out object result)
        {
            PropertyInfo pi = type.GetProperty(propertyName);
            if (pi == null)
            {
                result = null;
                return false;
            }

            result = pi.GetValue(GetDefaultDataModel(type), null);
            return true;
        }

        /// <summary>
        /// 指定した名前のデータモデルを取得する.
        /// </summary>
        /// <param name="dataModelType">データモデルのクラス名.</param>
        /// <returns>成功した場合trueを返す.</returns>
        public static DataModelBase GetDefaultDataModel(Type dataModelType)
        {
            if (dataModelType != null && dataModelType.IsAbstract == false)
            {
                return (DataModelBase)Activator.CreateInstance(dataModelType);
            }

            return null;
        }
    }
}
