﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.BusinessLogic.Commands;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.HookCommand;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the WorkspaceData.
    /// </summary>
    public class WorkspaceViewModel : WorkspaceNodeViewModelBase<WorkspaceData>
    {
        /// <summary>
        /// ワークスペースのファイル名を保持します。
        /// </summary>
        private string workspaceFilePath = string.Empty;

        /// <summary>
        /// The multiple open mode.
        /// </summary>
        private List<EmitterSetViewModel> multipleOpenList = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public WorkspaceViewModel(HierarchyViewModel parent, WorkspaceData dataModel)
            : base(parent, dataModel)
        {
            var typeOrdered = Children as TypeOrderedObservableCollection<IHierarchyObject>;

            if (typeOrdered != null)
            {
                typeOrdered.TypeOrder = new Type[]
                {
                    typeof(EmitterSetViewModel),
                    typeof(ViewerViewModel)
                };
                typeOrdered.Sort();
            }

            foreach (EmitterSetData emitterSetData in dataModel.EmitterSetList)
            {
                var emitterSetViewModel = new EmitterSetViewModel(this, emitterSetData);
                this.AddEmitterSetViewModel(emitterSetViewModel);
            }

            // WorkspaceViewModelとViewerViewModelをUI上では同階層にする
            var viewerViewModel = new ViewerViewModel(this, dataModel.ViewerData);
            this.Parent.Children.Add(this);
            ((WorkspaceRootViewModel)this.Parent).ViewerViewModel = viewerViewModel;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// Gets a value indicating whether is any value modified.
        /// </summary>
        public bool IsAnyValueModified
        {
            get { return false; }
        }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get { return null; }
        }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// Get the emitter set count in the workspace.
        /// </summary>
        public int EmitterSetCount
        {
            get
            {
                return this.Children.Count(ch => ch is EmitterSetViewModel);
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether multiple open mode.
        /// </summary>
        public bool MultipleOpenMode
        {
            get
            {
                return this.multipleOpenList != null;
            }

            set
            {
                if (value && this.multipleOpenList == null)
                {
                    this.multipleOpenList = new List<EmitterSetViewModel>();
                }
                else if (!value && this.multipleOpenList != null)
                {
                    var children = this.Children as TypeOrderedObservableCollection<IHierarchyObject>;
                    if (children != null)
                    {
                        children.AddRange(this.multipleOpenList);
                    }

                    this.multipleOpenList = null;
                }
            }
        }

        /// <summary>
        /// ワークスペースのファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get { return this.workspaceFilePath; }
        }

        /// <summary>
        /// 全てのファイルを閉じます。
        /// </summary>
        /// <param name="isFatalException">例外発生時に呼び出された場合trueになります。</param>
        /// <returns>処理されないファイルがあったときfalseを返します。</returns>
        public bool OnCloseAll(bool isFatalException)
        {
            // Childrenをforeachで使うため、ループ中はChildrenを変更できなくなる
            // そのため一度removeEsetListにビューモデルの参照をコピーして後でまとめて削除する
            var removeEsetList = new List<EmitterSetViewModel>();

            // 子供のEmitterSetViewModelについて処理
            foreach (var emitterSetViewModel in this.Children.OfType<EmitterSetViewModel>())
            {
                removeEsetList.Add(emitterSetViewModel);

                // 変更のないエミッタセットはスルー
                if (emitterSetViewModel.ModificationFlagViewModel.IsAnyValueModified == false)
                {
                    continue;
                }

                // 変更の保存確認ダイアログを表示
                string fileName = emitterSetViewModel.FileName + IOConstants.EmitterSetFileExtension;

                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowSaveModificationDialog(fileName);

                if (result == DialogResult.Yes)
                {
                    // Yesのとき、ファイルを保存する
                    bool saveResult = this.SaveEmitterSetFile(emitterSetViewModel, isFatalException);

                    // ファイルを保存できなかったとき、falseを返す
                    if (saveResult == false && isFatalException == false)
                    {
                        return false;
                    }
                }
                else if (result == DialogResult.Cancel)
                {
                    // Cancelのとき、falseを返す
                    return false;
                }

                // Noのとき、ファイルを保存せず次のファイルを処理する
            }

            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // エミッタビューモデル＆データモデルを削除
            foreach (var eset in removeEsetList)
            {
                // 閉じる前にビューアに削除命令を転送しておく.
                ViewerMessageHelper.RequestDeleteEmitterSet(eset.DataModel);
                proxy.RemoveEmitterSet(eset.DataModel);

                // ビューモデルをRemoveで消さないとオートバックアップから削除されない
                this.Children.Remove(eset);

                // FileMonitorManagerによるファイル監視を解除するため、EmitterSetのリソースを解放する。
                // リソースの解放時に、ファイル監視も解除される。
                eset.Dispose();
            }

            // エミッタセット以外のビューモデルを削除
            this.Children.Clear();

            this.workspaceFilePath = string.Empty;

            // モデルを外す
            var removeModelList = new List<ModelViewModel>();
            var viewerVM = WorkspaceRootViewModel.Instance.GetChild<ViewerViewModel>();
            foreach (var modelVM in viewerVM.Children.OfType<ModelViewModel>())
            {
                removeModelList.Add(modelVM);
                modelVM.SendModelInfo(true);
            }

            // ビューモデルも抹殺
            foreach (var model in removeModelList)
            {
                model.Dispose();
            }

            viewerVM.Children.Clear();

            // すべてのファイルを処理した
            return true;
        }

        #region Function - ノードの新規作成

        /// <summary>
        /// エミッタセットを新規作成します。
        /// </summary>
        /// <param name="selection">プリセットとして選択するインデックス</param>
        public void OnCreateNewEmitterSet(int selection)
        {
            int index = -1;
            HierarchyViewModel viewModel = this.GetSelectedViewModel();
            if (viewModel is EmitterSetViewModel)
            {
                index = this.Children.IndexOf(viewModel);
            }

            // エミッタセットの名前リストを取得
            var esets = from ch in this.Children
                        where ch is EmitterSetViewModel
                        select (EmitterSetViewModel)ch;

            IEnumerable<string> emitterSetNames =
                esets.Select(eset => eset.GetDataModelValue<string>(() => eset.DataModel.Name));

            // デフォルトのエミッタセット名を作成
            string emitterSetName =
                DefaultNameGenerator.GenerateDefaultEmitterSetName(emitterSetNames);

            string inputedName;
            int selectedPreset = selection;
            var presetList = this.CreatePresetEmitterSetItems(false, false);

            // エミッタセット名入力ダイアログを表示
            DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowEmitterSetNameInputDialog(
                out inputedName,
                emitterSetName,
                emitterSetNames,
                presetList,
                ref selectedPreset);

            // OKを押されなかったとき処理をキャンセル
            if (result != DialogResult.OK)
            {
                return;
            }

            emitterSetName = inputedName;

            // エミッタ, プレビューの名前を作成
            string emitterName = DefaultNameGenerator.GenerateDefaultEmitterName(null);
            string previewName = DefaultNameGenerator.GenerateDefaultPreviewName(emitterSetName);

            if (selectedPreset == 0)
            {
                // コマンドを発行
                CommandManager.Execute(new CreateEmitterSetCommand(this, emitterSetName, emitterName, previewName, index));
            }
            else
            {
                // コマンドを発行
                CommandManager.Execute(new CreateEmitterSetCommand(this, emitterSetName, presetList[selectedPreset], index));
            }
        }

        /// <summary>
        /// エミッタ新規作成イベントを処理します。(エミッタは同一階層に作成される)
        /// 名前入力ダイアログを表示した後、エミッタ作成コマンドを発行します。
        /// </summary>
        /// <param name="selection">プリセットとして選択するインデックス</param>
        public void OnCreateNewEmitter(int selection)
        {
            int index = -1;
            HierarchyViewModel viewModel = this.GetSelectedViewModel();
            if (viewModel is EmitterViewModel)
            {
                var selectedViewModel = viewModel;
                viewModel = viewModel.Parent as HierarchyViewModel;
                index = viewModel.Children.IndexOf(selectedViewModel);
            }

            HierarchyViewModel target = null;

            // エミッタセットが選択されているとき、そのエミッタセットをターゲットに設定.
            {
                var emitterSetViewModel = viewModel as EmitterSetViewModel;

                if (emitterSetViewModel != null)
                {
                    target = emitterSetViewModel;
                }
            }

            // エミッタが選択されているとき、そのエミッタをターゲットに設定.
            {
                var emitterViewModel = viewModel as EmitterViewModel;

                if (emitterViewModel != null)
                {
                    target = emitterViewModel;
                }
            }

            // プレビューが選択されているとき、親のエミッタセットをターゲットに設定.
            {
                var previewViewModel = viewModel as PreviewViewModel;

                if (previewViewModel != null)
                {
                    target = previewViewModel.Parent as EmitterSetViewModel;
                }
            }

            // ターゲットが見つからないとき処理をキャンセル
            if (target == null)
            {
                return;
            }

            // 所属するエミッタセットを検索
            var esetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(viewModel);

            // デフォルトのエミッタ名を作成
            IEnumerable<string> emitterNameList = EnumerateEmitterName(esetViewModel);

            string name = DefaultNameGenerator.GenerateDefaultEmitterName(emitterNameList);

            string inputedName;

            // プリセットを提示
            int selectedPreset = selection;
            var presetList = this.CreatePresetEmitterSetItems(true, false);

            // エミッタ名入力ダイアログを表示
            DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowEmitterNameInputDialog(
                out inputedName, name, emitterNameList, presetList, ref selectedPreset);

            // OKを押されなかったとき処理をキャンセル
            if (result != DialogResult.OK)
            {
                return;
            }

            name = inputedName;

            if (selectedPreset == 0)
            {
                CommandManager.Execute(new CreateEmitterCommand(target, name, null, index));
            }
            else
            {
                CommandManager.Execute(new CreateEmitterCommand(target, name, presetList[selectedPreset].Value, index));
            }
        }

        /// <summary>
        /// 子エミッタ新規作成イベントを処理します。
        /// 名前入力ダイアログを表示した後、エミッタ作成コマンドを発行します。
        /// </summary>
        /// <param name="selection">プリセットとして選択するインデックス</param>
        public void OnCreateNewChildEmitter(int selection)
        {
            HierarchyViewModel viewModel = this.GetSelectedViewModel();

            HierarchyViewModel target = null;

            // エミッタセットが選択されているとき、そのエミッタセットをターゲットに設定.
            {
                var emitterSetViewModel = viewModel as EmitterSetViewModel;

                if (emitterSetViewModel != null)
                {
                    target = emitterSetViewModel;
                }
            }

            // エミッタが選択されているとき、そのエミッタをターゲットに設定.
            {
                var emitterViewModel = viewModel as EmitterViewModel;

                if (emitterViewModel != null)
                {
                    target = emitterViewModel;
                }
            }

            // プレビューが選択されているとき、親のエミッタセットをターゲットに設定.
            {
                var previewViewModel = viewModel as PreviewViewModel;

                if (previewViewModel != null)
                {
                    target = previewViewModel.Parent as EmitterSetViewModel;
                }
            }

            // ターゲットが見つからないとき処理をキャンセル
            if (target == null)
            {
                return;
            }

            // 所属するエミッタセットを検索
            var esetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(viewModel);

            // デフォルトのエミッタ名を作成
            IEnumerable<string> emitterNameList = EnumerateEmitterName(esetViewModel);

            string name = DefaultNameGenerator.GenerateDefaultEmitterName(emitterNameList);

            string inputedName;
            int selectedPreset = selection;
            var presetList = this.CreatePresetEmitterSetItems(true, false);

            // エミッタ名入力ダイアログを表示
            DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowEmitterNameInputDialog(
                out inputedName, name, emitterNameList, presetList, ref selectedPreset);

            // OKを押されなかったとき処理をキャンセル
            if (result != DialogResult.OK)
            {
                return;
            }

            name = inputedName;

            if (selectedPreset == 0)
            {
                CommandManager.Execute(new CreateEmitterCommand(target, name, null));
            }
            else
            {
                CommandManager.Execute(new CreateEmitterCommand(target, name, presetList[selectedPreset].Value));
            }
        }

        /// <summary>
        /// チャイルドエミッタの階層化を解除します。
        /// </summary>
        public void OnUnparentEmitter()
        {
            var emitter = this.GetSelectedViewModel() as EmitterViewModel;
            if (emitter == null || !(emitter.Parent is EmitterViewModel))
            {
                return;
            }

            CommandManager.Execute(new UnparentEmitterCommand(emitter));
        }

        /// <summary>
        /// プレビューを新規作成します。
        /// </summary>
        public void OnCreateNewPreview()
        {
            HierarchyViewModel viewModel = this.GetSelectedViewModel();
            int index = -1;

            var emitterSetViewModel = viewModel as EmitterSetViewModel;

            // プレビューが選択されているとき、親のエミッタセットを取得.
            var previewViewModel = viewModel as PreviewViewModel;

            if (previewViewModel != null)
            {
                emitterSetViewModel = (EmitterSetViewModel)previewViewModel.Parent;
                index = emitterSetViewModel.Children.IndexOf(previewViewModel);
            }

            // エミッタが選択されているとき、親のエミッタを取得.
            var emitterViewModel = viewModel as EmitterViewModel;

            while (emitterViewModel != null)
            {
                if (emitterViewModel.Parent is EmitterSetViewModel)
                {
                    emitterSetViewModel = (EmitterSetViewModel)emitterViewModel.Parent;
                    emitterViewModel = null;
                }
                else
                {
                    emitterViewModel = (EmitterViewModel)emitterViewModel.Parent;
                }
            }

            if (emitterSetViewModel != null)
            {
                // プレビューの名前リストを取得
                IEnumerable<string> previewNameList = null;
                var previewList = emitterSetViewModel.Proxy.PreviewList as IEnumerable<PreviewData>;

                if (previewList != null)
                {
                    previewNameList = previewList.Select(preview => preview.Name).ToList();
                }

                // デフォルトのプレビュー名を作成
                string defaultName = DefaultNameGenerator.GenerateDefaultPreviewName(
                    emitterSetViewModel.GetDataModelValue(() => emitterSetViewModel.DataModel.Name), previewNameList);

                string inputedName;

                // プレビュー名入力ダイアログを表示
                DialogResult result = WorkspaceRootViewModel.Instance
                    .Dialogs.ShowPreviewNameInputDialog(out inputedName, defaultName, previewNameList);

                // OKを押されなかったとき処理をキャンセル
                if (result != DialogResult.OK)
                {
                    return;
                }

                CommandManager.Execute(new CreatePreviewCommand(emitterSetViewModel, inputedName, index));
            }
        }

        #endregion

        #region Function - エミッタセット管理

        /// <summary>
        /// エミッタセットを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterSetName">エミッタセット名</param>
        /// <returns>エミッタセットビューモデルを返します。</returns>
        public EmitterSetViewModel CreateNewEmitterSet(string emitterSetName)
        {
            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return null;
            }

            // データモデルを作成.
            EmitterSetData emitterSetData = proxy.CreateEmitterSet(emitterSetName);
            if (emitterSetData == null)
            {
                return null;
            }

            // ビューモデルを作成.
            EmitterSetViewModel emitterSetViewModel = new EmitterSetViewModel(this, emitterSetData);

            return emitterSetViewModel;
        }

        /// <summary>
        /// エミッタセットを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddEmitterSet(EmitterSetViewModel emitterSetViewModel)
        {
            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // データモデルを追加.
            proxy.AddEmitterSet(emitterSetViewModel.DataModel);

            // ビューモデルを追加
            this.AddEmitterSetViewModel(emitterSetViewModel);

            return true;
        }

        /// <summary>
        /// エミッタセットビューモデルをチャイルドに追加します.
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        public void AddEmitterSetViewModel(EmitterSetViewModel emitterSetViewModel)
        {
            // ビューアノードをエフェクトノードから分離したので、挿入は不要になった
            if (this.MultipleOpenMode)
            {
                this.multipleOpenList.Add(emitterSetViewModel);
            }
            else
            {
                this.Children.Add(emitterSetViewModel);
            }
        }

        /// <summary>
        /// エミッタセットを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveEmitterSet(EmitterSetViewModel emitterSetViewModel)
        {
            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // エミッタセットに変更があるとき
            if (emitterSetViewModel.ModificationFlagViewModel.IsAnyValueModified)
            {
                // 変更保存ダイアログを表示
                string fileName = emitterSetViewModel.FileName + IOConstants.EmitterSetFileExtension;

                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowSaveModificationDialog(fileName);

                if (result == DialogResult.Yes)
                {
                    // Yesのとき、ファイルを保存する
                    bool saveResult = this.SaveEmitterSetFile(emitterSetViewModel, false);

                    // ファイルを保存できなかったとき、falseを返す
                    if (saveResult == false)
                    {
                        return false;
                    }
                }
                else if (result == DialogResult.Cancel)
                {
                    // Cancelのとき、falseを返す
                    return false;
                }

                // Noのとき、ファイルを保存せずにエミッタセットを削除する
            }

            // データモデルを削除
            proxy.RemoveEmitterSet(emitterSetViewModel.DataModel);

            // ビューモデルを削除
            this.Children.Remove(emitterSetViewModel);

            return true;
        }

        /// <summary>
        /// エミッタセットのインデックスを取得します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        public void GetEmitterSetIndex(
            EmitterSetViewModel emitterSetViewModel,
            out int viewModelIndex,
            out int dataModelIndex)
        {
            dataModelIndex = -1;
            viewModelIndex = -1;

            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return;
            }

            // データモデルのインデックスを取得
            dataModelIndex = this.Proxy.EmitterSetList.IndexOf(
                emitterSetViewModel.DataModel);

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(emitterSetViewModel);
        }

        /// <summary>
        /// エミッタセットを指定インデックスに追加します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertEmitterSet(
            EmitterSetViewModel emitterSetViewModel,
            int viewModelIndex,
            int dataModelIndex)
        {
            var proxy = this.Proxy as WorkspaceDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // データモデルを追加
            proxy.InsertChildDataModel(dataModelIndex, emitterSetViewModel.DataModel);

            // ビューモデルを追加
            this.Children.Insert(viewModelIndex, emitterSetViewModel);

            return true;
        }

        #endregion

        #region Function - ファイルの読み込み

        /// <summary>
        /// エミッタセットファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="addPreview">プレビューをエミッタセットに追加するかどうか</param>
        /// <returns>エミッタセットのビューモデルを返します。</returns>
        public LoadEmitterSetResultWithData LoadEmitterSetFile(string filePath, bool addPreview)
        {
            // TODO:WorkspaceRootViewModelの関数をWorkspaceViewModelに移動させる
            var workspaceRootViewModel = this.Parent as WorkspaceRootViewModel;

            // 拡張子抜きのファイル名を取得
            string esetName = Path.GetFileNameWithoutExtension(filePath);

            // 名前の長さが制限を超えているときエラーを表示する
            if (esetName.Length > IOConstants.MaxFileNameLength)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowOpenFileNameLengthError(filePath);
                return new LoadEmitterSetResultWithData(LoadEmitterSetResults.FileNotFound, null);
            }

            // ワークスペースに同名のエミッタセットがあるとき
            if (workspaceRootViewModel.GetEmitterSet(esetName) != null)
            {
                if (OptionStore.RuntimeOptions.IsCommandLineMode)
                {
                    // コマンドラインだったらログを吐いて強制終了
                    Logger.Log(LogLevels.Error, "The eset file of the same name ({0}) was found. Aborting convert.", esetName);
                    Environment.Exit(4);
                }
                else
                {
                    // エディタだったらエラーを表示
                    WorkspaceRootViewModel.Instance.Dialogs.ShowOpenFileNameCollisionError(filePath);
                    return new LoadEmitterSetResultWithData(LoadEmitterSetResults.FileOpenError, null);
                }
            }

            // 実際にロードするファイルのパスを取得
            // FE1からアップデートしたファイルを読み込むための例外的な処理で、
            // entityFilePathはデシリアライズにのみ使用する
            string entityFilePath = AncientEsetUpdater.Instance.GetFe2FilePath(filePath);

            if (string.IsNullOrEmpty(entityFilePath))
            {
                return new LoadEmitterSetResultWithData(LoadEmitterSetResults.FileNotFound, null);
            }

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // デシリアライズを実行
            var result = serializer.DeserializeFromFile(entityFilePath);

            // エラー処理
            if (result.IsSuccess == false)
            {
                if (result.IsVersionMismatch)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowVersionMismatchDialog();

                    return new LoadEmitterSetResultWithData(LoadEmitterSetResults.VersionMismatch, null);
                }

                return new LoadEmitterSetResultWithData(LoadEmitterSetResults.DeserializeError, null);
            }

            EmitterSetData emitterSetData = result.DataModel;

            // エミッタセットの名前を設定
            emitterSetData.Name = Path.GetFileNameWithoutExtension(filePath);

            // ユーザー定義データが置き換えられたとき、警告ダイアログを表示する
            if (result.AnyUserDefinitionDataReplaced)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowEmitterUserDataDefinitionReplaced(emitterSetData.Name);
            }

            // ビューモデルを作成
            var emitterSetViewModel = new EmitterSetViewModel(this, emitterSetData);

            // Do not update the modification flags for now.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;

            // エミッタセットのファイルパスを設定
            emitterSetViewModel.FilePath = PathUtility.GetDirectoryName(filePath);

            // ワークスペースにエミッタセットを追加
            this.DataModel.EmitterSetList.Add(emitterSetData);
            this.DataModel.AddChild(emitterSetData);

            // ここで通信が飛ぶのは無駄に重いのでいったんブロッキング
            using (new DisableBinaryConvertSection())
            {
                this.AddEmitterSetViewModel(emitterSetViewModel);
            }

            // アセットファイルのパスチェックと読み込みを実行
            AssetsManager.CheckAndLoadAssetsFile(emitterSetViewModel, filePath, esetName);

            // エミッタセットにプレビューを追加
            if (addPreview)
            {
                PreviewData previewData = new PreviewData();

                previewData.Name = DefaultNameGenerator.GenerateDefaultPreviewName(emitterSetViewModel.GetDataModelValue(() => emitterSetViewModel.DataModel.Name));

                PreviewViewModel previewViewModel = new PreviewViewModel(emitterSetViewModel, previewData);

                emitterSetViewModel.Proxy.AddPreview(previewData);
                emitterSetViewModel.AddPreviewViewModel(previewViewModel);
            }

            WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(emitterSetViewModel);

            // Restore the flag so the modification flag view models can start listen to the events.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

            // 初期値から変更されたタブの色付け
            emitterSetViewModel.ModificationFlagViewModel.UpdateDefaultValues();

            // 最近開いたファイルにesetを追加
            OptionStore.RootOptions.FileEvent.AddLastLoadedFile(filePath);

            // 描画パスをチェック
            emitterSetViewModel.VerifyDrawPathId(false);

            // エミッタの名前が64バイト以下かチェック。NULL終端があるので63文字制限。
            // 63文字を超えていても、ログでメッセージを表示するだけにする。
            foreach (EmitterViewModel emitter in emitterSetViewModel.Emitters)
            {
                if (emitter.Name.Length > 63)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, Resources.LogMessageTooLongEmitterName, emitterSetData.Name, emitter.Name);
                }
            }

            return new LoadEmitterSetResultWithData(LoadEmitterSetResults.Success, emitterSetViewModel);
        }

        /// <summary>
        /// プレビューファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="loadedEmitterSetList">プレビューと一緒に読み込んだエミッタセット</param>
        /// <param name="emitterSetViewModel">追加先のエミッタセットビューモデル</param>
        /// <param name="previewViewModel">ロード先のプレビュービューモデル</param>
        /// <returns>プレビューのビューモデルを返します。</returns>
        public PreviewViewModel LoadPreviewFile(
            string filePath,
            List<string> loadedEmitterSetList,
            EmitterSetViewModel emitterSetViewModel,
            PreviewViewModel previewViewModel)
        {
            if (!IOConstants.CheckRootElement(filePath, "PreviewData"))
            {
                // 通知して中断
                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnShowErrorDialogExecutable;
                if (exe != null)
                {
                    var retMsgs = new string[2];
                    retMsgs[0] = string.Format(Resources.ErrorInvalidPreviewFile, filePath);
                    retMsgs[1] = Resources.ErrorLoadPrevFileCaption;
                    exe.Execute(retMsgs);
                    Logger.Log("LogView", LogLevels.Error, retMsgs[0]);
                }

                return null;
            }

            // 拡張子抜きのファイル名を取得
            string prevName = Path.GetFileNameWithoutExtension(filePath);

            // 名前の長さが制限を超えているときエラーを表示
            if (prevName.Length > IOConstants.MaxFileNameLength)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowOpenFileNameLengthError(filePath);

                return null;
            }

            PreviewDataSerializer serializer = new PreviewDataSerializer();

            // デシリアライズを実行
            var result = serializer.DeserializeFromFile(filePath);

            // エラー処理
            if (result.IsSuccess == false)
            {
                if (result.IsVersionMismatch)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowVersionMismatchDialog();
                }

                return null;
            }

            PreviewData previewData = result.DataModel;

            // プレビューのフォルダパスを設定
            previewData.FilePath = PathUtility.GetDirectoryName(filePath);

            string esetPath = null;

            if (emitterSetViewModel == null && previewViewModel == null)
            {
                // エミッタセットを取得
                var loadEmitterSetResultWithData = this.GetEmitterSetForPreview(
                    previewData.EsetName,
                    previewData.Name,
                    previewData.FilePath,
                    out esetPath);
                emitterSetViewModel = loadEmitterSetResultWithData.ViewModel;

                if (loadEmitterSetResultWithData.Result == LoadEmitterSetResults.VersionMismatch)
                {
                    return null;
                }
            }

            // エミッタセットをファイルから読み込んだとき、リストに追加する。
            if (esetPath != null && loadedEmitterSetList != null)
            {
                loadedEmitterSetList.Add(esetPath);
            }

            // エミッタセットに読み込んだプレビューを追加
            if (previewViewModel == null)
            {
                // エミッタセットを取得できないとき、エラーメッセージを表示
                if (emitterSetViewModel == null)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowEsetFileNotFoundError(previewData.EsetName);
                    return null;
                }

                // エミッタセットにこれ以上プレビューが追加できないとき、エラーメッセージを表示
                if (emitterSetViewModel.GetChildren<PreviewViewModel>().Count() >= 32)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowPreviewCountMaximumDialog(emitterSetViewModel.FileName);
                    return null;
                }

                // 名前をファイル名ベースにするのをここで
                previewData.Name = prevName;

                // エミッタセットに既に同名のプレビューがあるとき、エラーメッセージを表示
                if (emitterSetViewModel.GetPreviewViewModel(previewData.Name) != null)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowPreviewFileAlreadyExistsDialog();

                    return null;
                }

                // エミッタセットにプレビューを追加
                emitterSetViewModel.Proxy.AddPreview(previewData);

                previewViewModel = new PreviewViewModel(emitterSetViewModel, previewData);
                emitterSetViewModel.AddPreviewViewModel(previewViewModel);
                previewViewModel.ModificationFlagViewModel.UpdateDefaultValues();
            }
            else
            {
                ExportableViewModel.IsPasting = true;

                // 名前を既存のプレビューと同一にする
                previewData.Name = previewViewModel.GetDataModelValue(() => previewViewModel.DataModel.Name);

                // プレビューデータをセットする
                using (new EnableDataModelCloneSetter())
                {
                    previewViewModel.DataModel.SetWithoutGuid(previewData);
                }

                // Update the child view models with the new data model.
                previewViewModel.UpdateChildViewModels();

                // Notify the UI that the values had changed.
                previewViewModel.FirePropertyChanges();

                // Update UI tool bar states.
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                ExportableViewModel.IsPasting = false;
            }

            return previewViewModel;
        }

        /// <summary>
        /// プレビュー用のエミッタセットを取得します。
        /// </summary>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <param name="prevName">プレビューの名前</param>
        /// <param name="prevPath">プレビューのパス</param>
        /// <param name="esetPath">エミッタセットファイルのパス（エミッタセットファイルを開かなかったときnull）</param>
        /// <returns>成功した場合、有効なエミッタセットビューモデルを返します。</returns>
        public LoadEmitterSetResultWithData GetEmitterSetForPreview(
            string esetName,
            string prevName,
            string prevPath,
            out string esetPath)
        {
            esetPath = null;

            // ワークスペース内のエミッタセットを検索
            foreach (var item in this.MultipleOpenMode ? this.multipleOpenList : this.Children.OfType<EmitterSetViewModel>())
            {
                // 名前が一致したとき、そのエミッタセットを返す
                if (item.GetDataModelValue(() => item.DataModel.Name) == esetName)
                {
                    return new LoadEmitterSetResultWithData(LoadEmitterSetResults.Success, item);
                }
            }

            // エミッタセットのフルパスを取得
            string esetFullPath = Path.Combine(prevPath, esetName + IOConstants.EmitterSetFileExtension);

            // エミッタセットのパスが不正なとき
            if (Path.IsPathRooted(esetFullPath) == false ||
                File.Exists(esetFullPath) == false)
            {
                // ログにエラーを表示
                Logger.Log("LogView", LogLevels.Error, Resources.LogMessageFileNotFound, esetFullPath);

                return new LoadEmitterSetResultWithData(LoadEmitterSetResults.FileNotFound, null);
            }

            // エミッタセットを読み込む
            LoadEmitterSetResultWithData loadEmitterSetResultWithData =
                this.LoadEmitterSetFile(esetFullPath, false);
            EmitterSetViewModel emitterSetViewModel = loadEmitterSetResultWithData.ViewModel;

            if (emitterSetViewModel != null)
            {
                esetPath = esetFullPath;
            }

            return loadEmitterSetResultWithData;
        }

        /// <summary>
        /// ワークスペースの読み込み
        /// </summary>
        /// <param name="filePath">ワークスペースファイルのパス</param>
        /// <returns>読み込みに成功したらtrue,中断か失敗したらfalse.</returns>
        public bool LoadWorkspaceFile(string filePath)
        {
            WorkspaceDataSerializer serializer = new WorkspaceDataSerializer();

            var result = serializer.DeserializeFromFile(filePath);

            // エラー処理
            if (result.IsSuccess == false)
            {
                if (result.IsVersionMismatch)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowVersionMismatchDialog();
                }

                return false;
            }

            WorkspaceStorageData workspaceData = result.DataModel;

            // DataModelのコンストラクタで5つ追加されてるので、その分を消しておく
            workspaceData.ViewerData.ModelList.RemoveRange(0, 5);
            workspaceData.ViewerData.Children.RemoveRange(0, 5);

            using (new DisableBinaryConvertSection())
            {
                // 現在のViewerViewModelを取得
                var tmpVM = this.Parent.Children.OfType<ViewerViewModel>().FirstOrDefault();
                if (tmpVM == null)
                {
                    return false;
                }

                // いったんViewerViewModelも破棄
                this.Parent.Children.Remove(tmpVM);
                tmpVM.Dispose();

                var workingPath = Path.GetDirectoryName(filePath);

                // ビューアのファイルパスを解決
                {
                    // 背景画像パス
                    var orgPath = workspaceData.ViewerData.ViewerBackgroundData.ImageFilePath;
                    var absPath = PathUtility.ToAbsolutePath(orgPath, workingPath);
                    workspaceData.ViewerData.ViewerBackgroundData.ImageFilePath = absPath;

                    // PTCLバイナリパス
                    orgPath = workspaceData.ViewerData.ViewerBasicData.PtclData.FilePath;
                    absPath = PathUtility.ToAbsolutePath(orgPath, workingPath);
                    workspaceData.ViewerData.ViewerBasicData.PtclData.FilePath = absPath;
                }

                // モデルデータに記載されたパスを解決
                foreach (var modelData in workspaceData.ViewerData.ModelList)
                {
                    // モデルデータ
                    var orgPath = modelData.ModelBasicSettingData.ModelBasicModelData.ModelFilePath;
                    var absPath = PathUtility.ToAbsolutePath(orgPath, workingPath);
                    modelData.ModelBasicSettingData.ModelBasicModelData.ModelFilePath = absPath;

                    // アニメーションフォルダ
                    orgPath = modelData.ModelBasicSettingData.ModelBasicAnimationData.AnimationFolderPath;
                    absPath = PathUtility.ToAbsolutePath(orgPath, workingPath);
                    modelData.ModelBasicSettingData.ModelBasicAnimationData.AnimationFolderPath = absPath;

                    // アニメーションファイル
                    orgPath = modelData.ModelBasicSettingData.ModelBasicAnimationData.AnimationFilePath;
                    absPath = PathUtility.ToAbsolutePath(orgPath, workingPath);
                    modelData.ModelBasicSettingData.ModelBasicAnimationData.AnimationFilePath = absPath;
                }

                // ViewerData以下を差し替え
                this.DataModel.ViewerData.Set(workspaceData.ViewerData);

                // ModelInfoManagerを初期化
                ModelInfoManager.Initialize(this.DataModel.ViewerData, SynchronizationContext.Current);

                // ViewerViewModel以下を再生成
                var viewerVM = new ViewerViewModel(this, this.DataModel.ViewerData);
                ((WorkspaceRootViewModel)this.Parent).ViewerViewModel = viewerVM;
                viewerVM.UpdateChildViewModels();

                // デフォルト値を更新
                WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(viewerVM);

                // データ変更フラグをクリア。
                viewerVM.ModificationFlagViewModel.ClearModificationFlags();
                viewerVM.ModificationFlagViewModel.ClearChildModificationFlags();

                // 編集済みタブカラーの更新
                viewerVM.ModificationFlagViewModel.UpdateDefaultValues();

                // エミッタセットの読み込み
                LoadEmitterSetCommand.SuppressSelectionNode = true;
                foreach (var eset in workspaceData.EmitterSetList)
                {
                    // ワークスペースに記述されたパスから環境変数を展開し、
                    // 相対パスだった時はワークスペースからの絶対パスに変換して取得します。
                    var esetPathInWorkspace = PathUtility.ToAbsolutePath(eset.FilePath, workingPath);
                    var esetFilePath = Path.Combine(
                        esetPathInWorkspace,
                        eset.Name + IOConstants.EmitterSetFileExtension);

                    if (!File.Exists(esetFilePath))
                    {
                        WorkspaceRootViewModel.Instance.Dialogs.ShowEsetFileNotFoundError(esetFilePath);
                        Logger.Log(
                            "LogView",
                            LogLevels.Error,
                            Resources.LogMessageFileNotFound,
                            esetFilePath);
                        continue;
                    }

                    // 読み込み
                    CommandManager.Execute(new LoadEmitterSetCommand(this, esetFilePath, false));

                    string dummy;
                    var loadEmitterSetResultWithData =
                        this.GetEmitterSetForPreview(eset.Name, string.Empty, string.Empty, out dummy);
                    var esetVM = loadEmitterSetResultWithData.ViewModel;
                    if (esetVM == null)
                    {
                        continue;
                    }

                    foreach (var prev in eset.PreviewList)
                    {
                        // エミッタセットにプレビューを追加
                        prev.PreviewData.Name = prev.Name;
                        prev.PreviewData.FilePath = Path.GetDirectoryName(filePath);
                        esetVM.Proxy.AddPreview(prev.PreviewData);
                        var prevVM = new PreviewViewModel(esetVM, prev.PreviewData);
                        esetVM.AddPreviewViewModel(prevVM);
                        prevVM.ModificationFlagViewModel.UpdateDefaultValues();
                    }
                }

                LoadEmitterSetCommand.SuppressSelectionNode = false;

                // ワークスペース名を設定
                this.workspaceFilePath = filePath;
            }

            using (new ForceRefreshBinary())
            {
                // Send all the emitter sets to the viewer.
                ViewerMessageHelper.SendEmitterSets(
                    this.DataModel.EmitterSetList);

                // Send all the models to the viewer.
                ViewerMessageHelper.SendModels(
                    this.DataModel.ViewerData.ModelList);

                // Send Viewer data to the viewer.
                ViewerMessageHelper.SendViewer(
                    this.DataModel.ViewerData);
            }

            return true;
        }

        #endregion

        #region Function - ファイルの保存

        /// <summary>
        /// エミッタセットファイル保存処理のうち、アセットパスの問い合わせやファイル名の確定だけを行い、リストに格納します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="fileList">対象ファイルリスト</param>
        /// <param name="actionList">実処理のアクションリスト</param>
        /// <returns>格納が完了した時はtrue, 途中でキャンセルがあった場合はfalseを返します。</returns>
        public bool GetListAndActionForSaveEmitterSet(
            EmitterSetViewModel emitterSetViewModel,
            List<string> fileList,
            List<Action> actionList)
        {
            // テクスチャパスの書き換え等によりランタイムが落ちることがあるので
            // 一時的にメッセージをブロックする
            using (new MessageDropSection())
            {
                // 保存前のファイルパスを取得
                string oldFileName = emitterSetViewModel.FileName + IOConstants.EmitterSetFileExtension;
                string oldFilePath = null;

                if (string.IsNullOrEmpty(emitterSetViewModel.FilePath) == false)
                {
                    oldFilePath = Path.Combine(emitterSetViewModel.FilePath, oldFileName);
                }

                // アセットリストを取得
                var assetInfo = AssetsManager.GetAssetInfoRecursive(emitterSetViewModel);

                // テクスチャのパスが正しく登録されているかチェック
                DialogResult resTexturesAssigned = AssetsManager.AreTexturesAssigned(
                    oldFilePath,
                    emitterSetViewModel.FileName,
                    assetInfo,
                    true);
                if (resTexturesAssigned == DialogResult.No)
                {
                    // 「いいえ」を選択したときは処理中のエミッタセットを保存リストに登録せず、
                    // 残りのエミッタセットの処理を引き続き行う
                    return true;
                }
                else if (resTexturesAssigned == DialogResult.Cancel)
                {
                    // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                    return false;
                }

                // 描画パスをチェック
                bool drawPathVerified = emitterSetViewModel.VerifyDrawPathId(true);
                if (drawPathVerified == false)
                {
                    return false;
                }

                // 終了処理タイプをチェック
                DialogResult resFadeTypeCheck = emitterSetViewModel.CheckEndingType(true);
                if (resFadeTypeCheck == DialogResult.No)
                {
                    // 「いいえ」を選択したときは処理中のエミッタセットを保存リストに登録せず、
                    // 残りのエミッタセットの処理を引き続き行う
                    return true;
                }
                else if (resFadeTypeCheck == DialogResult.Cancel)
                {
                    // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                    return false;
                }

                string filePath;

                // 新規作成したエミッタセットか、別名で保存を選択したときファイル保存ダイアログを表示
                if (string.IsNullOrEmpty(emitterSetViewModel.FilePath))
                {
                    // ファイル保存ダイアログを表示
                    filePath = WorkspaceRootViewModel.Instance.Dialogs.ShowSaveEmitterSetDialog(oldFileName);

                    // ファイルが選択されなかったとき、保存処理をキャンセル
                    if (string.IsNullOrEmpty(filePath))
                    {
                        return false;
                    }
                }
                else
                {
                    // 上書き保存のため、同じファイル名にする
                    filePath = oldFilePath;
                }

                // アセットの参照チェック
                var assetCopyPaths = new List<FileCopyInfo>();

                // パスを設定するアクションのリスト
                var pathSettingActs = new List<Action>();

                AssetsManager.VerifyAssetsLocation(emitterSetViewModel, filePath, assetCopyPaths, pathSettingActs);

                // アセットのパス設定アクション
                foreach (var action in pathSettingActs)
                {
                    action();
                }

                // コマンド実行対象リスト追加
                fileList.Add(filePath);
                fileList.AddRange(assetCopyPaths.Select(x => x.DstPath));

                // コピーと保存処理をアクションとして追加
                actionList.Add(() =>
                {
                    // アセットのコピー
                    foreach (var pair in assetCopyPaths)
                    {
                        IOConstants.AdjustPathCase(pair);
                        pair.DoCopy();
                    }

                    // 保存実処理
                    SaveEmitterSetFileCore(emitterSetViewModel, filePath, true);
                });
            }

            return true;
        }

        /// <summary>
        /// エミッタセットファイルを保存します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="showNameInputDialog">名前入力ダイアログの表示/非表示</param>
        /// <returns>ファイルの保存が成功したときはtrue, それ以外はfalseを返します。</returns>
        public bool SaveEmitterSetFile(EmitterSetViewModel emitterSetViewModel, bool showNameInputDialog)
        {
            string filePath;

            // アセットのパス設定アクションのリスト
            var pathSettingActs = new List<Action>();

            // テクスチャパスの書き換え等によりランタイムが落ちることがあるので
            // 一時的にメッセージをブロックする
            using (new MessageDropSection())
            {
                // 保存前のファイルパスを取得
                string oldFileName = emitterSetViewModel.FileName + IOConstants.EmitterSetFileExtension;
                string oldFilePath = null;

                if (string.IsNullOrEmpty(emitterSetViewModel.FilePath) == false)
                {
                    oldFilePath = Path.Combine(emitterSetViewModel.FilePath, oldFileName);
                }

                // アセットリストを取得
                var assetInfo = AssetsManager.GetAssetInfoRecursive(emitterSetViewModel);

                // テクスチャのパスが正しく登録されているかチェック
                DialogResult resTexturesAssigned = AssetsManager.AreTexturesAssigned(
                    oldFilePath, emitterSetViewModel.FileName, assetInfo, false);
                if (resTexturesAssigned == DialogResult.No)
                {
                    return false;
                }

                // 描画パスをチェック
                bool drawPathVerified = emitterSetViewModel.VerifyDrawPathId(true);
                if (drawPathVerified == false)
                {
                    return false;
                }

                // 終了処理タイプをチェック
                DialogResult resFadeTypeCheck = emitterSetViewModel.CheckEndingType(false);
                if (resFadeTypeCheck == DialogResult.No)
                {
                    return false;
                }

                // 新規作成したエミッタセットか、別名で保存を選択したときファイル保存ダイアログを表示
                if (showNameInputDialog || string.IsNullOrEmpty(emitterSetViewModel.FilePath))
                {
                    // ファイル保存ダイアログを表示
                    filePath = WorkspaceRootViewModel.Instance.Dialogs.ShowSaveEmitterSetDialog(oldFileName, emitterSetViewModel.FilePath);

                    // ファイルが選択されなかったとき、保存処理をキャンセル
                    if (string.IsNullOrEmpty(filePath))
                    {
                        return false;
                    }
                }
                else
                {
                    // 上書き保存のため、同じファイル名にする
                    filePath = oldFilePath;
                }

                // アセットの参照チェック
                var assetCopyPaths = new List<FileCopyInfo>();

                AssetsManager.VerifyAssetsLocation(emitterSetViewModel, filePath, assetCopyPaths, pathSettingActs);

                // コマンド実行対象リスト作成
                var targetList = new List<string> { filePath };
                targetList.AddRange(assetCopyPaths.Select(x => x.DstPath));

                // 保存前コマンド
                if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PreSave, targetList))
                {
                    return false;
                }

                // アセットのコピー
                foreach (var pair in assetCopyPaths)
                {
                    pair.DoCopy();
                }

                // パスの変更中は逐次ビューアへ送信するとデータの整合性が取れないため
                // ビューアへの送信を一時的にブロックする
                using (new DisableBinaryConvertSection())
                {
                    // アセットのパス設定アクション
                    foreach (var action in pathSettingActs)
                    {
                        action();
                    }
                }

                // 実保存処理
                SaveEmitterSetFileCore(emitterSetViewModel, filePath, true);

                // 保存後コマンド
                if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PostSave, targetList))
                {
                    return false;
                }
            }

            // パスの変更があった場合にはリネームに対応するためビューアに送信する
            if (pathSettingActs.Any())
            {
                ViewerMessageHelper.SendEmitterSet(emitterSetViewModel.DataModel);
            }

            // 最近開いたファイルにesetを追加
            OptionStore.RootOptions.FileEvent.AddLastLoadedFile(filePath);

            return true;
        }

        /// <summary>
        /// 編集中のエミッタセットのバックアップを保存します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="filePath">バックアップファイルのファイルパス</param>
        /// <returns>ファイルの保存が成功したときはtrue, それ以外はfalseを返します。</returns>
        public bool BackupEmitterSetFile(EmitterSetViewModel emitterSetViewModel, string filePath)
        {
            EmitterSetData cloneData;

            // エミッタセットデータのコピーを作成
            using (new EnableDataModelCloneSetter())
            {
                cloneData = (EmitterSetData)emitterSetViewModel.DataModel.Clone();
            }

            // プレビューを削除
            cloneData.PreviewList.Clear();

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // シリアライズを実行
            serializer.SerializeToFile(cloneData, filePath);

            return true;
        }

        /// <summary>
        /// プレビューファイルを保存します。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <param name="showNameInputDialog">名前入力ダイアログの表示/非表示</param>
        /// <returns>ファイルの保存が成功したときはtrue, それ以外はfalseを返します。</returns>
        public bool SavePreviewFile(
            PreviewViewModel previewViewModel,
            bool showNameInputDialog)
        {
            // 保存前のファイルパスを取得
            string oldFileName = previewViewModel.FileName + IOConstants.PreviewFileExtension;
            string oldFilePath = null;

            if (string.IsNullOrEmpty(previewViewModel.FilePath) == false)
            {
                oldFilePath = Path.Combine(previewViewModel.FilePath, oldFileName);
            }

            string filePath;

            // 新規作成したプレビューか、別名で保存を選択したときファイル保存ダイアログを表示
            if (showNameInputDialog || string.IsNullOrEmpty(previewViewModel.FilePath))
            {
                // ファイル保存ダイアログを表示
                filePath = WorkspaceRootViewModel.Instance.Dialogs.ShowSavePreviewDialog(oldFileName);

                // ファイルが選択されなかったとき、保存処理をキャンセル
                if (string.IsNullOrEmpty(filePath))
                {
                    return false;
                }
            }
            else
            {
                // 上書き保存のため、同じファイル名にする
                filePath = oldFilePath;
            }

            SavePreviewFileCore(previewViewModel, filePath, true);

            return true;
        }

        /// <summary>
        /// ワークスペースファイルを保存します。
        /// </summary>
        /// <param name="showNameInputDialog">保存ダイアログを開くならtrue,上書きならfalse.</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        public bool SaveWorkspaceFile(bool showNameInputDialog)
        {
            // 保存パスが決まっていないエミッタセットがいる場合には警告を出して止める。
            if (this.DataModel.EmitterSetList.Count > 0)
            {
                if (this.DataModel.EmitterSetList.Any(emitterSetData => string.IsNullOrEmpty(emitterSetData.FilePath)))
                {
                    var errMsg = new string[2];
                    errMsg[0] = Resources.WarningSaveWorkspaceByNoNameEmitterSet;
                    errMsg[1] = Resources.WarningCaption;
                    WorkspaceRootViewModel.Instance.Dialogs.OnShowWarningDialogExecutable.Execute(errMsg);
                    return false;
                }
            }

            if (string.IsNullOrEmpty(this.workspaceFilePath) || showNameInputDialog)
            {
                // ファイル保存ダイアログを表示
                var filePath = WorkspaceRootViewModel.Instance.Dialogs.ShowSaveWorkspaceDialog("default1");

                // ファイルが選択されなかったとき、保存処理をキャンセル
                if (string.IsNullOrEmpty(filePath))
                {
                    return false;
                }

                this.workspaceFilePath = filePath;
            }

            WorkspaceDataSerializer.SerializeResult result;

            {
                // プレビューリストを名前とセットのインスタンスリストに変換
                Func<List<PreviewData>, List<PreviewInstanceData>> conv =
                    previewData => previewData.Select(data => new PreviewInstanceData
                    {
                        Name = string.Copy(data.Name), PreviewData = data
                    }).ToList();

                // 名前とファイルパスとプレビューリストのみを抽出
                var esetRefList = this.DataModel.EmitterSetList.Select(emitterSetData => new EmitterSetReferenceData()
                {
                    Name = emitterSetData.Name,
                    FilePath = emitterSetData.FilePath,
                    PreviewList = conv(emitterSetData.PreviewList),
                }).ToList();

                // 記録用データモデルに格納
                var workspaceStorage = new WorkspaceStorageData
                {
                    EmitterSetList = esetRefList,
                    ViewerData = this.DataModel.ViewerData,
                };

                WorkspaceDataSerializer serializer = new WorkspaceDataSerializer();

                // シリアライズ
                result = serializer.SerializeToFile(workspaceStorage, this.workspaceFilePath);
            }

            if (result.IsSuccess == false)
            {
                var retMsgs = new string[2];
                retMsgs[0] = string.Format(Resources.ErrorSaveWorkspace, this.workspaceFilePath);
                retMsgs[1] = Resources.ErrorSaveCaption;

                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnShowErrorDialogExecutable;
                if (exe != null)
                {
                    exe.Execute(retMsgs);
                }

                Logger.Log("LogView", LogLevels.Error, retMsgs[0]);

                return false;
            }

            // ViewerViewModelを取得
            var viewerVM = this.Parent.Children.OfType<ViewerViewModel>().First();

            // デフォルト値を更新
            WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(viewerVM);

            // データ変更フラグをクリア。
            viewerVM.ModificationFlagViewModel.ClearModificationFlags();
            viewerVM.ModificationFlagViewModel.ClearChildModificationFlags();

            return true;
        }

        /// <summary>
        /// エクスポートの実行
        /// </summary>
        /// <param name="parameter">使わないパラメータです</param>
        public void ExportAll(object parameter)
        {
            var fileList = new List<string>();
            var esetDic = new Dictionary<string, EmitterSetViewModel>();
            var prevDic = new Dictionary<string, PreviewViewModel>();

            // 描画パスダイログの表示情報をリセット
            WorkspaceRootViewModel.Instance.Dialogs.ResetInvalidDrawPathDialogInfo();

            // 全エミッタセットについて関連するファイルリストを取得する
            foreach (var emitterSet in this.Children.OfType<EmitterSetViewModel>())
            {
                // 保存前のファイルパスを取得
                string oldFileName = emitterSet.FileName + IOConstants.EmitterSetFileExtension;
                string oldFilePath = null;

                if (string.IsNullOrEmpty(emitterSet.FilePath) == false)
                {
                    oldFilePath = Path.Combine(emitterSet.FilePath, oldFileName);
                }

                // アセットリストを取得
                var assetInfo = AssetsManager.GetAssetInfoRecursive(emitterSet);

                // テクスチャのパスが正しく登録されているかチェック
                DialogResult resTexturesAssigned = AssetsManager.AreTexturesAssigned(
                    oldFilePath,
                    emitterSet.FileName,
                    assetInfo,
                    true);
                if (resTexturesAssigned == DialogResult.No)
                {
                    // 「いいえ」を選択したときは処理中のエミッタセットを保存リストに登録せず、
                    // 残りのエミッタセットの処理を引き続き行う
                    continue;
                }
                else if (resTexturesAssigned == DialogResult.Cancel)
                {
                    // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                    return;
                }

                // 終了処理タイプをチェック
                DialogResult resFadeTypeCheck = emitterSet.CheckEndingType(true);
                if (resFadeTypeCheck == DialogResult.No)
                {
                    // 「いいえ」を選択したときは処理中のエミッタセットを保存リストに登録せず、
                    // 残りのエミッタセットの処理を引き続き行う
                    continue;
                }
                else if (resFadeTypeCheck == DialogResult.Cancel)
                {
                    // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                    return;
                }

                fileList.Add(emitterSet.FileName + IOConstants.EmitterSetFileExtension);
                esetDic[emitterSet.FileName + IOConstants.EmitterSetFileExtension] = emitterSet;
                foreach (var preview in emitterSet.GetChildren<PreviewViewModel>())
                {
                    fileList.Add(preview.FileName + IOConstants.PreviewFileExtension);
                    prevDic[preview.FileName + IOConstants.PreviewFileExtension] = preview;
                }

                foreach (var asset in assetInfo)
                {
                    if (string.IsNullOrEmpty(asset.FilePath))
                    {
                        continue;
                    }

                    if (asset.Type == AssetTypes.EmitterTexture0 ||
                        asset.Type == AssetTypes.EmitterTexture1 ||
                        asset.Type == AssetTypes.EmitterTexture2)
                    {
                        fileList.Add(asset.FilePath);
                    }
                    else if (asset.Type == AssetTypes.EmitterVolumePrimitive ||
                        asset.Type == AssetTypes.EmitterParticlePrimitive)
                    {
                        fileList.Add(asset.FilePath);
                    }
                    else if (asset.Type == AssetTypes.EmitterCombinerShader)
                    {
                        fileList.Add(asset.FilePath);
                    }
                }
            }

            if (fileList.Count == 0)
            {
                return;
            }

            // 0:eset, 1:prev, 2:Asset(ftxb&fmdb)
            var result = WorkspaceRootViewModel.Instance.Dialogs.ShowExportDilaog(fileList)
                as List<List<FileCopyInfo>>;

            if (result != null)
            {
                var targetList = result[0].Select(pair => pair.DstPath).ToList();
                targetList.AddRange(result[1].Select(pair => pair.DstPath));
                targetList.AddRange(result[2].Select(pair => pair.DstPath));

                using (new MessageDropSection())
                {
                    if (!WorkspaceRootViewModel.CallHookCommand(HookCommand.PreSave, targetList))
                    {
                        return;
                    }

                    // esetの保存
                    foreach (var pair in result[0])
                    {
                        SaveEmitterSetFileCore(esetDic[pair.SrcPath], pair.DstPath, false);
                    }

                    // アセットのコピー
                    foreach (var pair in result[2])
                    {
                        IOConstants.AdjustPathCase(pair);
                        pair.DoCopy();
                    }

                    WorkspaceRootViewModel.CallHookCommand(HookCommand.PostSave, targetList);

                    // プレビューはフックの対象外なので外で普通に
                    foreach (var pair in result[1])
                    {
                        SavePreviewFileCore(prevDic[pair.SrcPath], pair.DstPath, false);
                    }
                }
            }
        }

        #endregion

        /// <summary>
        /// プリセットの選択項目を更新します.
        /// </summary>
        /// <param name="targetIsEmitter">対象がエミッタならtrue,エミッタセットならfalse.</param>
        /// <param name="needSeparator">セパレータが必要ならtrue</param>
        /// <returns>列挙したプリセットのリスト</returns>
        public List<PresetItem> CreatePresetEmitterSetItems(bool targetIsEmitter, bool needSeparator = true)
        {
            List<PresetItem> result = new List<PresetItem>();

            // 先頭に「プリセット未選択」を挿入する
            var headerItem = new PresetItem(Properties.Resources.PresetEmpty, null, false, true);
            result.Add(headerItem);

            string pathUser = OptionStore.RootOptions.FileEvent.PresetFolderPath;
            var itemsUser = CreatePresetEmitterSetItems(targetIsEmitter, pathUser, PresetSettingTypes.User);

            string pathProj = OptionStore.ProjectConfig.FileEventProject.PresetFolderPath;
            var itemsProj = CreatePresetEmitterSetItems(targetIsEmitter, pathProj, PresetSettingTypes.Project);

            result.AddRange(itemsUser);

            // セパレータの挿入
            if (needSeparator && itemsUser.Count > 0 && itemsProj.Count > 0)
            {
                PresetItem separator = new PresetItem(null, null, true, settingType: PresetSettingTypes.None);
                result.Add(separator);
            }

            result.AddRange(itemsProj);

            return result;
        }

        /// <summary>
        /// プリセットの選択項目を更新します.
        /// </summary>
        /// <param name="targetIsEmitter">対象がエミッタならtrue,エミッタセットならfalse.</param>
        /// <param name="path">プリセットのパス</param>
        /// <param name="settingType">設定タイプ</param>
        /// <returns>列挙したプリセットのリスト</returns>
        public List<PresetItem> CreatePresetEmitterSetItems(bool targetIsEmitter, string path, PresetSettingTypes settingType)
        {
            const string Filter = "*" + IOConstants.EmitterSetFileExtension;

            var esetFileItems = new List<PresetItem>();

            // 指定されたフォルダパスをチェック
            if (string.IsNullOrEmpty(path) || Directory.Exists(path) == false)
            {
                return esetFileItems;
            }

            // プリセットファイルリストを取得
            DirectoryInfo rootDirInfo = new DirectoryInfo(path);
            foreach (DirectoryInfo dirInfo in PathUtility.EnumerateSubFolders(rootDirInfo))
            {
                foreach (FileInfo fileInfo in dirInfo.EnumerateFiles(Filter))
                {
                    if (targetIsEmitter)
                    {
                        var emitterNameList = this.EnumerateEmitterFromFileFast(fileInfo.FullName);
                        foreach (var emitterName in emitterNameList)
                        {
                            esetFileItems.Add(new PresetItem(
                                Path.GetFileNameWithoutExtension(fileInfo.Name) + " / " + emitterName,
                                new KeyValuePair<string, string>(fileInfo.FullName, emitterName),
                                settingType: settingType));

                        }
                    }
                    else
                    {
                        esetFileItems.Add(new PresetItem(
                            Path.GetFileNameWithoutExtension(fileInfo.Name), fileInfo.FullName, settingType: settingType));
                    }
                }
            }

            return esetFileItems;
        }

        /// <summary>
        /// esetファイルからなるはやでエミッタ名を列挙する
        /// </summary>
        /// <param name="filepath">esetファイルのパス</param>
        /// <returns>エミッタ名リスト</returns>
        public List<string> EnumerateEmitterFromFileFast(string filepath)
        {
            var retList = new List<string>();

            using (var fs = new FileStream(filepath, FileMode.Open, FileAccess.Read))
            {
                var reader = XmlReader.Create(
                    fs,
                    new XmlReaderSettings
                    {
                        IgnoreComments = true,
                        IgnoreWhitespace = true
                    });

                var inEmitterData = false;
                var inName = false;

                while (reader.Read())
                {
                    if (reader.IsEmptyElement)
                    {
                        continue;
                    }

                    var name = reader.Name;

                    switch (name)
                    {
                        case "EmitterData":
                            switch (reader.NodeType)
                            {
                                case XmlNodeType.Element:
                                    inEmitterData = true;
                                    break;
                                case XmlNodeType.EndElement:
                                    inEmitterData = false;
                                    break;
                            }

                            break;

                        case "Name":
                            switch (reader.NodeType)
                            {
                                case XmlNodeType.Element:
                                    inName = true;
                                    break;
                                case XmlNodeType.EndElement:
                                    inName = false;
                                    break;
                            }

                            break;
                    }

                    if (inEmitterData)
                    {
                        if (inName && !string.IsNullOrEmpty(reader.Value))
                        {
                            retList.Add(reader.Value);
                        }
                    }
                }
            }

            return retList;
        }

        /// <summary>
        /// プリセットを適用したエミッタセットを生成します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <returns>エミッタセットのビューモデルを返します。</returns>
        public EmitterSetViewModel AdaptPresetForEmitterSet(string filePath, string esetName)
        {
            var workspaceRootViewModel = this.Parent as WorkspaceRootViewModel;

            // ワークスペースに同名のエミッタセットがあるときエラーを表示する
            if (workspaceRootViewModel.GetEmitterSet(esetName) != null)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowOpenFileNameCollisionError(filePath);
                return null;
            }

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // デシリアライズを実行
            var result = serializer.DeserializeFromFile(filePath);

            // エラー処理
            if (result.IsSuccess == false)
            {
                if (result.IsVersionMismatch)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowVersionMismatchDialog();
                }

                return null;
            }

            EmitterSetData emitterSetData = result.DataModel;

            // ビューモデルを作成
            var emitterSetViewModel = new EmitterSetViewModel(this, emitterSetData);

            // エミッタセットの名前を設定
            emitterSetViewModel.SetName(esetName);

            // エミッタセットのファイルパスを設定(アセットロケート用)
            emitterSetViewModel.FilePath = PathUtility.GetDirectoryName(filePath);

            // アセットファイルのパスチェックと読み込みを実行
            AssetsManager.CheckAndLoadAssetsFile(emitterSetViewModel, filePath, esetName);

            // エミッタセットのファイルパスを初期化
            emitterSetViewModel.FilePath = string.Empty;

            // 初期値から変更されたタブの色付け
            emitterSetViewModel.ModificationFlagViewModel.UpdateDefaultValues();

            return emitterSetViewModel;
        }

        /// <summary>
        /// プリセットを適用したエミッタを取得します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="emitterName">エミッタの名前</param>
        /// <returns>エミッタのデータモデルを返します。</returns>
        public EmitterData AdaptPresetForEmitter(string filePath, string emitterName)
        {
            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // デシリアライズを実行
            var result = serializer.DeserializeFromFile(filePath);

            // エラー処理
            if (result.IsSuccess == false)
            {
                if (result.IsVersionMismatch)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowVersionMismatchDialog();
                }

                return null;
            }

            EmitterSetData presetEset = result.DataModel;

            var presetEmitter = presetEset.AllChildEmitters.FirstOrDefault(x => x.Name == emitterName);
            if (presetEmitter == null)
            {
                return null;
            }

            return presetEmitter;
        }

        /// <summary>
        /// Creates a WorkspaceDataProxy instance.
        /// </summary>
        /// <param name="dataModel">The data model to associate with the DataModelProxy.</param>
        /// <returns>Returns a WorkspaceDataProxy instance.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new WorkspaceDataProxy(dataModel);
        }

        /// <summary>
        /// Retrieve the currently selected view model.
        /// </summary>
        /// <returns>Returns the currently selected view model, or null otherwise.</returns>
        private HierarchyViewModel GetSelectedViewModel()
        {
            var workspaceRootViewModel = this.Parent as WorkspaceRootViewModel;
            return workspaceRootViewModel.SelectedNodeViewModel as HierarchyViewModel;
        }

        /// <summary>
        /// Enumerate all the emitter names, perform a deep search to
        /// all the emitters in the emitter set, including the child emitters.
        /// </summary>
        /// <param name="eset">The emitter set view model.</param>
        /// <returns>The emitter names.</returns>
        private static IEnumerable<string> EnumerateEmitterName(EmitterSetViewModel eset)
        {
            return eset.Emitters.Select(emitter => emitter.Name);
        }

        /// <summary>
        /// エミッタセットの保存処理本体
        /// </summary>
        /// <param name="emitterSetViewModel">保存するVM</param>
        /// <param name="filePath">保存先のパス</param>
        /// <param name="changePath">ファイルパスを書き換えるか否か</param>
        private static void SaveEmitterSetFileCore(EmitterSetViewModel emitterSetViewModel, string filePath, bool changePath)
        {
            // エミッタセットデータのコピーを作成
            EmitterSetData cloneData = null;
            using (new EnableDataModelCloneSetter())
            {
                cloneData = (EmitterSetData)emitterSetViewModel.DataModel.Clone();
            }

            // アセットパスをファイル名だけに書き換え
            AssetsManager.ReplaceAssetsPathToFileName(cloneData, filePath);

            // プレビューを削除
            cloneData.PreviewList.Clear();

            // ファイルパスを記録しない
            cloneData.FilePath = string.Empty;

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // シリアライズを実行
            var result = serializer.SerializeToFile(cloneData, filePath);

            if (result.IsSuccess == false)
            {
                var retMsgs = new string[2];
                retMsgs[0] = string.Format(Resources.ErrorSaveEmitterSet, filePath);
                retMsgs[1] = Resources.ErrorSaveCaption;
                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnShowErrorDialogExecutable;
                if (exe != null)
                {
                    exe.Execute(retMsgs);
                }

                Logger.Log("LogView", LogLevels.Error, retMsgs[0]);
                return;
            }

            // エミッタセットの名前を設定
            var esetName = Path.GetFileNameWithoutExtension(filePath);
            emitterSetViewModel.SetDataModelValue(esetName, () => emitterSetViewModel.DataModel.Name);
            emitterSetViewModel.EmitterSetBasicViewModel.EmitterSetBasicBasicViewModel.RaiseNameChanged();

            // エミッタセットのファイルパスを設定
            var esetPath = PathUtility.GetDirectoryName(filePath);
            if (changePath && emitterSetViewModel.FilePath != esetPath)
            {
                emitterSetViewModel.FilePath = esetPath;  // ビューアに繋いでいると重い
                emitterSetViewModel.EmitterSetBasicViewModel.EmitterSetBasicBasicViewModel.RaiseFilePathChanged();
            }

            if (changePath)
            {
                // デフォルト値を更新
                WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(emitterSetViewModel);

                // データ変更フラグをクリア(子要素のフラグが親に作用することがあるため、子を先にクリアする)
                emitterSetViewModel.ModificationFlagViewModel.ClearChildModificationFlags(typeof(PreviewViewModel));
                emitterSetViewModel.ModificationFlagViewModel.ClearModificationFlags();
            }
        }

        /// <summary>
        /// プレビューの保存処理本体
        /// </summary>
        /// <param name="previewViewModel">保存するVM</param>
        /// <param name="filePath">保存先のパス</param>
        /// <param name="changePath">ファイルパスを書き換えるか否か</param>
        private static void SavePreviewFileCore(PreviewViewModel previewViewModel, string filePath, bool changePath)
        {
            // 親のエミッタセット名を設定
            var emitterSetViewModel =
                ViewModelBase.GetParent<EmitterSetViewModel>(previewViewModel);

            previewViewModel.SetEsetName(emitterSetViewModel.FileName);

            var pbfv = previewViewModel.GetChild<PreviewBasicViewModel>().PreviewBasicFileViewModel;

            if (changePath)
            {
                // プレビューのフォルダパスを設定
                var folderPath = PathUtility.GetDirectoryName(filePath);
                previewViewModel.FilePath = folderPath;
                pbfv.RaiseFilePathChanged();
            }

            // プレビューの名前を設定
            var prevName = Path.GetFileNameWithoutExtension(filePath);
            previewViewModel.SetDataModelValue(prevName, () => previewViewModel.DataModel.Name);
            pbfv.RaiseNameChanged();

            // ファイルパスを記録させないようにする
            var tmpName = previewViewModel.DataModel.FilePath;
            previewViewModel.DataModel.FilePath = string.Empty;

            PreviewDataSerializer serializer = new PreviewDataSerializer();

            // シリアライズを実行
            var result = serializer.SerializeToFile(previewViewModel.DataModel, filePath);

            // パス復帰
            previewViewModel.DataModel.FilePath = tmpName;

            if (result.IsSuccess == false)
            {
                var retMsgs = new string[2];
                retMsgs[0] = string.Format(Resources.ErrorSavePreview, filePath);
                retMsgs[1] = Resources.ErrorSaveCaption;
                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnShowErrorDialogExecutable;
                if (exe != null)
                {
                    exe.Execute(retMsgs);
                }

                Logger.Log("LogView", LogLevels.Error, retMsgs[0]);
                return;
            }

            if (changePath)
            {
                // デフォルト値を更新
                WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(previewViewModel);

                // データ変更フラグをクリア。
                previewViewModel.ModificationFlagViewModel.ClearModificationFlags();
                previewViewModel.ModificationFlagViewModel.ClearChildModificationFlags();
            }
        }
    }
}
