﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels.Extenders;
using EffectMaker.UILogic.ViewModels.Factories;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// A base class for view model representing workspace node
    /// </summary>
    public abstract class WorkspaceNodeViewModelBase : HierarchyViewModel, IPropertyPageOwner
    {
        /// <summary>
        /// A regular expression that detects the number of duplication in a name.
        /// </summary>
        private static readonly Regex DuplicatedNameRegex =
            new Regex(@"^(?<name>.+)_Copy(?<num>\d+)$", RegexOptions.Compiled);

        /// <summary>Extender for manipulating child property pages.</summary>
        private PropertyPageOwnerViewModelExtender propertyPageOwnerExtender;

        /// <summary>
        /// Backing field for the IsExpanded property.
        /// </summary>
        private bool isExpanded;

        /// <summary>
        /// Backing field for the IsSelected property.
        /// </summary>
        private bool isSelected;

        /// <summary>
        /// 複数選択服従側フラグのバッキングフィールドです。
        /// </summary>
        private bool isMultiSelected;

        /// <summary>
        /// Initializes the WorkspaceNodeViewModelBase instance.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The corresponding data model.</param>
        protected WorkspaceNodeViewModelBase(HierarchyViewModel parent, DataModelBase dataModel)
            : base(parent, dataModel)
        {
            this.propertyPageOwnerExtender = new PropertyPageOwnerViewModelExtender(this);

            this.SwapNodesExecutable = new AnonymousExecutable(this.OnSwapNodes);

            var nodeCopyExecutable = new AnonymousExecutable(this.OnNodeCopy);
            var nodePasteExecutable = new AnonymousExecutable(this.OnNodePaste);
            var nodeDuplicateExecutable = new AnonymousExecutable(this.OnNodeDuplicate);
            var nodePasteAsChildExecutable = new AnonymousExecutable(this.OnNodePasteAsChild);

            var canCopy =
                this is EmitterViewModel ||
                this is FieldViewModel ||
                this is PreviewViewModel ||
                this is ReservedShaderNodeViewModel ||
                this is CustomActionViewModel ||
                this is EmitterExtParamsViewModel;

            var canPaste =
                this is EmitterViewModel ||
                this is EmitterSetViewModel ||
                this is FieldViewModel ||
                this is PreviewViewModel ||
                this is ReservedShaderNodeViewModel ||
                this is CustomActionViewModel ||
                this is EmitterExtParamsViewModel;

            var canDuplicate =
                this is EmitterViewModel ||
                this is EmitterSetViewModel ||
                this is FieldViewModel ||
                this is PreviewViewModel;

            var canPasteAsChild =
                this is EmitterViewModel;

            nodeCopyExecutable.IsEnabled = canCopy;
            nodePasteExecutable.IsEnabled = canPaste;
            nodeDuplicateExecutable.IsEnabled = canDuplicate;
            nodePasteAsChildExecutable.IsEnabled = canPasteAsChild;

            this.NodeCopyExecutable = nodeCopyExecutable;
            this.NodePasteExecutable = nodePasteExecutable;
            this.NodeDuplicateExecutable = nodeDuplicateExecutable;
            this.NodePasteAsChildExecutable = nodePasteAsChildExecutable;

            this.EvaluateCopyPasteAvailabilityExecutable
                = new AnonymousExecutable(this.OnEvaluateCopyPasteAvailability);
        }

        /// <summary>
        /// Gets the executable to run when nodes are swapped, from the UI level.
        /// </summary>
        public IExecutable SwapNodesExecutable { get; private set; }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public abstract HierarchyViewModel DefaultPropertyPageViewModel { get; }

        /// <summary>
        /// Get or set the selected property page.
        /// </summary>
        public virtual HierarchyViewModel SelectedPropertyPage
        {
            get
            {
                return this.propertyPageOwnerExtender.SelectedPropertyPage;
            }

            set
            {
                this.propertyPageOwnerExtender.SelectedPropertyPage = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// ツリーノードの展開状態です。
        /// </summary>
        public bool IsExpanded
        {
            get { return this.isExpanded; }
            set { this.SetValue(ref this.isExpanded, value); }
        }

        /// <summary>
        /// Gets or set the node selection.
        /// </summary>
        public bool IsSelected
        {
            get { return this.isSelected; }
            set { this.SetValue(ref this.isSelected, value); }
        }

        /// <summary>
        /// 複数選択されているかどうかを表すフラグを取得または設定します。
        /// </summary>
        public bool IsMultiSelected
        {
            get
            {
                return this.isMultiSelected;
            }

            set
            {
                this.SetValue(ref this.isMultiSelected, value);
                var nodes = WorkspaceRootViewModel.Instance.MultiSelectedNodes;
                if (value)
                {
                    if (!nodes.Contains(this))
                    {
                        nodes.Add(this);
                    }
                }
                else
                {
                    if (nodes.Contains(this))
                    {
                        nodes.Remove(this);
                    }
                }
            }
        }

        /// <summary>
        /// Gets the Executable instance to run when copy/paste/duplication
        /// availability evaluation is required.
        /// </summary>
        public IExecutable EvaluateCopyPasteAvailabilityExecutable { get; private set; }

        /// <summary>
        /// Gets the Executable instance to run when the Copy Node menu is clicked.
        /// </summary>
        public IExecutable NodeCopyExecutable { get; private set; }

        /// <summary>
        /// Gets the Executable instance to run when the Paste Node menu is clicked.
        /// </summary>
        public IExecutable NodePasteExecutable { get; private set; }

        /// <summary>
        /// Gets the Executable instance to run when the Duplicate Node menu is clicked.
        /// </summary>
        public IExecutable NodeDuplicateExecutable { get; private set; }

        /// <summary>
        /// Gets the Executable instance to run when the Paste as child Node menu is clicked.
        /// </summary>
        public IExecutable NodePasteAsChildExecutable { get; private set; }

        /// <summary>
        /// Get a new name for a duplicated node, base on the name of the current node.
        /// </summary>
        /// <param name="instance">Instance used to determine
        /// a duplicated name among its siblings.</param>
        /// <param name="namesInEmitterSet">Names already in use in the emitter set.</param>
        /// <returns>Returns a new duplicated name.</returns>
        protected virtual string GetDuplicatedNodeName(
            HierarchyViewModel instance,
            IEnumerable<string> namesInEmitterSet)
        {
            string name = (string)((dynamic)instance).Name;

            // extracts the already existing copy number, per node base name
            var numbers = namesInEmitterSet
                .Where(n => string.IsNullOrWhiteSpace(n) == false)
                .Select(n => DuplicatedNameRegex.Match(n))
                .Where(m => m.Success)
                .Where(m => m.Groups["name"].Value == name)
                .Select(m => int.Parse(m.Groups["num"].Value))
                .Distinct()
                .OrderBy(i => i)
                .ToArray();

            if (numbers.Length == 0)
            {
                if (namesInEmitterSet.Contains(name) == false)
                {
                    return name;
                }
            }

            var num = this.GetUniqueAvailableNumber(numbers);

            return string.Format("{0}_Copy{1}", name, num);
        }

        /// <summary>
        /// Gets the name of an instance through dynamic access.
        /// </summary>
        /// <param name="instance">Instance to get the name.</param>
        /// <returns>Returns the name of a dynamic instance, of null if not available.</returns>
        protected string GetName(dynamic instance)
        {
            try
            {
                return instance.Name;
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        /// Finds the smallest integer number the fit in the
        /// sequence without overlapping any value.
        /// </summary>
        /// <param name="numbers">The sequence of number.</param>
        /// <returns>Returns the smallest integer that can fit in the
        /// input sequence without overlapping any value.</returns>
        protected int GetUniqueAvailableNumber(IList<int> numbers)
        {
            if (numbers == null || numbers.Count == 0)
            {
                return 1;
            }

            if (numbers.Count == 1)
            {
                if (numbers[0] != 1)
                {
                    return 1;
                }
                else
                {
                    return numbers[0] + 1;
                }
            }

            int current = 0;

            for (int i = 0; i < numbers.Count; i++)
            {
                if (current + 1 != numbers[i])
                {
                    break;
                }

                current = numbers[i];
            }

            return current + 1;
        }

        /// <summary>
        /// Creates a view model depending on the type of data model,
        /// and add it to the children collection of the current node.
        /// </summary>
        /// <param name="parent">The node where to add the new view model.</param>
        /// <param name="genericDataModel">Data model to crate a view model for.</param>
        /// <param name="isDuplicatedNode">複製ノードの時はフラグをクリアするのでtrue,そうでなければfalse.</param>
        /// <returns>複製したノードのViewModel.</returns>
        protected HierarchyViewModel CreateAndAddViewModel(
            HierarchyViewModel parent,
            DataModelBase genericDataModel,
            bool isDuplicatedNode = false)
        {
            EmitterSetViewModel emitterSet = null;

            if (parent is EmitterSetViewModel)
            {
                emitterSet = (EmitterSetViewModel)parent;
            }
            else
            {
                emitterSet = parent.FindNearestParentOfType<EmitterSetViewModel>();
            }

            string[] alreadyExistingNames = null;

            if (emitterSet != null)
            {
                if (genericDataModel is EmitterData)
                {
                    // 同じエミッタセットに属するエミッタの名前リストを取得
                    var alreadyPresentEmitters = new List<EmitterViewModel>();
                    this.FindObjectsOfType(emitterSet, alreadyPresentEmitters);

                    alreadyExistingNames = alreadyPresentEmitters
                        .Select(e => this.GetName(e))
                        .Where(s => string.IsNullOrWhiteSpace(s) == false)
                        .ToArray();
                }
                else if (genericDataModel is PreviewData)
                {
                    // 同じエミッタセットに属するプレビューの名前リストを取得
                    var alreadyPresetPreviews = new List<PreviewViewModel>();
                    this.FindObjectsOfType(parent, alreadyPresetPreviews);

                    alreadyExistingNames = alreadyPresetPreviews
                        .Select(e => this.GetName(e))
                        .Where(s => string.IsNullOrWhiteSpace(s) == false)
                        .ToArray();
                }
            }
            else
            {
                // エミッタセットの名前リストを取得
                var esetList = ((dynamic)parent).EmitterSetList as IEnumerable<EmitterSetData>;

                if (esetList != null)
                {
                    alreadyExistingNames = esetList
                        .Select(e => e.Name)
                        .Where(s => string.IsNullOrWhiteSpace(s) == false)
                        .ToArray();
                }
                else
                {
                    alreadyExistingNames = new string[0];
                }
            }

            WorkspaceNodeViewModelBase viewModel = null;

            if (genericDataModel is EmitterData)
            {
                var dataModel = (EmitterData)genericDataModel;
                viewModel = new EmitterViewModel(parent, dataModel);
            }
            else if (genericDataModel is EmitterSetData)
            {
                var dataModel = (EmitterSetData)genericDataModel;
                viewModel = new EmitterSetViewModel(parent, dataModel);
            }
            else if (genericDataModel is PreviewData)
            {
                var dataModel = (PreviewData)genericDataModel;
                viewModel = new PreviewViewModel(parent, dataModel);
            }
            else if (genericDataModel is FieldDataBase)
            {
                viewModel = FieldViewModelFactory.CreateFieldViewModel(
                    parent,
                    (FieldDataBase)genericDataModel);
            }
            else
            {
                // TODO: log an error
            }

            if (viewModel == null)
            {
                return null;
            }

            if ((viewModel is FieldViewModel) == false)
            {
                if (emitterSet != null)
                {
                    if (viewModel is EmitterViewModel)
                    {
                        // エミッタの名前を設定
                        var newlyAddedEmitters = new List<EmitterViewModel>();
                        this.FindObjectsOfType(viewModel, newlyAddedEmitters);

                        foreach (var emitter in newlyAddedEmitters)
                        {
                            emitter.SetName(this.GetDuplicatedNodeName(emitter, alreadyExistingNames));
                        }
                    }
                    else if (viewModel is PreviewViewModel)
                    {
                        // プレビューの名前を設定
                        var preview = (PreviewViewModel)viewModel;
                        preview.SetName(this.GetDuplicatedNodeName(viewModel, alreadyExistingNames));
                    }
                }
                else
                {
                    // エミッタセットの名前を設定
                    var emitterSetViewModel = viewModel as EmitterSetViewModel;
                    if (emitterSetViewModel != null)
                    {
                        emitterSetViewModel.SetName(this.GetDuplicatedNodeName(viewModel, alreadyExistingNames));
                    }
                }
            }

            // 複製時は自分のノードのスターマークをクリアしておく
            if (isDuplicatedNode)
            {
                var mod = viewModel as IModificationFlagOwner;
                if (mod != null)
                {
                    mod.ModificationFlagViewModel.ClearChildModificationFlags();
                    mod.ModificationFlagViewModel.ClearModificationFlags();
                }
            }

            CommandManager.Execute(new AddHierarchicalNodeCommand(parent, viewModel));

            viewModel.IsSelected = true;

            return viewModel;
        }

        /// <summary>
        /// ノードのペーストが可能かチェックする.
        /// </summary>
        /// <param name="data">クリップボードのデータ.</param>
        /// <returns>ペースト可能ならtrueを返す.</returns>
        protected virtual bool CanPasteNode(NodeClipboardData data)
        {
            // EffectMakerを終了するときに発生する例外の対応
            if (this.Proxy == null || this.Proxy.DataModel == null)
            {
                System.Diagnostics.Debug.Assert(false, "Proxyがnull");
                return false;
            }

            DataModelBase dataModel = this.Proxy.DataModel;

            return dataModel.GetType().FullName == data.ClassFullName;
        }

        /// <summary>
        /// Called when nodes are swapped from the UI level.
        /// </summary>
        /// <param name="parameter">A custom parameter.
        /// Should be Tuple(int, int) that respectively contains
        /// the source and the target indices.</param>
        private void OnSwapNodes(object parameter)
        {
            var p = parameter as Tuple<int, int>;

            if (p == null)
            {
                return;
            }

            int sourceIndex = p.Item1;
            int targetIndex = p.Item2;

            if (sourceIndex < 0 || targetIndex < 0)
            {
                return;
            }

            var children = this.Children
                .OfType<WorkspaceNodeViewModelBase>()
                .ToArray();

            if (sourceIndex >= children.Length || targetIndex >= children.Length)
            {
                return;
            }

            WorkspaceNodeViewModelBase sourceNode = children[sourceIndex];
            WorkspaceNodeViewModelBase targetNode = children[targetIndex];

            if (sourceNode == null || targetNode == null)
            {
                return;
            }

            if (sourceNode.GetType() != targetNode.GetType())
            {
                Type commonType = TypeExtensions.FindCommonAncestorType(
                    sourceNode.GetType(),
                    targetNode.GetType());

                if (commonType == typeof(WorkspaceNodeViewModelBase))
                {
                    return;
                }
            }

            sourceIndex = this.Children.IndexOf(sourceNode);
            targetIndex = this.Children.IndexOf(targetNode);

            if (sourceIndex == -1 || targetIndex == -1)
            {
                return;
            }

            CommandManager.Execute(new SwapNodesCommand(this, sourceIndex, targetIndex));
        }

        /// <summary>
        /// Called when the copy/paste/duplicate context menu is being opened.
        /// </summary>
        /// <param name="parameter">A custom parameter.</param>
        private void OnEvaluateCopyPasteAvailability(object parameter)
        {
            if (Clipboard.ContainsData(NodeClipboardData.CustomerFormatName) == false)
            {
                ((AnonymousExecutable)this.NodePasteExecutable).IsEnabled = false;
            }

            var value = Clipboard.GetData(NodeClipboardData.CustomerFormatName) as NodeClipboardData;
            if (value == null)
            {
                ((AnonymousExecutable)this.NodePasteExecutable).IsEnabled = false;
            }
            else
            {
                ((AnonymousExecutable)this.NodePasteExecutable).IsEnabled = this.CanPasteNode(value);
            }
        }

        /// <summary>
        /// Called when the NodeCopyExecutable is run.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnNodeCopy(object parameter)
        {
            string value = null;

            {
                var result = DataModelSerializeUtility.SerializeToClipboardData(this.Proxy.DataModel, out value);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceNodeViewModelBase.OnNodeCopy : Failed to execute SerializeNode.");

                    return;
                }
            }

            string extraFlag = string.Empty;
            if (this is EmitterViewModel)
            {
                var emitterData = (EmitterData)this.Proxy.DataModel;
                if (emitterData.Parent is EmitterSetData)
                {
                    extraFlag = "Discard Inheritance.";
                }
            }

            var data = new NodeClipboardData()
            {
                ClassFullName = this.Proxy.DataModel.GetType().FullName,
                XmlData = value,
                ExtraData = extraFlag,
            };

            Clipboard.SetData(NodeClipboardData.CustomerFormatName, data);
        }

        /// <summary>
        /// Called when the NodePasteExecutable is run.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnNodePaste(object parameter)
        {
            DataModelBase dataModel = this.Proxy.DataModel;

            var value = Clipboard.GetData(NodeClipboardData.CustomerFormatName) as NodeClipboardData;
            if (value == null)
            {
                return;
            }

            DataModelBase genericDataModel = null;

            {
                var result = DataModelSerializeUtility.DeserializeFromClipboardData(value.XmlData, value.ClassFullName);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceNodeViewModelBase.OnNodePaste : Failed to execute Deserialize.");

                    return;
                }

                genericDataModel = result.DataModel;
            }

            using (new EnableDataModelCloneSetter())
            {
                genericDataModel = (DataModelBase)genericDataModel.CloneWithoutGuid();
            }

            if (dataModel is EmitterSetData)
            {
                var addedVm = this.CreateAndAddViewModel(this, genericDataModel) as IModificationFlagOwner;
                if (addedVm != null)
                {
                    addedVm.ModificationFlagViewModel.UpdateDefaultValues();
                }
            }
            else
            {
                var srcEmitter = genericDataModel as EmitterData;
                var dstEmitter = dataModel as EmitterData;
                if (srcEmitter != null && dstEmitter != null)
                {
                    var name = dstEmitter.Name;
                    srcEmitter.Name = name;

                    // コピー元が親エミッタだったら継承設定をコピー先で上書きしておく
                    if (value.ExtraData.Equals("Discard Inheritance."))
                    {
                        using (new EnableDataModelCloneSetter())
                        {
                            srcEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.SetWithoutGuid(
                                dstEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData);
                        }
                    }

                    // ペースト先がエミッタセット内唯一のアクティブエミッタである場合、コピー元が非アクティブでも強制的にアクティブにする
                    var dstEmitterSet = dstEmitter.FindParentOfType<EmitterSetData>();
                    if (dstEmitterSet != null)
                    {
                        bool isActiveOnlyOne = dstEmitterSet.ActiveEmitterList.Count() == 1;
                        bool isActiveContains = dstEmitterSet.ActiveEmitterList.Contains(dstEmitter);
                        if (isActiveOnlyOne && isActiveContains && !srcEmitter.EnableConvert)
                        {
                            srcEmitter.EnableConvert = true;
                        }
                    }
                }

                CommandManager.Execute(new SetDataModelCommand(this, genericDataModel));
            }
        }

        /// <summary>
        /// Called when the NodeDuplicateExecutable is run.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnNodeDuplicate(object parameter)
        {
            DataModelBase dataModel = this.Proxy.DataModel;

            string value;

            {
                var result = DataModelSerializeUtility.SerializeToClipboardData(dataModel, out value);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceNodeViewModelBase.OnNodeCopy : Failed to execute SerializeNode.");

                    return;
                }
            }

            DataModelBase genericDataModel;

            {
                var result = DataModelSerializeUtility.DeserializeFromClipboardData(value, dataModel.GetType().FullName);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceNodeViewModelBase.OnNodeDuplicate : Failed to execute Deserialize.");

                    return;
                }

                genericDataModel = result.DataModel;
            }

            using (new EnableDataModelCloneSetter())
            {
                genericDataModel = (DataModelBase)genericDataModel.CloneWithoutGuid();
            }

            // プレビューのGUIDを新しく割り当てる.
            if (genericDataModel is EmitterSetData)
            {
                var eset = genericDataModel as EmitterSetData;
                for (int i = 0; i < eset.PreviewList.Count; ++i)
                {
                    eset.PreviewList[i].Name = ((EmitterSetData)dataModel).PreviewList[i].Name;
                    using (new EnableDataModelCloneSetter())
                    {
                        eset.PreviewList[i] = (PreviewData)eset.PreviewList[i].CloneWithoutGuid();
                    }
                }

                // ファイルパスを初期化
                eset.Name = ((EmitterSetData)dataModel).Name;
                eset.FilePath = string.Empty;
            }

            if (genericDataModel is PreviewData)
            {
                var prev = genericDataModel as PreviewData;

                // ファイルパスを初期化
                prev.Name = ((PreviewData)dataModel).Name;
                prev.FilePath = string.Empty;

                // プレビューの場合はフラグ更新をブロックする
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                var addedVm = this.CreateAndAddViewModel((HierarchyViewModel)this.Parent, prev, true) as IModificationFlagOwner;
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
                if (addedVm != null)
                {
                    addedVm.ModificationFlagViewModel.UpdateDefaultValues();
                }
            }
            else
            {
                var addedVm = this.CreateAndAddViewModel((HierarchyViewModel)this.Parent, genericDataModel, true) as IModificationFlagOwner;
                if (addedVm != null)
                {
                    addedVm.ModificationFlagViewModel.UpdateDefaultValues();
                }
            }
        }

        /// <summary>
        /// Called when the NodePasteAsChildExecutable is run.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnNodePasteAsChild(object parameter)
        {
            var value = Clipboard.GetData(NodeClipboardData.CustomerFormatName) as NodeClipboardData;
            if (value == null)
            {
                return;
            }

            DataModelBase genericDataModel = null;

            {
                var result = DataModelSerializeUtility.DeserializeFromClipboardData(value.XmlData, value.ClassFullName);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceNodeViewModelBase.OnNodePaste : Failed to execute Deserialize.");

                    return;
                }

                genericDataModel = result.DataModel;
            }

            using (new EnableDataModelCloneSetter())
            {
                genericDataModel = (DataModelBase)genericDataModel.CloneWithoutGuid();
            }

            var emitterData = genericDataModel as EmitterData;
            if (emitterData != null)
            {
                var childEmitterList = genericDataModel.Children.OfType<EmitterData>().ToList();
                foreach (var emitter in childEmitterList)
                {
                    genericDataModel.RemoveChild(emitter);
                }

                emitterData.EmitterList.Clear();

                // コピー元が親エミッタだったら、ペーストデータを初期値で上書きしておく
                if (value.ExtraData.Equals("Discard Inheritance."))
                {
                    using (new EnableDataModelCloneSetter())
                    {
                        emitterData.EmitterBasicSettingData.EmitterBasicInheritanceData.SetWithoutGuid(
                            new EmitterBasicInheritanceData());
                    }
                }
            }

            var addedVm = this.CreateAndAddViewModel(this, genericDataModel, true) as IModificationFlagOwner;
            if (addedVm != null)
            {
                addedVm.ModificationFlagViewModel.UpdateDefaultValues();
            }
        }

        /// <summary>
        /// Gather all the object of type T.
        /// </summary>
        /// <typeparam name="T">The tpy eof object to gather.</typeparam>
        /// <param name="node">The node from where to start to gather.</param>
        /// <param name="instances">A writable collection where to gather.</param>
        private void FindObjectsOfType<T>(IHierarchyObject node, ICollection<T> instances)
            where T : IHierarchyObject
        {
            if (node is T)
            {
                instances.Add((T)node);
            }

            foreach (IHierarchyObject child in node.Children)
            {
                this.FindObjectsOfType<T>(child, instances);
            }
        }

        /// <summary>
        /// ノードクリップボードのデータ.
        /// </summary>
        [Serializable]
        protected class NodeClipboardData
        {
            /// <summary>
            /// クリップボード用の識別文字列.
            /// </summary>
            public const string CustomerFormatName = "EffectMaker4F_NodeClipboardData";

            /// <summary>
            /// フルネームのクラス名.
            /// </summary>
            public string ClassFullName { get; set; }

            /// <summary>
            /// XMLのデータ.
            /// </summary>
            public string XmlData { get; set; }

            /// <summary>
            /// コピー時に記録し、ペースト時に参照したい追加データ.
            /// </summary>
            public string ExtraData { get; set; }
        }
    }

    /// <summary>
    /// ジェネリック版のWorkspaceNodeViewModelBaseです。
    /// </summary>
    /// <typeparam name="T">データモデルの型</typeparam>
    public abstract class WorkspaceNodeViewModelBase<T> : WorkspaceNodeViewModelBase where T : DataModelBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">関連付けるデータモデル</param>
        protected WorkspaceNodeViewModelBase(HierarchyViewModel parent, T dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 関連付けられたデータモデルを取得します。
        /// </summary>
        public T DataModel
        {
            get { return Proxy.DataModel; }
        }
    }
}
