﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ViewerBasicSceneViewModel.
    /// </summary>
    public class ViewerBasicSceneViewModel : PropertyGroupViewModel<ViewerBasicSceneData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "GridTypeItems",
            "GridSizeItems",
            "GridDivisionNumberItems",
            "GlareLevelItems",
            "FloorType",
            "EnabledGridSize",
            "EnabledGridDivisionNumber"
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ViewerBasicSceneViewModel(HierarchyViewModel parent, ViewerBasicSceneData dataModel)
            : base(parent, dataModel)
        {
            // グリッドタイプの選択項目を作成
            this.GridTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.ViewerGridTypeGrid, 0),
                new KeyValuePair<string, object>(Properties.Resources.ViewerGridTypeBlock, 2)
            };

            // グリッドサイズの選択項目を作成
            {
                List<KeyValuePair<string, object>> items = new List<KeyValuePair<string, object>>();

                for (int i = 1; i <= 10; ++i)
                {
                    KeyValuePair<string, object> item = new KeyValuePair<string, object>(i.ToString(), i);
                    items.Add(item);
                }

                this.GridSizeItems = items;
            }

            // グリッド分割数の選択項目を作成
            this.GridDivisionNumberItems = new []
            {
                new KeyValuePair<string, object>("1", 1),
                new KeyValuePair<string, object>("10", 10),
            };

            // グレアの強さの選択項目を作成
            {
                List<KeyValuePair<string, object>> items = new List<KeyValuePair<string, object>>();

                for (int i = 0; i <= 6; ++i)
                {
                    KeyValuePair<string, object> item = new KeyValuePair<string, object>(i.ToString(), i);
                    items.Add(item);
                }

                this.GlareLevelItems = items;
            }

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignoreModifyPropertyNames);
        }

        /// <summary>
        /// グリッド表示の On/Off を取得または設定します。
        /// </summary>
        public bool DisplayGrid
        {
            get
            {
                return this.GetDataModelValue<bool>(nameof(this.DataModel.DisplayGrid));
            }

            set
            {
                this.SetDataModelValue(value, nameof(this.DataModel.DisplayGrid));

                this.OnPropertyChanged(nameof(this.FloorType));
                this.OnPropertyChanged(nameof(this.EnabledGridSize));
                this.OnPropertyChanged(nameof(this.EnabledGridDivisionNumber));
            }
        }

        /// <summary>
        /// グリッドタイプを取得または設定します。
        /// </summary>
        public int GridType
        {
            get
            {
                return this.GetDataModelValue<int>(nameof(this.DataModel.GridType));
            }

            set
            {
                this.SetDataModelValue(value, nameof(this.DataModel.GridType));

                this.OnPropertyChanged(nameof(this.FloorType));
                this.OnPropertyChanged(nameof(this.EnabledGridSize));
                this.OnPropertyChanged(nameof(this.EnabledGridDivisionNumber));
            }
        }

        /// <summary>
        /// 床の表示タイプを取得または設定します。
        /// ViewerPanel 向けのプロパティです。
        /// </summary>
        public int FloorType
        {
            get
            {
                if (this.DisplayGrid)
                {
                    return this.GridType + 1;
                }
                else
                {
                    return 0;
                }
            }

            set
            {
                using (new CommandCombiner())
                {
                    if (value != 0)
                    {
                        this.DisplayGrid = true;
                        this.GridType = value - 1;
                    }
                    else
                    {
                        this.DisplayGrid = false;
                    }
                }
            }
        }

        /// <summary>
        /// 床のサイズが有効かどうかを取得します。
        /// </summary>
        public bool EnabledGridSize
        {
            get
            {
                // 床の表示タイプがグリッドかブロックのときに true
                return (this.DisplayGrid && (this.GridType == 0 || this.GridType == 2));
            }
        }

        /// <summary>
        /// グリッドの分割数が有効かどうか取得します。
        /// </summary>
        public bool EnabledGridDivisionNumber
        {
            get
            {
                // 床の表示タイプがグリッドのときに true
                return (this.DisplayGrid && (this.GridType == 0));
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// グリッドタイプの選択項目を取得します。
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> GridTypeItems { get; private set; }

        /// <summary>
        /// グリッドサイズの選択項目を取得します。
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> GridSizeItems { get; private set; }

        /// <summary>
        /// グリッド分割数の選択項目を取得します。
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> GridDivisionNumberItems { get; private set; }

        /// <summary>
        /// グレアの強さの選択項目を取得します。
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> GlareLevelItems { get; private set; }
    }
}
